/* =========================================================
 * employees.js
 * Guard + IIFE wrapper
 * ========================================================= */

if (window.__EMPLOYEES_JS_LOADED__) {
	console.warn('employees.js already loaded, skip execution.');
} else {
	window.__EMPLOYEES_JS_LOADED__ = true;

	(function () {
		'use strict';

		/* employees.js for employees.html */

		currentPage = 1; // 預設第一頁
		let currentGenderFilter = "";
		let currentMaritalStatusFilter = "";
		let currentBloodTypeFilter = "";
		let employeeOptionsCache = [];

		document.addEventListener("DOMContentLoaded", function() {
			// 初始載入資料
			loadEmployees();

			// - 員工區塊 -
			document.getElementById('search').addEventListener('input', searchEmployees);
			document.getElementById('createEmployeeForm').addEventListener('submit', submitFormHandler);

			// — 部門區塊 —
			document.getElementById('searchDepartmentsInput').addEventListener('input', searchDepartments);
			document.getElementById('createDepartmentForm').addEventListener('submit', submitAddDepartmentFormHandler);

			// — 職稱區塊 —
			document.getElementById('searchJobtitlesInput').addEventListener('input', searchJobtitles);
			document.getElementById('createJobtitleForm').addEventListener('submit', submitAddJobtitleFormHandler);

			// 點擊其他地方時清除建議清單
			document.addEventListener('click', function (e) {
				if (!e.target.classList.contains('editable')) {
					clearSuggestions();
				}
			});

			// 初始化模態對話框的顯示
			var dialogs = document.querySelectorAll('dialog');
			dialogs.forEach(function(dialog) {
				if (typeof HTMLDialogElement !== 'function') {
					dialogPolyfill.registerDialog(dialog);
				}
			});
		});

		/* 載入員工列表 */
		function loadEmployees(page = 1) {
			currentPage = page;  // 當前頁數
			const pSize = 15;
			var conditionObj = {};

			currentGenderFilter = document.getElementById('searchGender').value;  // 取得搜尋性別
			currentBloodTypeFilter = document.getElementById('searchBloodType').value;  // 取得搜尋血型
			currentMaritalStatusFilter = document.getElementById('searchMaritalStatus').value;  // 取得搜尋婚姻狀況
			var query = document.getElementById('search').value.trim().toLowerCase();  // 取得搜尋關鍵字
			const orderby = document.getElementById('orderby').value;
			const desc = document.getElementById('desc').value;

			if (currentGenderFilter === 'none')
				currentGenderFilter = '';

			if (currentBloodTypeFilter === 'none')
				currentBloodTypeFilter = '';

			if (currentMaritalStatusFilter === 'none')
				currentMaritalStatusFilter = '';	

			// 組合查詢參數（假設 API 支援 category 與 subcategory 參數）
			const params = new URLSearchParams({
				page: page,
				size: pSize,
				condition: query,
				gender: currentGenderFilter,
				marital_status: currentMaritalStatusFilter,
				blood_type: currentBloodTypeFilter,
				orderby: orderby,
				desc: desc
			});

			fetch(`${API_BASE}/employees/get_employees.php?${params.toString()}`)
				.then(response => response.json())
				.then(data => {
					if (data.result) {
						const employeesTableBody = document.querySelector('#employees-table tbody');
						employeesTableBody.innerHTML = ''; // 清空現有的表格內容
						data.data.records.forEach((record, index) => {
							const row = document.createElement('tr');

							row.innerHTML = `
			<td><input type="checkbox" name="select_employee" id="employee-${record.employee_id}" value="${record.employee_id}">${record.employee_id}</td>
			<td class="employee-cname">${record.cname}</td>
			<td class="employee-national_id">${record.national_id}</td>
			<td class="employee-gender">${record.gender}</td>
			<td class="employee-birth_date">${record.birth_date}</td>
			<td class="employee-nationality">${record.nationality}</td>
			<td class="employee-marital_status">${record.marital_status}</td>
			<td class="employee-blood_type">${record.blood_type}</td>
			<td class="employee-mobile_phone">${record.mobile_phone}</td>
			<td class="employee-email">${record.email}</td>
		    `;

							// 點擊 "名稱" 欄位彈出修改視窗
							row.querySelector('.employee-cname').addEventListener('click', function(event) {
								event.stopPropagation(); // 阻止事件冒泡
								showEditEmployeeModal(record.employee_id); // 點擊名稱彈出修改視窗
							});

							// 為「性別」欄位加入點擊事件
							row.querySelector('.employee-gender').addEventListener('click', function() {
								currentGenderFilter = record.gender; // 記住性別值
								document.getElementById('searchGender').value = currentGenderFilter;
								document.getElementById('orderby').value = "gender";
								loadEmployees(1);  // 重新載入時會帶入 currentGenderFilter
							});

							// 為「婚姻狀況」欄位加入點擊事件
							row.querySelector('.employee-marital_status').addEventListener('click', function() {
								currentMaritalStatusFilter = record.marital_status; // 記住婚姻狀況值
								document.getElementById('searchMaritalStatus').value = currentMaritalStatusFilter;
								document.getElementById('orderby').value = "marital_status";
								loadEmployees(1);  // 重新載入時會帶入 currentMaritalStatusFilter
							});

							// 為「血型」欄位加入點擊事件
							row.querySelector('.employee-blood_type').addEventListener('click', function() {
								currentBloodTypeFilter = record.blood_type; // 記住血型值
								document.getElementById('searchBloodType').value = currentBloodTypeFilter;
								document.getElementById('orderby').value = "blood_type";
								loadEmployees(1);
							});

							// 點擊其他欄位時，改變背景顏色並更新右側內容
							row.querySelectorAll('td').forEach((cell, cellIndex) => {
								if (cellIndex !== 0 && cellIndex !== 1) { // 排除 "編號" 和 "名稱" 欄位
									cell.addEventListener('click', function() {
										// 清除其他行的背景顏色
										document.querySelectorAll('#employees-table tbody tr').forEach(tr => {
											tr.style.backgroundColor = '';
										});
										// 設置當前行背景顏色
										row.style.backgroundColor = '#CCFFFF';
										// 更新右側 "相關証清單"
										//loadRelatedSyndrome(employee.employee_id);
									});
								}
							});

							employeesTableBody.appendChild(row);
						});
						// 生成分頁按鈕
						totalPages = Math.ceil(data.data.ttl / pSize); // 計算總頁數
						generatePaginationControls(); // 生成分頁按鈕
						loadDepartments();
						loadJobtitles();
					} else {
						alert('無法獲取員工列表');
					}
				})
				.catch(error => console.error('Error loading Employees list:', error));
		}

		function onOrderbyChange() {
			const orderby = document.getElementById('orderby').value;
			if(orderby === "all") {
				document.getElementById('searchEmployee').value = 'none';
				currentGenderFilter = "";
				currentMaritalStatusFilter = "";
				currentBloodTypeFilter = "";
			}
			// 重新載入資料，帶入新的 orderby 與可能的篩選條件
			loadEmployees(1);
		}

		function onDescChange() {
			const orderby = document.getElementById('orderby').value;
			if(orderby === "all") {
				document.getElementById('searchEmployee').value = 'none';
				currentGenderFilter = "";
				currentMaritalStatusFilter = "";
				currentBloodTypeFilter = "";
			}
			// 重新載入資料，帶入新的 orderby 與可能的篩選條件
			loadEmployees(1);
		}

		function autoResizeSelect(selectEl, options = {}) {
			if (!selectEl) return;

			const {
				minWidth = 80,
				maxWidth = 120,
				extra = 24
			} = options;

			const idx = selectEl.selectedIndex;

			// 🔒 關鍵防呆：沒有選取任何 option 就不要算
			if (idx < 0 || !selectEl.options[idx]) {
				selectEl.style.width = minWidth + 'px';
				return;
			}

			const text = selectEl.options[idx].text;
			if (!text) {
				selectEl.style.width = minWidth + 'px';
				return;
			}

			const temp = document.createElement('span');
			temp.style.visibility = 'hidden';
			temp.style.whiteSpace = 'nowrap';
			temp.style.font = getComputedStyle(selectEl).font;
			temp.textContent = text;

			document.body.appendChild(temp);
			let width = temp.offsetWidth + extra;
			document.body.removeChild(temp);

			width = Math.max(minWidth, Math.min(width, maxWidth));
			selectEl.style.width = width + 'px';
		}

		function openModalAtTop(modal, focusEl = null) {
			modal.showModal();

			requestAnimationFrame(() => {
				modal.scrollTop = 0;
				if (focusEl) focusEl.focus();
			});
		}

		async function loadJobtitlesToSelect(selectEl, selectedId = null) {
			const res = await fetch('/api/employees/get_jobtitles.php');
			const json = await res.json();
			if (!json.result) return;

			selectEl.innerHTML = '<option value="">請選擇職稱</option>';
			json.data.records.forEach(r => {
				const opt = document.createElement('option');
				opt.value = r.position_id;
				opt.textContent = r.title;
				if (selectedId && Number(selectedId) === Number(r.id)) {
					opt.selected = true;
				}
				selectEl.appendChild(opt);
			});
		}

		async function loadDepartmentsToSelect(selectEl, selectedId = null) {
			const res = await fetch('/api/employees/get_departments.php');
			const json = await res.json();
			if (!json.result) return;

			selectEl.innerHTML = '<option value="">請選擇部門</option>';
			json.data.records.forEach(r => {
				const opt = document.createElement('option');
				opt.value = r.department_id;
				opt.textContent = r.name;
				if (selectedId && Number(selectedId) === Number(r.id)) {
					opt.selected = true;
				}
				selectEl.appendChild(opt);
			});
		}

		/***************************/

		function truncateText(text, len=30) {
			return text.length > len ? text.substring(0, len - 1) + '...' : text;
		}
		/***************************/

		/* 更新分頁按鈕區塊 */
		function generatePaginationControls() {
			const paginationControls = document.getElementById('pagination-controls');
			paginationControls.innerHTML = ''; // 清空現有的分頁按鈕
			if (currentPage > totalPages) currentPage = totalPages;

			// 回首頁
			const firstPageBtn = createPaginationButton('firstpage_icon.png', 1);
			paginationControls.appendChild(firstPageBtn);

			// 上一頁
			const prevPageBtn = createPaginationButton('previous_icon.png', Math.max(currentPage - 1, 1));
			paginationControls.appendChild(prevPageBtn);

			// 最近 5 頁的頁碼按鈕
			const startPage = Math.max(currentPage - 2, 1);
			const endPage = Math.min(currentPage + 2, totalPages);
			for (let i = startPage; i <= endPage; i++) {
				const pageBtn = createPageNumberButton(i);
				paginationControls.appendChild(pageBtn);
			}

			// 下一頁
			const nextPageBtn = createPaginationButton('nextpage_icon.png', Math.min(currentPage + 1, totalPages));
			paginationControls.appendChild(nextPageBtn);

			// 最後一頁
			const lastPageBtn = createPaginationButton('endpage_icon.png', totalPages);
			paginationControls.appendChild(lastPageBtn);

			// 頁數輸入框
			const pageInput = document.createElement('input');
			pageInput.type = 'number';
			pageInput.min = 1;
			pageInput.max = totalPages;
			pageInput.value = currentPage;
			pageInput.addEventListener('change', () => {
				const selectedPage = parseInt(pageInput.value);
				if (selectedPage >= 1 && selectedPage <= totalPages) {
					loadEmployees(selectedPage);  // 加載選中的頁面
				}
			});
			paginationControls.appendChild(pageInput);
		}

		function createPaginationButton(iconSrc, page) {
			const btn = document.createElement('a');
			btn.href = 'javascript:void(0)';
			btn.onclick = () => loadEmployees(page);
			btn.innerHTML = `<img src="${IMG_BASE}/${iconSrc}" alt="icon" class="pagination-icon">`;
			return btn;
		}

		function createPageNumberButton(page) {
			const btn = document.createElement('a');
			btn.href = 'javascript:void(0)';
			btn.textContent = page;
			btn.onclick = () => loadEmployees(page);
			btn.style.margin = '0 5px';  // 增加頁碼按鈕間距
			if (page === currentPage) {
				btn.classList.add('current-page'); // 添加 'current-page' 樣式
			}
			return btn;
		}
		/***************************/

		/* 搜尋功能區塊 */
		function searchEmployees() {
			loadEmployees(1); // 重新載入員工清單
		}
		/***************************/

		/* 刪除區塊 */
		function deleteSelectedEmployees() {
			const selectedEmployees = document.querySelectorAll('input[name="select_employee"]:checked');
			if (selectedEmployees.length === 0) {
				alert('請選擇要刪除的員工');
				return;
			}

			if (confirm('確定要刪除選中的員工嗎？')) {
				const ids = Array.from(selectedEmployees).map(checkbox => checkbox.value);

				fetch(`${API_BASE}/employees/delete_employees.php`, {
					method: 'POST',
					body: JSON.stringify({ ids }),
					headers: { 'Content-Type': 'application/json' }
				})
					.then(response => response.json())
					.then(data => {
						if (data.result) {
							alert(data.msg);
							loadEmployees(currentPage);
						} else {
							alert(data.msg);
						}
					})
					.catch(error => console.error('Error:', error));
			}
		}
		/***************************/

		/* 新增員工區塊 */
		/* 新增員工 */
		async function showAddEmployeeModal() {
			const form = document.getElementById('createEmployeeForm');
			form.reset(); // 清空表單

			// 載入職稱 / 部門（await，確保選單完成）
			const jobSel = document.getElementById('jobtitle_id');
			const depSel = document.getElementById('department_id');

			await loadJobtitlesToSelect(jobSel);
			await loadDepartmentsToSelect(depSel);

			// 自動依文字調整寬度（如果你有用 autoResizeSelect）
			autoResizeSelect(jobSel);
			autoResizeSelect(depSel);

			const modal = document.getElementById('addEmployeeModal');
			modal.removeAttribute('data-id'); // 確保是 add 模式
			modal.showModal();

			// submit handler（add 模式）
			form.removeEventListener('submit', submitFormHandler);
			form.addEventListener('submit', e => submitFormHandler(e, false));
		}

		// 初始化 Sndcategory 選單
		async function loadSndcategories(category = null) {
			let employee = document.querySelector('input[name="employee"]:checked').value;

			console.log("loadSndcategories initial category: " + category);

			try {
				const response = await fetch(`${API_BASE}/employees/get_sndcategories.php`, {
					method: 'POST',
					headers: { 'Content-Type': 'application/json' },
					body: JSON.stringify({ employee, category })
				});

				const data = await response.json();

				if (data.result) {
					const sndcategoryDropdown = document.getElementById('sndcategoryDropdown');
					sndcategoryDropdown.innerHTML = ''; // 清空現有項目

					data.data.forEach(item => {
						console.log("SNDCATEGORY: " + item.name);
						const div = document.createElement('div');
						div.className = 'dropdown-item';
						div.textContent = item.name;
						div.onclick = () => selectSndcategory(item.name);
						sndcategoryDropdown.appendChild(div);
					});
				} else {
					console.error('類別載入失敗:', data.msg);
				}
			} catch (error) {
				console.error('Error:', error);
			}
		}

		// 初始化 Trdcategory 選單，依據 Category 的值
		async function loadTrdcategories(category = null, sndcategory = null) {
			let employee = document.querySelector('input[name="employee"]:checked').value;

			console.log("loadTrdcategories initial category: " + category + ", sndcategory: " + sndcategory);

			try {
				const response = await fetch(`${API_BASE}/employees/get_trdcategories.php`, {
					method: 'POST',
					headers: { 'Content-Type': 'application/json' },
					body: JSON.stringify({ employee, category, sndcategory })
				});

				const data = await response.json();

				if (data.result) {
					const trdcategoryDropdown = document.getElementById('trdcategoryDropdown');
					trdcategoryDropdown.innerHTML = ''; // 清空現有項目

					data.data.forEach(item => {
						console.log("TRDCATEGORY: " + item.name);
						const div = document.createElement('div');
						div.className = 'dropdown-item';
						div.textContent = item.name;
						div.onclick = () => selectTrdcategory(item.name);
						trdcategoryDropdown.appendChild(div);
					});
				} else {
					console.error('細類載入失敗:', data.msg);
				}
			} catch (error) {
				console.error('Error:', error);
			}
		}

		// 顯示下拉選單
		function showDropdown(dropdownId) {
			const dropdown = document.getElementById(dropdownId);
			dropdown.style.display = 'block';
		}

		// 選擇 Category 選項
		function selectCategory(category) {
			const input = document.getElementById('categoryCombo');
			input.value = category; // 設置輸入框的值
			document.getElementById('categoryDropdown').style.display = 'none'; // 隱藏選單
			loadSndcategories(category); // 加載對應的 Sndcategory
		}

		// 選擇 Sndcategory 選項
		function selectSndcategory(sndcategory) {
			var category = document.getElementById('categoryCombo').value;
			const input = document.getElementById('sndcategoryCombo');
			input.value = sndcategory; // 設置輸入框的值
			document.getElementById('sndcategoryDropdown').style.display = 'none'; // 隱藏選單
			loadTrdcategories(category, sndcategory); // 加載對應的 Trdcategory
		}

		// 選擇 Trdcategory 選項
		function selectTrdcategory(trdcategory) {
			const input = document.getElementById('trdcategoryCombo');
			input.value = trdcategory; // 設置輸入框的值
			document.getElementById('trdcategoryDropdown').style.display = 'none'; // 隱藏選單
		}

		// 當 Category 的值改變時
		function onCategoryChange(value) {
			console.log('Bodypart 值改變為:', value);
			loadCategories(value); // 根據新的值加載 Trdcategory
		}

		// 當 Sndcategory 的值改變時
		function onSndcategoryChange(value) {
			var category = document.getElementById('categoryCombo').value;
			console.log('Sndcategory 值改變為:', value);
			loadTrdcategories(category, value); // 根據新的值加載 Trdcategory
		}

		// 切換下拉選單顯示狀態
		function toggleDropdown(dropdownId) {
			const dropdown = document.getElementById(dropdownId);
			if (dropdown.style.display === 'block') {
				dropdown.style.display = 'none'; // 隱藏下拉選單
			} else {
				dropdown.style.display = 'block'; // 顯示下拉選單
			}
		}

		// 過濾選項
		function filterOptions(dropdownId, filterValue) {
			const dropdown = document.getElementById(dropdownId);
			const items = dropdown.querySelectorAll('.dropdown-item');
			let hasVisibleItems = false;

			items.forEach(item => {
				if (item.textContent.toLowerCase().includes(filterValue.toLowerCase())) {
					item.style.display = 'block';
					hasVisibleItems = true;
				} else {
					item.style.display = 'none';
				}
			});

			dropdown.style.display = hasVisibleItems ? 'block' : 'none'; // 過濾後控制顯示
		}

		// 點擊外部隱藏下拉選單
		document.addEventListener('click', function (event) {
			const categoryInput = document.getElementById('categoryCombo');
			const categoryDropdown = document.getElementById('categoryDropdown');
			const sndcategoryInput = document.getElementById('sndcategoryCombo');
			const sndcategoryDropdown = document.getElementById('sndcategoryDropdown');
			const trdcategoryInput = document.getElementById('trdcategoryCombo');
			const trdcategoryDropdown = document.getElementById('trdcategoryDropdown');

			// 如果點擊的不是 category 輸入框或選單，隱藏 category 下拉選單
			if (categoryDropdown && !categoryInput.contains(event.target) && !categoryDropdown.contains(event.target)) {
				categoryDropdown.style.display = 'none';
			}

			// 如果點擊的不是 sndcategory 輸入框或選單，隱藏 sndcategory 下拉選單
			if (sndcategoryDropdown && !sndcategoryInput.contains(event.target) && !sndcategoryDropdown.contains(event.target)) {
				sndcategoryDropdown.style.display = 'none';
			}

			// 如果點擊的不是 trdcategory 輸入框或選單，隱藏 trdcategory 下拉選單
			if (trdcategoryDropdown && !trdcategoryInput.contains(event.target) && !trdcategoryDropdown.contains(event.target)) {
				trdcategoryDropdown.style.display = 'none';
			}
		});

		// 偵測 employee 的 click/change 事件
		document.addEventListener("DOMContentLoaded", function () {
			document.querySelectorAll('input[name="employee"]').forEach((radio) => {
				radio.addEventListener('change', loadCategories);
			});
		});

		// 提交新增員工的處理函數
		function submitFormHandler(event) {
			event.preventDefault(); // 防止表單的預設提交行為

			// —— 驗證開始 ——  
			// 1. 出生日期
			const birthDateEl = document.getElementById('birth_date');
			if (!birthDateEl.value) {
				alert('請填寫出生日期！(222222222222222)');
				birthDateEl.focus();
				return;
			}

			// 2. 到職日期
			const hireDateEl = document.getElementById('hire_date');
			if (!hireDateEl.value) {
				alert('請填寫到職日期！');
				hireDateEl.focus();
				return;
			}

			// 3. 國籍
			const countryEl = document.getElementById('address_country');
			if (!countryEl.value.trim()) {
				alert('請填寫國籍／國家！');
				countryEl.focus();
				return;
			}

			// 4. 證件號碼
			const idEl = document.getElementById('national_id');
			if (!idEl.value.trim()) {
				alert('請填寫證件號碼！');
				idEl.focus();
				return;
			}

			// 5. 手機號碼
			const mobileEl = document.getElementById('mobile_phone');
			const mobileVal = mobileEl.value.trim();
			const mobileRegex = /^[0-9]{10}$/;
			if (!mobileRegex.test(mobileVal)) {
				alert('手機號碼格式錯誤，請輸入 10 位數字！');
				mobileEl.focus();
				return;
			}

			// 6. 最高學歷
			const eduEl = document.getElementById('education_level');
			if (!eduEl.value) {
				alert('請選擇最高學歷！');
				eduEl.focus();
				return;
			}

			// 7. 血型
			const bloodEl = document.getElementById('blood_type');
			if (!bloodEl.value) {
				alert('請選擇血型！');
				bloodEl.focus();
				return;
			}

			// 8. 畢業年份
			const graduationEl = document.getElementById('graduation_year');
			const graduationVal = graduationEl.value.trim();
			const graduationRegex = /^[0-9]{4}$/;
			if (!graduationRegex.test(graduationVal)) {
				alert('畢業年份格式錯誤，請輸入西元年份 4 位數字！');
				graduationEl.focus();
				return;
			}

			const jobtitleEl = document.querySelector('[name="jobtitle_id"]');
			if (!jobtitleEl || !jobtitleEl.value) {
				alert('請選擇職稱！');
				jobtitleEl.focus();
				return;
			}

			const departmentEl = document.querySelector('[name="department_id"]');
			if (!departmentEl || !departmentEl.value) {
				alert('請選擇部門！');
				departmentEl.focus();
				return;
			}
			// —— 驗證結束 ——

			// 收集表單數據
			const formData = new FormData(event.target);
			const newData = {
				cname: formData.get('cname'),
				ename: formData.get('ename'),
				gender: formData.get('gender'),
				birth_date: formData.get('birth_date'),
				blood_type: formData.get('blood_type'),
				nationality: formData.get('nationality'),
				marital_status: formData.get('marital_status'),
				hire_date: formData.get('hire_date'),
				national_id_type: formData.get('national_id_type'),
				national_id: formData.get('national_id'),
				mobile_phone: formData.get('mobile_phone'),
				home_phone: formData.get('home_phone'),
				email: formData.get('email'),
				address_country: formData.get('address_country'),
				address_city: formData.get('address_city'),
				address_district: formData.get('address_district'),
				address_postcode: formData.get('address_postcode'),
				address_detail: formData.get('address_detail'),
				emergency_name: formData.get('emergency_name'),
				emergency_relation: formData.get('emergency_relation'),
				emergency_phone: formData.get('emergency_phone'),
				education_level: formData.get('education_level'),
				graduation_year: formData.get('graduation_year'),
				graduated_from: formData.get('graduated_from'),
				major: formData.get('major'),
				work_experience: formData.get('work_experience'),
				jobtitle_id: formData.get('jobtitle_id'),
				department_id: formData.get('department_id')
			};

			console.log('NewData: ' + JSON.stringify(newData));

			// 發送 AJAX 請求
			fetch(`${API_BASE}/employees/create_employee.php`, {
				method: 'POST',
				headers: { 'Content-Type': 'application/json' },
				body: JSON.stringify(newData),
			})
				.then(response => response.json())
				.then(data => {
					if (data.result) {
						alert('員工新增成功');
						closeAddEmployeeModal(); // 關閉模態視窗
						loadEmployees(totalPages+1); // 重新載入員工清單
						// 重置表單內容
						this.reset();
					} else {
						alert('新增失敗: ' + data.msg);
					}
				})
				.catch(error => {
					console.error('Error:', error);
				});
		}

		// 關閉新增員工的 Modal
		function closeAddEmployeeModal() {
			addEmployeeModal.close();
		}
		/***************************/


		/* 編輯/修改 區塊 */
		/* 編輯員工 */
		async function showEditEmployeeModal(employeeId) {
			if (!employeeId) return;

			console.log('edit employee id:', employeeId);

			const modal = document.getElementById('editEmployeeModal');
			const form  = document.getElementById('editEmployeeForm');

			modal.setAttribute('data-id', employeeId); // 記錄員工 ID

			try {
				// ① 先抓員工資料
				const response = await fetch(`${API_BASE}/employees/get_employee.php?id=${employeeId}`);
				const data = await response.json();

				if (!data.result) {
					alert('無法加載員工數據：' + data.msg);
					return;
				}

				const row = data.data;

				// ② 載入職稱 / 部門（帶入 selected）
				const jobSel = document.getElementById('editJobtitle_id');
				const depSel = document.getElementById('editDepartment_id');

				await loadJobtitlesToSelect(jobSel, row.jobtitle_id);
				await loadDepartmentsToSelect(depSel, row.department_id);

				autoResizeSelect(jobSel);
				autoResizeSelect(depSel);

				// ③ 填入其他欄位
				document.getElementById('editCname').value = row.cname;
				document.getElementById('editEname').value = row.ename;
				document.getElementById('editGender').value = row.gender;
				document.getElementById('editBirth_date').value = row.birth_date;
				document.getElementById('editHire_date').value = row.hire_date;
				document.getElementById('editNationality').value = row.nationality;
				document.getElementById('editMarital_status').value = row.marital_status;
				document.getElementById('editBlood_type').value = row.blood_type;
				document.getElementById('editNational_id_type').value = row.national_id_type;
				document.getElementById('editNational_id').value = row.national_id;
				document.getElementById('editMobile_phone').value = row.mobile_phone;
				document.getElementById('editHome_phone').value = row.home_phone;
				document.getElementById('editEmail').value = row.email;
				document.getElementById('editAddress_country').value = row.address_country;
				document.getElementById('editAddress_city').value = row.address_city;
				document.getElementById('editAddress_district').value = row.address_district;
				document.getElementById('editAddress_postcode').value = row.address_postcode;
				document.getElementById('editAddress_detail').value = row.address_detail;
				document.getElementById('editEmergency_name').value = row.emergency_name;
				document.getElementById('editEmergency_relation').value = row.emergency_relation;
				document.getElementById('editEmergency_phone').value = row.emergency_phone;
				document.getElementById('editEducation_level').value = row.education_level;
				document.getElementById('editGraduated_from').value = row.graduated_from;
				document.getElementById('editGraduation_year').value = row.graduation_year;
				document.getElementById('editMajor').value = row.major;
				document.getElementById('editWork_experience').value = row.work_experience;
				document.getElementById('editJobtitle_id').value = row.jobtitle_id;
				document.getElementById('editDepartment_id').value = row.department_id;

				// ④ 資料都準備好，再顯示 modal
				modal.showModal();

				// ⭐ 強制捲回 dialog 最上方
				openModalAtTop(
					document.getElementById('editEmployeeModal'),
					document.getElementById('editCname')
				);
			} catch (error) {
				console.error('Error loading employee:', error);
				alert('讀取員工資料時發生錯誤');
			}
		}

		// 關閉編輯模態框
		function closeEditEmployeeModal() {
			document.getElementById('editEmployeeModal').close();
		}

		// 選擇 editCategoryCombo
		function selectEditCategory(name) {
			const categoryInput = document.getElementById('editCategoryCombo');
			categoryInput.value = name; // 更新輸入框值
			loadEditSndcategories(name); // 加載對應的細類
			document.getElementById('editCategoryDropdown').style.display = 'none'; // 隱藏下拉選單
		}

		// 選擇 editSndcategoryCombo
		function selectEditSndcategory(name) {
			var category = document.getElementById('editCategoryCombo').value;
			const sndcategoryInput = document.getElementById('editSndcategoryCombo');
			sndcategoryInput.value = name; // 更新輸入框值
			loadEditTrdcategories(category, name); // 加載對應的細類
			document.getElementById('editSndcategoryDropdown').style.display = 'none'; // 隱藏下拉選單
		}

		// 選擇 editTrdcategoryCombo
		function selectEditTrdcategory(name) {
			const trdcategoryInput = document.getElementById('editTrdcategoryCombo');
			trdcategoryInput.value = name; // 更新輸入框值
			document.getElementById('editTrdcategoryDropdown').style.display = 'none'; // 隱藏下拉選單
		}

		// 輸入選擇 editCategoryCombo
		function onEditCategoryChange(newCategory) {
			if (newBodygory) {
				console.log('Category 改變為:', newCategory);
				loadEditSndcategories(newCategory); // 根據新主類加載類別
			} else {
				console.log('Category 為空，清空 Sndcategory 及 Trdcategory');
				document.getElementById('editSndcategoryCombo').value = ''; // 清空輸入框
				document.getElementById('editSndcategoryDropdown').innerHTML = ''; // 清空類別選單
				document.getElementById('editTrdcategoryCombo').value = ''; // 清空輸入框
				document.getElementById('editTrdcategoryDropdown').innerHTML = ''; // 清空細類選單
			}
		}

		// 輸入選擇 editSndcategoryCombo
		function onEditSndcategoryChange(newSndcategory) {
			if (newSndcategory) {
				var category = document.getElementById('editCategoryCombo').value;
				console.log('Sndcategory 改變為:', newSndcategory);
				loadEditTrdcategories(category, newSndcategory); // 根據新類別加載細類
			} else {
				console.log('Category 為空，清空 Trdcategory');
				document.getElementById('editTrdcategoryCombo').value = ''; // 清空輸入框
				document.getElementById('editTrdcategoryDropdown').innerHTML = ''; // 清空細類選單
			}
		}

		// 定義 submitEditFormHandler 函數
		document.getElementById('editEmployeeForm').addEventListener('submit', function (event) {
			event.preventDefault(); // 防止表單的預設提交行為

			// —— 驗證開始 ——  
			// 1. 出生日期
			const birthDateEl = document.getElementById('editBirth_date');
			if (!birthDateEl.value) {
				alert('請填寫出生日期！');
				birthDateEl.focus();
				return;
			}

			// 2. 到職日期
			const hireDateEl = document.getElementById('editHire_date');
			if (!hireDateEl.value) {
				alert('請填寫到職日期！');
				hireDateEl.focus();
				return;
			}

			// 3. 國籍
			const countryEl = document.getElementById('editAddress_country');
			if (!countryEl.value.trim()) {
				alert('請填寫國籍／國家！');
				countryEl.focus();
				return;
			}

			// 4. 證件號碼
			const idEl = document.getElementById('editNational_id');
			if (!idEl.value.trim()) {
				alert('請填寫證件號碼！');
				idEl.focus();
				return;
			}

			// 5. 手機號碼
			const mobileEl = document.getElementById('editMobile_phone');
			const mobileVal = mobileEl.value.trim();
			const mobileRegex = /^[0-9]{10}$/;
			if (!mobileRegex.test(mobileVal)) {
				alert('手機號碼格式錯誤，請輸入 10 位數字！');
				mobileEl.focus();
				return;
			}

			// 6. 最高學歷
			const eduEl = document.getElementById('editEducation_level');
			if (!eduEl.value) {
				alert('請選擇最高學歷！');
				eduEl.focus();
				return;
			}

			// 7. 血型
			const bloodEl = document.getElementById('editBlood_type');
			if (!bloodEl.value) {
				alert('請選擇血型！');
				bloodEl.focus();
				return;
			}

			// 8. 畢業年份
			const graduationEl = document.getElementById('editGraduation_year');
			const graduationVal = graduationEl.value.trim();
			const graduationRegex = /^[0-9]{4}$/;
			if (!graduationRegex.test(graduationVal)) {
				alert('畢業年份格式錯誤，請輸入西元年份 4 位數字！');
				graduationEl.focus();
				return;
			}

			// —— 驗證結束 ——

			// 取得表單中基本資料
			const modal = document.getElementById('editEmployeeModal');
			const employeeId = modal.getAttribute('data-id');
			const cname = document.getElementById('editCname').value;
			const ename = document.getElementById('editEname').value;
			const gender = document.getElementById('editGender').value;
			const birth_date = document.getElementById('editBirth_date').value;
			const blood_type = document.getElementById('editBlood_type').value;
			const nationality = document.getElementById('editNationality').value;
			const marital_status = document.getElementById('editMarital_status').value;
			const hire_date = document.getElementById('editHire_date').value;
			const national_id_type = document.getElementById('editNational_id_type').value;
			const national_id = document.getElementById('editNational_id').value;
			const mobile_phone = document.getElementById('editMobile_phone').value;
			const home_phone = document.getElementById('editHome_phone').value;
			const email = document.getElementById('editEmail').value;
			const address_country = document.getElementById('editAddress_country').value;
			const address_city = document.getElementById('editAddress_city').value;
			const address_district = document.getElementById('editAddress_district').value;
			const address_postcode = document.getElementById('editAddress_postcode').value;
			const address_detail = document.getElementById('editAddress_detail').value;
			const emergency_name = document.getElementById('editEmergency_name').value;
			const emergency_relation = document.getElementById('editEmergency_relation').value;
			const emergency_phone = document.getElementById('editEmergency_phone').value;
			const education_level = document.getElementById('editEducation_level').value;
			const graduation_year = document.getElementById('editGraduation_year').value;
			const graduated_from = document.getElementById('editGraduated_from').value;
			const major = document.getElementById('editMajor').value;
			const work_experience = document.getElementById('editWork_experience').value;
			const jobtitle_id = document.getElementById('editJobtitle_id').value;
			const department_id = document.getElementById('editDepartment_id').value;

			if (!employeeId) {
				alert('員工 ID 丟失，無法更新');
				return;
			}

			// 組合 newData 物件屬性維持為陣列型態
			const newData = {
				id: employeeId,
				cname: cname,
				ename: ename,
				gender: gender,
				birth_date: birth_date,
				blood_type: blood_type,
				nationality: nationality,
				marital_status: marital_status,
				hire_date: hire_date,
				national_id_type: national_id_type,
				national_id: national_id,
				mobile_phone: mobile_phone,
				home_phone: home_phone,
				email: email,
				address_country: address_country,
				address_city: address_city,
				address_district: address_district,
				address_postcode: address_postcode,
				address_detail: address_detail,
				emergency_name: emergency_name,
				emergency_relation: emergency_relation,
				emergency_phone: emergency_phone,
				education_level: education_level,
				graduation_year: graduation_year,
				graduated_from: graduated_from,
				major: major,
				work_experience: work_experience,
				jobtitle_id: jobtitle_id,
				department_id: department_id
			};

			console.log('提交數據:', newData);

			// 發送 AJAX 請求
			fetch(`${API_BASE}/employees/update_employee.php`, {
				method: 'POST',
				headers: { 'Content-Type': 'application/json' },
				body: JSON.stringify(newData)
			})
				.then(async response => {
					if (!response.ok) {
						// 先拿到原始文字，幫你快速定位 PHP 報錯
						const text = await response.text();
						console.error('伺服器回傳錯誤：', response.status, text);
						throw new Error(`Server error ${response.status}`);
					}
					// 正常才 parse JSON
					return response.json();
				})
				.then(data => {
					if (data.result) {
						alert('員工更新成功');
						closeEditEmployeeModal();
						loadEmployees(currentPage);
					} else {
						alert('更新失敗：' + data.msg);
					}
				})
				.catch(err => {
					console.error('Fetch/解析失敗：', err);
				});
		});

		// 清除先前的建議清單
		function clearSuggestions() {
			const autocompleteLists = document.querySelectorAll('.autocomplete-list');
			autocompleteLists.forEach(list => list.remove());
		}

		// 顯示建議列表
		function displaySuggestions(items, inputElement) {
			const autocomplete = document.createElement('div');
			autocomplete.className = 'autocomplete-list';
			items.forEach((item, index) => {
				const option = document.createElement('div');
				option.textContent = item.name; // 顯示藥材或方劑名稱
				option.dataset.id = item.id; // 儲存該項目的 id
				option.dataset.index = index;
				option.addEventListener('click', function () {
					inputElement.value = item.name;
					inputElement.dataset.id = item.id; // 將 id 存入 inputElement 的 dataset
					clearSuggestions();
				});
				autocomplete.appendChild(option);
			});
			if (items.length > 0) {
				inputElement.parentNode.appendChild(autocomplete);
			}
		}

		// 取得藥材建議：以 GET 方式呼叫 API
		function fetchMaterialsSuggestions(query, inputElement) {
			clearSuggestions();
			if (query.length < 1) {
				return;
			}
			fetch(`${API_BASE}/materials/search_materials.php?query=${encodeURIComponent(query)}`)
				.then(response => response.json())
				.then(data => {
					console.log("藥材 API 回傳資料：", data);
					if (data.result) {
						displaySuggestions(data.materials, inputElement);
					} else {
						console.error('Error fetching materials:', data.msg);
					}
				})
				.catch(error => {
					console.error('Error fetching materials:', error);
				});
		}

		// 取得方劑建議（假設 API 為 search_formulas.php）
		function fetchFormulasSuggestions(query, inputElement) {
			clearSuggestions();
			if (query.length < 1) {
				return;
			}
			fetch(`${API_BASE}/formulas/search_formulas.php?query=${encodeURIComponent(query)}`)
				.then(response => response.json())
				.then(data => {
					console.log("方劑 API 回傳資料：", data);
					if (data.result) {
						displaySuggestions(data.formulas, inputElement);
					} else {
						console.error('Error fetching formulas:', data.msg);
					}
				})
				.catch(error => {
					console.error('Error fetching formulas:', error);
				});
		}

		/*****************************/
		// 部門區塊
		/* 初始化 category 選單 */
		async function loadCategories() {
			let employee = document.querySelector('input[name="employee"]:checked').value;
			console.log("loadCategories initial ...");

			try {
				const response = await fetch(`${API_BASE}/employees/get_categories.php`, {
					method: 'POST',
					headers: { 'Content-Type': 'application/json' },
					body: JSON.stringify({ employee })
				});
				const data = await response.json();

				if (data.result) {
					const categoryDropdown = document.getElementById('categoryDropdown');
					categoryDropdown.innerHTML = ''; // 清空現有項目

					data.data.forEach(item => {
						console.log("CATEGORY: " + item.name);
						const div = document.createElement('div');
						div.className = 'dropdown-item';
						div.textContent = item.name;
						div.onclick = () => selectCategory(item.name);
						categoryDropdown.appendChild(div);
					});
				} else {
					console.error('主類載入失敗:', data.msg);
				}
			} catch (error) {
				console.error('Error:', error);
			}
		}

		/* 載入函式部門主管選單 */
		async function loadEmployeeOptions() {
			// 已載入過就不要再抓
			if (Array.isArray(window.employeeOptions) && window.employeeOptions.length > 0) {
				return window.employeeOptions;
			}

			const response = await fetch(`${API_BASE}/employees/get_employees.php`);
			const data = await response.json();

			if (!data.result) {
				console.error('Failed to load employees:', data.msg);
				window.employeeOptions = [];
				return window.employeeOptions;
			}

			// ⭐ 關鍵：一定要掛在 window 上
			window.employeeOptions = data.data.records || [];
			return window.employeeOptions;
		}

		function renderEmployeeDropdown(dropdownId, filterText = '') {
			const dropdown = document.getElementById(dropdownId);
			if (!dropdown) return;

			dropdown.innerHTML = '';
			const keyword = filterText.toLowerCase();
			const employees = window.employeeOptions || [];

			employees
				.filter(emp => {
					const text = `${emp.employee_id} ${emp.cname}`.toLowerCase();
					return text.includes(keyword);
				})
				.forEach(emp => {
					const div = document.createElement('div');
					div.className = 'option';
					div.textContent = `${emp.employee_id} ${emp.cname}`;
					div.dataset.value = emp.employee_id;

					// ✅ 就改這裡
					div.addEventListener('click', () => {
						const inputId = dropdownId.replace('Dropdown', 'Combo');
						const input = document.getElementById(inputId);

						if (input) {
							input.value = `${emp.employee_id} ${emp.cname}`;
							input.dataset.value = emp.employee_id;
						}

						dropdown.classList.remove('active');
					});

					dropdown.appendChild(div);
				});
		}

		function toggleDropdown(dropdownId) {
			const dropdown = document.getElementById(dropdownId);
			if (!dropdown) return;

			dropdown.classList.toggle('active');
		}

		function filterOptions(dropdownId, value) {
			renderEmployeeDropdown(dropdownId, value);
			const dropdown = document.getElementById(dropdownId);
			if (dropdown) dropdown.classList.add('active');
		}

		function enableComboKeyboard(inputId, dropdownId) {
			const input = document.getElementById(inputId);
			const dropdown = document.getElementById(dropdownId);

			if (!input || !dropdown) return;

			let currentIndex = -1;

			input.addEventListener('keydown', (e) => {
				const options = Array.from(dropdown.querySelectorAll('.option'));
				if (options.length === 0) return;

				switch (e.key) {
					case 'ArrowDown':
						e.preventDefault();
						dropdown.classList.add('active');
						currentIndex = (currentIndex + 1) % options.length;
						updateActiveOption(options, currentIndex);
						break;

					case 'ArrowUp':
						e.preventDefault();
						dropdown.classList.add('active');
						currentIndex =
							(currentIndex - 1 + options.length) % options.length;
						updateActiveOption(options, currentIndex);
						break;

					case 'Enter':
						if (currentIndex >= 0 && options[currentIndex]) {
							e.preventDefault();
							options[currentIndex].click();
							currentIndex = -1;
						}
						break;

					case 'Escape':
						dropdown.classList.remove('active');
						currentIndex = -1;
						break;
				}
			});

			// 當重新 render dropdown 時，重置 index
			const observer = new MutationObserver(() => {
				currentIndex = -1;
			});
			observer.observe(dropdown, { childList: true });
		}

		function updateActiveOption(options, index) {
			options.forEach(opt => opt.classList.remove('active'));
			const active = options[index];
			if (active) {
				active.classList.add('active');
				active.scrollIntoView({ block: 'nearest' });
			}
		}

		/* 載入 */
		function loadDepartments(page = 1) {
			currentPage = page;  // 當前頁數
			const pSize = 15;
			var conditionObj = {};

			var query = document.getElementById('searchDepartmentsInput').value.trim().toLowerCase();  // 取得搜尋關鍵字

			// 組合查詢參數
			const params = new URLSearchParams({
				page: page,
				size: pSize,
				condition: query,
			});

			fetch(`${API_BASE}/employees/get_departments.php?${params.toString()}`)
				.then(response => response.json())
				.then(data => {
					if (data.result) {
						const departmentsTableBody = document.querySelector('#departments-table tbody');
						departmentsTableBody.innerHTML = ''; // 清空現有的表格內容
						data.data.records.forEach((record, index) => {
							const row = document.createElement('tr');

							row.innerHTML = `
			<td><input type="checkbox" name="select_department" id="department-${record.department_id}" value="${record.department_id}">${record.department_id}</td>
			<td class="department-name">${record.name}</td>
			<td class="department-name">${record.en_name}</td>
			<td class="department-parent_dept">${record.parent_dept}</td>
			<td class="department-manager_id">${record.manager_id} ${record.cname}</td>
			<td class="department-extension">${record.extension}</td>
		    `;

							// 點擊 "名稱" 欄位彈出修改視窗
							row.querySelector('.department-name').addEventListener('click', function(event) {
								event.stopPropagation(); // 阻止事件冒泡
								showEditDepartmentModal(record.department_id); // 點擊名稱彈出修改視窗
							});

							departmentsTableBody.appendChild(row);
						});
						// 生成分頁按鈕
						totalPages = Math.ceil(data.data.ttl / pSize); // 計算總頁數
						generatePaginationControls(); // 生成分頁按鈕
					} else {
						alert('無法獲取部門列表');
					}
				})
				.catch(error => console.error('Error loading Departments list:', error));
		}

		/* 搜尋 */
		function searchDepartments() {
			loadDepartments(); // 重新載入部門清單
		}

		/* 刪除 */
		function deleteSelectedDepartments() {
			const selectedDepartments = document.querySelectorAll('input[name="select_department"]:checked');
			if (selectedDepartments.length === 0) {
				alert('請選擇要刪除的部門');
				return;
			}

			if (confirm('確定要刪除選中的部門嗎？')) {
				const ids = Array.from(selectedDepartments).map(checkbox => checkbox.value);

				fetch(`${API_BASE}/employees/del_departments.php`, {
					method: 'POST',
					body: JSON.stringify({ ids }),
					headers: { 'Content-Type': 'application/json' }
				})
					.then(response => response.json())
					.then(data => {
						if (data.result) {
							alert(data.msg);
							loadDepartments();
						} else {
							alert(data.msg);
						}
					})
					.catch(error => console.error('Error:', error));
			}
		}

		/* 新增 */
		async function showAddDepartmentModal() {
			document.getElementById('createDepartmentForm').reset(); // 清空表單

			const modal = document.getElementById('addDepartmentModal');
			await loadEmployeeOptions();
			renderEmployeeDropdown('manager_idDropdown', '');
			enableComboKeyboard('manager_idCombo', 'manager_idDropdown');
			modal.showModal(); // 顯示模態視窗

			console.log("Modal is open");

			// 先解除以前綁定的 submit 事件，再綁定新的事件
			const createForm = document.getElementById('createDepartmentForm');
			createForm.removeEventListener('submit', submitAddDepartmentFormHandler);  // 如果以前有綁定過
			createForm.addEventListener('submit', submitAddDepartmentFormHandler);
		}

		/* 提交新增的處理函數 */
		function submitAddDepartmentFormHandler(event) {
			event.preventDefault(); // 防止表單的預設提交行為

			// 收集表單數據
			const formData = new FormData(event.target);
			const parent_dept = formData.get('parent_dept').trim() === ''
				? 0 : parseInt(formData.get('parent_dept').trim());
			const manager_id = formData.get('manager_idCombo').trim() === ''
				? null : formData.get('manager_idCombo').trim();
			const extension = formData.get('extension').trim() === ''
				? null : formData.get('extension').trim();
			const newData = {
				name: formData.get('name'),
				en_name: formData.get('en_name'),
				parent_dept: parent_dept,
				manager_id: manager_id,
				extension: extension
			};

			console.log('NewData: ' + JSON.stringify(newData));

			// 發送 AJAX 請求
			fetch(`${API_BASE}/employees/create_department.php`, {
				method: 'POST',
				headers: { 'Content-Type': 'application/json' },
				body: JSON.stringify(newData),
			})
				.then(response => response.json())
				.then(data => {
					if (data.result) {
						alert('部門新增成功');
						closeAddDepartmentModal(); // 關閉模態視窗
						loadDepartments(totalPages+1); // 重新載入部門清單
						// 重置表單內容
						this.reset();
					} else {
						alert('新增失敗: ' + data.msg);
					}
				})
				.catch(error => {
					console.error('Error:', error);
				});
		}

		/* 關閉新增部門的 Modal */
		function closeAddDepartmentModal() {
			addDepartmentModal.close();
		}

		/* 編輯/修改 區塊 */
		async function showEditDepartmentModal(departmentId) {
			console.log('sid:', departmentId);

			const modal = document.getElementById('editDepartmentModal');
			modal.showModal();

			try {
				// 1️⃣ 確保員工選項已載入（Combo-box 的資料來源）
				await loadEmployeeOptions();
				renderEmployeeDropdown('editManager_idDropdown', '');
				enableComboKeyboard('editManager_idCombo', 'editManager_idDropdown');

				const employees = window.employeeOptions || [];

				// 2️⃣ 取得部門資料
				const response = await fetch(
					`${API_BASE}/employees/get_department.php?id=${departmentId}`
				);
				const data = await response.json();

				if (!data.result) {
					alert('無法加載部門數據：' + data.msg);
					return;
				}

				const row = data.data;

				// 3️⃣ 填充部門基本資料
				document.getElementById('editDepartmentId').value = departmentId;
				document.getElementById('editName').value = row.name || '';
				document.getElementById('editEnName').value = row.en_name || '';
				document.getElementById('editParent_dept').value = row.parent_dept || '';
				document.getElementById('editExtension').value = row.extension || '';

				// 4️⃣ 回填「部門主管」Combo-box（重點）
				const managerInput = document.getElementById('editManager_idCombo');
				managerInput.value = '';
				managerInput.dataset.value = '';

				if (row.manager_id) {
					const emp = employees.find(
						e => String(e.employee_id) === String(row.manager_id)
					);
					if (emp) {
						managerInput.value = `${emp.employee_id} ${emp.cname}`;
						managerInput.dataset.value = emp.employee_id;
					}
				}

				// 5️⃣ 預先渲染下拉選單（不強制打開）
				renderEmployeeDropdown('editManager_idDropdown', '');

			} catch (error) {
				console.error('Error in showEditDepartmentModal:', error);
				alert('載入部門資料時發生錯誤');
			}
		}

		/* 關閉編輯模態框 */
		function closeEditDepartmentModal() {
			document.getElementById('editDepartmentModal').close();
		}

		/* 定義 submitEditDepartmentFormHandler 函數 */
		document.getElementById('editDepartmentForm').addEventListener('submit', function (event) {
			event.preventDefault(); // 防止表單的預設提交行為

			// 取得表單中基本資料
			const modal = document.getElementById('editDepartmentModal');
			// const departmentId = modal.getAttribute('data-id');
			const departmentId = document.getElementById('editDepartmentId').value;
			const name = document.getElementById('editName').value;
			const enname = document.getElementById('editEnName').value;
			const parent_dept = document.getElementById('editParent_dept').value;
			const manager_id = document.getElementById('editManager_idCombo').value;
			const extension = document.getElementById('editExtension').value;

			if (!departmentId) {
				alert('部門 ID 丟失，無法更新');
				return;
			}

			// 組合 newData 物件屬性維持為陣列型態
			const newData = {
				department_id: departmentId,
				name: name,
				en_name: enname,
				parent_dept: parent_dept,
				manager_id: manager_id,
				extension: extension
			};

			console.log('提交數據:', newData);

			// 發送 AJAX 請求
			fetch(`${API_BASE}/employees/update_department.php`, {
				method: 'POST',
				headers: { 'Content-Type': 'application/json' },
				body: JSON.stringify(newData)
			})
				.then(async response => {
					if (!response.ok) {
						// 先拿到原始文字，幫你快速定位 PHP 報錯
						const text = await response.text();
						console.error('伺服器回傳錯誤：', response.status, text);
						throw new Error(`Server error ${response.status}`);
					}
					// 正常才 parse JSON
					return response.json();
				})
				.then(data => {
					if (data.result) {
						alert('部門更新成功');
						closeEditDepartmentModal();
						loadDepartments();
					} else {
						alert('更新失敗：' + data.msg);
					}
				})
				.catch(err => {
					console.error('Fetch/解析失敗：', err);
				});
		});
		/*****************************/
		// 職稱使用
		/* 載入 */
		function loadJobtitles(page = 1) {
			currentPage = page;  // 當前頁數
			const pSize = 15;
			var conditionObj = {};

			var query = document.getElementById('searchJobtitlesInput').value.trim().toLowerCase();  // 取得搜尋關鍵字

			// 組合查詢參數
			const params = new URLSearchParams({
				page: page,
				size: pSize,
				condition: query,
			});

			fetch(`${API_BASE}/employees/get_jobtitles.php?${params.toString()}`)
				.then(response => response.json())
				.then(data => {
					if (data.result) {
						const jobtitlesTableBody = document.querySelector('#jobtitles-table tbody');
						jobtitlesTableBody.innerHTML = ''; // 清空現有的表格內容
						data.data.records.forEach((record, index) => {
							const row = document.createElement('tr');

							row.innerHTML = `
			<td><input type="checkbox" name="select_jobtitle" id="jobtitle-${record.position_id}" value="${record.position_id}">${record.position_id}</td>
			<td class="jobtitle-title">${record.title}</td>
			<td class="jobtitle-title">${record.en_title}</td>
			<td class="jobtitle-category">${record.category}</td>
			<td class="jobtitle-salary_range">${record.salary_range}</td>
		    `;

							// 點擊 "名稱" 欄位彈出修改視窗
							row.querySelector('.jobtitle-title').addEventListener('click', function(event) {
								event.stopPropagation(); // 阻止事件冒泡
								showEditJobtitleModal(record.position_id); // 點擊名稱彈出修改視窗
							});

							jobtitlesTableBody.appendChild(row);
						});
						// 生成分頁按鈕
						totalPages = Math.ceil(data.data.ttl / pSize); // 計算總頁數
						generatePaginationControls(); // 生成分頁按鈕
					} else {
						alert('無法獲取職稱列表');
					}
				})
				.catch(error => console.error('Error loading Jobtitles list:', error));
		}

		/* 搜尋 */
		function searchJobtitles() {
			loadJobtitles(); // 重新載入職稱清單
		}

		/* 新增 */
		function showAddJobtitleModal() {
			document.getElementById('createJobtitleForm').reset(); // 清空表單

			const modal = document.getElementById('addJobtitleModal');
			modal.showModal(); // 顯示模態視窗

			console.log("Modal is open");

			// 先解除以前綁定的 submit 事件，再綁定新的事件
			const createForm = document.getElementById('createJobtitleForm');
			createForm.removeEventListener('submit', submitJobtitleFormHandler);  // 如果以前有綁定過
			createForm.addEventListener('submit', submitJobtitleFormHandler);
		}

		/* 提交新增的處理函數 */
		function submitJobtitleFormHandler(event) {
			event.preventDefault(); // 防止表單的預設提交行為

			// 收集表單數據
			const formData = new FormData(event.target);
			const newData = {
				title: formData.get('title'),
				en_title: formData.get('entitle'),
				category: formData.get('category'),
				salary_range: formData.get('salary_range'),
			};

			console.log('NewData: ' + JSON.stringify(newData));

			// 發送 AJAX 請求
			fetch(`${API_BASE}/employees/create_jobtitle.php`, {
				method: 'POST',
				headers: { 'Content-Type': 'application/json' },
				body: JSON.stringify(newData),
			})
				.then(response => response.json())
				.then(data => {
					if (data.result) {
						alert('職稱新增成功');
						closeAddJobtitleModal(); // 關閉模態視窗
						loadJobtitles(); // 重新載入職稱清單
						// 重置表單內容
						this.reset();
					} else {
						alert('新增失敗: ' + data.msg);
					}
				})
				.catch(error => {
					console.error('Error:', error);
				});
		}

		/* 關閉新增職稱的 Modal */
		function closeAddJobtitleModal() {
			addJobtitleModal.close();
		}

		/* 編輯/修改 區塊 */
		async function showEditJobtitleModal(jobtitleId) {
			console.log('sid: '+jobtitleId);

			const modal = document.getElementById('editJobtitleModal');
			modal.setAttribute('data-id', jobtitleId); // 記錄職稱 ID
			modal.showModal();

			try {
				// 從伺服器獲取職稱數據
				const response = await fetch(`${API_BASE}/employees/get_jobtitle.php?id=${jobtitleId}`);
				const data = await response.json();

				if (data.result) {
					const row = data.data;

					// 填充職稱數據到表單
					document.getElementById('editTitle').value = row.title;
					document.getElementById('editEnTitle').value = row.en_title;
					document.getElementById('editCategory').value = row.category;
					document.getElementById('editSalary_range').value = row.salary_range;
				} else {
					alert('無法加載職稱數據：' + data.msg);
				}
			} catch (error) {
				console.error('Error:', error);
			}
		}

		/* 關閉編輯模態框 */
		function closeEditJobtitleModal() {
			document.getElementById('editJobtitleModal').close();
		}

		/* 定義 submitEditJobtitleFormHandler 函數 */
		document.getElementById('editJobtitleForm').addEventListener('submit', function (event) {
			event.preventDefault(); // 防止表單的預設提交行為

			// 取得表單中基本資料
			const modal = document.getElementById('editJobtitleModal');
			const positionId = modal.getAttribute('data-id');
			const title = document.getElementById('editTitle').value;
			const entitle = document.getElementById('editEnTitle').value;
			const category = document.getElementById('editCategory').value;
			const salary_range = document.getElementById('editSalary_range').value;

			if (!positionId) {
				alert('職稱 ID 丟失，無法更新');
				return;
			}

			// 組合 newData 物件屬性維持為陣列型態
			const newData = {
				position_id: positionId,
				title: title,
				en_title: entitle,
				category: category,
				salary_range: salary_range
			};

			console.log('提交數據:', newData);

			// 發送 AJAX 請求
			fetch(`${API_BASE}/employees/update_jobtitle.php`, {
				method: 'POST',
				headers: { 'Content-Type': 'application/json' },
				body: JSON.stringify(newData)
			})
				.then(async response => {
					if (!response.ok) {
						// 先拿到原始文字，幫你快速定位 PHP 報錯
						const text = await response.text();
						console.error('伺服器回傳錯誤：', response.status, text);
						throw new Error(`Server error ${response.status}`);
					}
					// 正常才 parse JSON
					return response.json();
				})
				.then(data => {
					if (data.result) {
						alert('職稱更新成功');
						closeEditJobtitleModal();
						loadJobtitles();
					} else {
						alert('更新失敗：' + data.msg);
					}
				})
				.catch(err => {
					console.error('Fetch/解析失敗：', err);
				});
		});


		/* =========================================================
		 * Job Titles Module (Merged)
		 * 職稱設定（buttom-widget）
		 * ========================================================= */

		document.addEventListener('DOMContentLoaded', () => {
			loadJobTitles();

			document.getElementById('chkJobTitleAll')
				?.addEventListener('change', e => {
					document.querySelectorAll('.chk-jobtitle')
						.forEach(chk => chk.checked = e.target.checked);
				});
		});

		function loadJobTitles() {
			fetch(`${API_BASE}/employees/get_jobtitles.php`)
				.then(res => res.json())
				.then(json => {
					if (!json.result) {
						alert(json.msg);
						return;
					}

					const tbody = document.querySelector('#jobtitles-table tbody');
					if (!tbody) return;

					tbody.innerHTML = '';

					json.data.records.forEach(row => {
						const tr = document.createElement('tr');
						tr.dataset.id = row.position_id;

						tr.innerHTML = `
		    <td><input type="checkbox" class="chk-jobtitle"></td>
		    <td>${row.title}</td>
		    <td>${row.en_title}</td>
		    <td>${row.category}</td>
		`;

						tr.addEventListener('click', e => {
							if (e.target.tagName === 'INPUT') return;
							loadJobTitleDetail(row.position_id);
						});

						tbody.appendChild(tr);
					});
				});
		}

		async function loadJobTitleDetail(positionId) {
			const res = await fetch(`${API_BASE}/employees/get_jobtitle.php?position_id=${positionId}`);
			const json = await res.json();

			if (!json.result) {
				alert(json.msg);
				return;
			}

			const d = json.data;
			const dlg = document.getElementById('jobtitleDialog');

			dlg.dataset.id = d.position_id;
			document.getElementById('jobTitleInput').value    = d.title;
			document.getElementById('jobEnTitleInput').value  = d.en_title;
			document.getElementById('jobPhoneticInput').value = d.phonetic;
			document.getElementById('jobPinyinInput').value   = d.pinyin;
			document.getElementById('jobCategorySelect').value = d.category;
			document.getElementById('jobSalaryInput').value   = d.salary_range ?? '';

			dlg.showModal();
		}

		async function updateJobTitle() {
			const dlg = document.getElementById('jobtitleDialog');

			const payload = {
				position_id: dlg.dataset.id,
				title: document.getElementById('jobTitleInput').value.trim(),
				en_title: document.getElementById('jobEnTitleInput').value.trim(),
				phonetic: document.getElementById('jobPhoneticInput').value.trim(),
				pinyin: document.getElementById('jobPinyinInput').value.trim(),
				category: document.getElementById('jobCategorySelect').value,
				salary_range: document.getElementById('jobSalaryInput').value.trim()
			};

			const res = await fetch(`${API_BASE}/employees/update_jobtitle.php`, {
				method: 'POST',
				headers: { 'Content-Type': 'application/json' },
				body: JSON.stringify(payload)
			});

			const json = await res.json();
			alert(json.msg);

			if (json.result) {
				dlg.close();
				loadJobTitles();
			}
		}

		function getSelectedJobTitleIds() {
			return Array.from(
				document.querySelectorAll('.chk-jobtitle:checked')
			).map(chk => chk.closest('tr').dataset.id);
		}

		async function deleteSelectedJobTitles() {
			const ids = getSelectedJobTitleIds();

			if (ids.length === 0) {
				alert('請先選擇要刪除的職稱');
				return;
			}

			if (!confirm(`確定要刪除 ${ids.length} 筆職稱？`)) return;

			const res = await fetch(`${API_BASE}/employees/del_jobtitles.php`, {
				method: 'POST',
				headers: { 'Content-Type': 'application/json' },
				body: JSON.stringify({ position_ids: ids })
			});

			const json = await res.json();

			if (!json.result) {
				alert(json.msg);
				return;
			}

			let msg = `成功刪除 ${json.data.deleted_count} 筆職稱。`;

			if (json.data.blocked_ids.length > 0) {
				msg += `\n以下職稱因已有員工使用，無法刪除：\n`;
				msg += json.data.blocked_ids.join(', ');
			}

			alert(msg);
			loadJobTitles();
		}

		// ===== ⭐ 對外匯出（非常重要） =====
		window.loadEmployees = loadEmployees;
		window.searchEmployees = searchEmployees;
		window.showAddEmployeeModal = showAddEmployeeModal;
		window.showEditEmployeeModal = showEditEmployeeModal;
		window.closeAddEmployeeModal = closeAddEmployeeModal;
		window.closeEditEmployeeModal = closeEditEmployeeModal;
		window.deleteSelectedEmployees = deleteSelectedEmployees;

		// 部門
		window.loadDepartments = loadDepartments;
		window.searchDepartments = searchDepartments;
		window.toggleDropdown = toggleDropdown;
		window.filterOptions = filterOptions;
		window.showAddDepartmentModal = showAddDepartmentModal;
		window.showEditDepartmentModal = showEditDepartmentModal;
		window.closeAddDepartmentModal = closeAddDepartmentModal;
		window.closeEditDepartmentModal = closeEditDepartmentModal;
		window.deleteSelectedDepartments = deleteSelectedDepartments;

		// 職稱
		window.loadJobtitles = loadJobtitles;
		window.searchJobtitles = searchJobtitles;
		window.showAddJobtitleModal = showAddJobtitleModal;
		window.showEditJobtitleModal = showEditJobtitleModal;
		window.closeAddJobtitleModal = closeAddJobtitleModal;
		window.closeEditJobtitleModal = closeEditJobtitleModal;
		window.deleteSelectedJobTitles = deleteSelectedJobTitles;
	})();
}

