<?php
include '../config.php';
if (session_status() == PHP_SESSION_NONE) session_start();

$ret = false;
$data = NULL;
$msg = '取得掛號列表失敗';
$args = [];

header('Content-Type: application/json');

try {
    mysqli_report(MYSQLI_REPORT_ERROR | MYSQLI_REPORT_STRICT);

    $mode = isset($_GET['mode']) ? $_GET['mode'] : 'list';

    // 當 $default_pagesize 小於 15 時，強制使用 15 作為預設分頁大小
    $default_size = ($default_pagesize < 15) ? 15 : $default_pagesize;

    if ($mode === 'doctors') {
        $date = isset($_GET['date']) ? $_GET['date'] : '';
        $args = ['mode'=>$mode, 'date'=>$date];
        DEBUG(__FILE__, __LINE__, "ARGS: " . json_encode($args));

        if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $date)) throw new Exception("缺少或無效日期($date)");

        // 依排班表：該日非休假者視為可掛號醫師
        $sql = "SELECT DISTINCT s.employee_id, e.cname, e.ename
                FROM SCHEDULES s
                JOIN EMPLOYEES e ON s.employee_id = e.employee_id
                WHERE s.shift_date = ? AND s.shift_type <> '休假'
                ORDER BY e.cname ASC";
        DEBUG(__FILE__, __LINE__, "SQL: $sql");

        $stmt = $conn->prepare($sql);
        $stmt->bind_param('s', $date);
        $stmt->execute();
        $rs = $stmt->get_result();
        $records = [];
        while ($row = $rs->fetch_assoc()) $records[] = $row;
        $stmt->close();

        $data = ['ttl'=>count($records), 'records'=>$records];
        $ret = true;
        $msg = '';
    } else if ($mode === 'next_no') {
        $doctor_id = isset($_GET['doctor_id']) ? $_GET['doctor_id'] : '';
        $appt_date = isset($_GET['appt_date']) ? $_GET['appt_date'] : '';
        $args = ['mode'=>$mode, 'doctor_id'=>$doctor_id, 'appt_date'=>$appt_date];
        DEBUG(__FILE__, __LINE__, "ARGS: " . json_encode($args));

        if ($doctor_id === '') throw new Exception("缺少醫師編號");
        if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $appt_date)) throw new Exception("缺少或無效日期($appt_date)");

        $sql = "SELECT COALESCE(MAX(appt_no),0) AS max_no
                FROM APPOINTMENTS
                WHERE doctor_id=? AND appt_date=? AND status<>'取消'";
        DEBUG(__FILE__, __LINE__, "SQL: $sql");

        $stmt = $conn->prepare($sql);
        $stmt->bind_param('ss', $doctor_id, $appt_date);
        $stmt->execute();
        $rs = $stmt->get_result();
        $row = $rs->fetch_assoc();
        $stmt->close();

        $next = intval($row['max_no']) + 1;
        $data = ['appt_no'=>$next];
        $ret = true;
        $msg = '';
    } else { // list
        $page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
        $size = isset($_GET['size']) ? max(1, (int)$_GET['size']) : $default_size;
        $condition = isset($_GET['condition']) ? trim($_GET['condition']) : '';
        $appt_date = isset($_GET['appt_date']) ? trim($_GET['appt_date']) : '';
        $doctor_id = isset($_GET['doctor_id']) ? trim($_GET['doctor_id']) : '';

        $args = ['mode'=>$mode,'page'=>$page,'size'=>$size,'condition'=>$condition,'appt_date'=>$appt_date,'doctor_id'=>$doctor_id];
        DEBUG(__FILE__, __LINE__, "ARGS: " . json_encode($args));

        $where = [];
        $types = '';
        $binds = [];

        if ($appt_date !== '') {
            if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $appt_date)) throw new Exception("無效日期($appt_date)");
            $where[] = "a.appt_date = ?";
            $types .= 's';
            $binds[] = $appt_date;
        }

        if ($doctor_id !== '') {
            $where[] = "a.doctor_id = ?";
            $types .= 's';
            $binds[] = $doctor_id;
        }

        if ($condition !== '') {
            // 支援：病患編號 / 病患姓名 / 身分證 / 電話 / 掛號編號
            $where[] = "(a.patient_id LIKE ? OR p.cname LIKE ? OR p.national_id LIKE ? OR p.phone LIKE ? OR CAST(a.id AS CHAR) LIKE ?)";
            $types .= 'sssss';
            $like = '%' . $condition . '%';
            $binds[] = $like; $binds[] = $like; $binds[] = $like; $binds[] = $like; $binds[] = $like;
        }

        $whereSql = (count($where) ? ('WHERE ' . implode(' AND ', $where)) : '');

        $tbl = "APPOINTMENTS a
                JOIN PATIENTS p ON a.patient_id = p.patient_id
                JOIN EMPLOYEES e ON a.doctor_id = e.employee_id
                LEFT JOIN ROOMS r ON a.room_id = r.room_id";

        // total
        $totalSql = "SELECT COUNT(*) AS total FROM $tbl $whereSql";
        DEBUG(__FILE__, __LINE__, "QUERY: $totalSql");

        $totalStmt = $conn->prepare($totalSql);
        if ($types !== '') $totalStmt->bind_param($types, ...$binds);
        $totalStmt->execute();
        $totalRes = $totalStmt->get_result();
        $totalRow = $totalRes->fetch_assoc();
        $total = intval($totalRow['total']);
        $totalStmt->close();

        $maxPage = ($size > 0) ? max(1, (int)ceil($total / $size)) : 1;
        if ($page > $maxPage) $page = $maxPage;
        $start = ($page - 1) * $size;

        // records
        $sql = "SELECT a.*,
                       p.cname AS patient_cname, p.ename AS patient_ename, p.phone AS patient_phone,
                       e.cname AS doctor_cname, e.ename AS doctor_ename,
                       r.name AS room_name
                FROM $tbl
                $whereSql
                ORDER BY a.id DESC
                LIMIT ?, ?";
        DEBUG(__FILE__, __LINE__, "SQL: $sql");
        DEBUG(__FILE__, __LINE__, "START: $start, SIZE: $size");

        $stmt = $conn->prepare($sql);

        $types2 = $types . 'ii';
        $binds2 = $binds;
        $binds2[] = $start;
        $binds2[] = $size;

        if ($types2 !== '') $stmt->bind_param($types2, ...$binds2);
        $stmt->execute();
        $rs = $stmt->get_result();
        $records = [];
        while ($row = $rs->fetch_assoc()) $records[] = $row;
        $stmt->close();

        // summary：同條件下（不含分頁）統計各狀態
        $sumSql = "SELECT
                    COUNT(*) AS total,
                    SUM(CASE WHEN a.status='預約' THEN 1 ELSE 0 END) AS booked,
                    SUM(CASE WHEN a.status='看診中' THEN 1 ELSE 0 END) AS in_progress,
                    SUM(CASE WHEN a.status='完成' THEN 1 ELSE 0 END) AS done,
                    SUM(CASE WHEN a.status='取消' THEN 1 ELSE 0 END) AS canceled
                   FROM $tbl $whereSql";
        DEBUG(__FILE__, __LINE__, "SQL_SUM: $sumSql");

        $sumStmt = $conn->prepare($sumSql);
        if ($types !== '') $sumStmt->bind_param($types, ...$binds);
        $sumStmt->execute();
        $sumRes = $sumStmt->get_result();
        $summary = $sumRes->fetch_assoc();
        $sumStmt->close();

        $data = [
            'ttl'=>$total,
            'page'=>$page,
            'records'=>$records,
            'summary'=>$summary
        ];
        $ret = true;
        $msg = '';
    }
} catch (Exception $e) {
    $msg = $e->getMessage();
}

DEBUG(__FILE__, __LINE__, "MSG: $msg");
logAction($conn, __FILE__, '', $args, json_encode(['result'=>$ret, 'msg'=>$msg]));
DEBUG(__FILE__, __LINE__, "Ret: ".json_encode($data));
echo json_encode(['result'=>$ret, 'data'=>$data, 'msg'=>$msg]);
?>

