/* =====================================================
   registered.js - No inline handlers version
   - All events are bound via addEventListener
   - Safe under dynamic reloading
   ===================================================== */

(function () {
  "use strict";

  // 防止同一頁面重複載入後重複綁定
  if (window.__registeredModuleLoaded_v2) {
    // 若 loadRegistered 不見了，補回（理論上不會）
    if (typeof window.loadRegistered !== "function") window.loadRegistered = function () {};
    return;
  }
  window.__registeredModuleLoaded_v2 = true;

  const API_BASE = window.API_BASE || "/api";
  const IMG_BASE = window.IMG_BASE || "/img";

  let currentPage = 1;
  let totalPages = 1;
  let patientSuggestTimer = null;

  // dashboard 動態載入後會呼叫
  window.loadRegistered = function () {
    init();
  };

  function init() {
    setDefaultDate();
    bindOnce();
    refreshDoctorsThenList();
  }

  function setDefaultDate() {
    const el = document.getElementById("filterDate");
    if (!el) return;

    if (!el.value) {
      const d = new Date();
      const today = `${d.getFullYear()}-${String(d.getMonth() + 1).padStart(2, "0")}-${String(
        d.getDate()
      ).padStart(2, "0")}`;
      el.value = today;
    }
  }

  function bindOnce() {
    // 只綁一次（靠 dataset.bound）
    const root = document.getElementById("main-list");
    if (root && root.dataset.bound === "1") return;
    if (root) root.dataset.bound = "1";

    // toolbar
    document.getElementById("btnAddRegistered")?.addEventListener("click", openAddModal);
    document.getElementById("btnCancelRegistered")?.addEventListener("click", cancelSelectedRegistered);

    document.getElementById("filterDate")?.addEventListener("change", () => refreshDoctorsThenList(1));
    document.getElementById("filterDoctor")?.addEventListener("change", () => loadList(1));

    // 搜尋：輸入就更新（可視需要加 debounce）
    document.getElementById("searchRegistered")?.addEventListener("input", () => loadList(1));

    // dialog 按鈕
    document.getElementById("btnCloseRegisteredModal")?.addEventListener("click", closeModal);
    document.getElementById("btnCancelModal")?.addEventListener("click", closeModal);
    document.getElementById("btnResetModal")?.addEventListener("click", () => resetForm(true));
    document.getElementById("btnNextNumber")?.addEventListener("click", getNextNumber);
    document.getElementById("btnGotoPatients")?.addEventListener("click", gotoPatients);

    // form submit
    const form = document.getElementById("registeredForm");
    if (form && !form.dataset.bound) {
      form.addEventListener("submit", submitForm);
      form.dataset.bound = "1";
    }

    // 病患建議
    const kw = document.getElementById("patient_kw");
    if (kw && !kw.dataset.bound) {
      kw.addEventListener("input", onSuggestInput);
      kw.dataset.bound = "1";
    }

    // 點空白關建議清單
    if (!document.body.dataset.regSuggestBound) {
      document.addEventListener("click", (e) => {
        const box = document.getElementById("patient_suggest");
        if (!box) return;
        if (!e.target.closest("#patient_suggest") && !e.target.closest("#patient_kw")) {
          box.style.display = "none";
        }
      });
      document.body.dataset.regSuggestBound = "1";
    }
  }

  function refreshDoctorsThenList(page = 1) {
    const date = document.getElementById("filterDate")?.value || "";
    return loadDoctorsByDate(date).then(() => loadList(page));
  }

  function loadDoctorsByDate(dateStr) {
    const params = new URLSearchParams({ mode: "doctors", date: dateStr || "" });
    return fetch(`${API_BASE}/registered/get_registered.php?${params.toString()}`)
      .then((r) => r.json())
      .then((res) => {
        const sel = document.getElementById("filterDoctor");
        if (!sel) return;
        const keep = sel.value || "";

        sel.innerHTML = `<option value="">全部醫師</option>`;
        if (res.result && Array.isArray(res.data?.records)) {
          res.data.records.forEach((d) => {
            const opt = document.createElement("option");
            opt.value = d.employee_id;
            opt.textContent = `${d.cname} (${d.employee_id})`;
            sel.appendChild(opt);
          });
        }
        sel.value = keep;
      })
      .catch((err) => console.error("loadDoctorsByDate error:", err));
  }

  function loadDoctorsToModal(dateStr) {
    const params = new URLSearchParams({ mode: "doctors", date: dateStr || "" });
    return fetch(`${API_BASE}/registered/get_registered.php?${params.toString()}`)
      .then((r) => r.json())
      .then((res) => {
        const sel = document.getElementById("doctor_id");
        if (!sel) return;

        sel.innerHTML = "";
        if (res.result && Array.isArray(res.data?.records) && res.data.records.length) {
          res.data.records.forEach((d) => {
            const opt = document.createElement("option");
            opt.value = d.employee_id;
            opt.textContent = `${d.cname} (${d.employee_id})`;
            sel.appendChild(opt);
          });
        } else {
          const opt = document.createElement("option");
          opt.value = "";
          opt.textContent = "（此日期無排班醫師）";
          sel.appendChild(opt);
        }
      })
      .catch((err) => console.error("loadDoctorsToModal error:", err));
  }

  function loadList(page = 1) {
    currentPage = page;

    const size = 15;
    const condition = (document.getElementById("searchRegistered")?.value || "").trim();
    const appt_date = document.getElementById("filterDate")?.value || "";
    const doctor_id = document.getElementById("filterDoctor")?.value || "";

    const params = new URLSearchParams({
      mode: "list",
      page: String(page),
      size: String(size),
      condition,
      appt_date,
      doctor_id,
    });

    fetch(`${API_BASE}/registered/get_registered.php?${params.toString()}`)
      .then((r) => r.json())
      .then((res) => {
        if (!res.result) {
          alert(res.msg || "無法取得掛號列表");
          return;
        }

        const tbody = document.querySelector("#registered-table tbody");
        if (!tbody) return;
        tbody.innerHTML = "";

        const records = res.data?.records || [];
        records.forEach((rec) => {
          const tr = document.createElement("tr");

          tr.innerHTML = `
            <td>
              <input type="checkbox" name="select_registered" value="${esc(rec.id)}">
              <span class="reg-id" style="margin-left:6px;">${esc(rec.id)}</span>
            </td>
            <td>${esc(rec.patient_id || "")}</td>
            <td class="clickable patientName">${esc(rec.patient_cname || "")}</td>
            <td>${esc(rec.doctor_cname || rec.doctor_id || "")}</td>
            <td>${esc(rec.appt_date || "")}</td>
            <td style="text-align:center;">${esc(rec.appt_no || "")}</td>
            <td>${esc(rec.type || "")}</td>
            <td>${esc(rec.status || "")}</td>
            <td>${esc(rec.room_name || rec.room_id || "")}</td>
          `;

          // 點姓名：編輯
          tr.querySelector(".patientName")?.addEventListener("click", (e) => {
            e.stopPropagation();
            openEditModal(rec);
          });

          // 點列：右側明細（點 checkbox 不觸發）
          tr.addEventListener("click", (e) => {
            if (e.target && e.target.tagName === "INPUT") return;
            document.querySelectorAll("#registered-table tbody tr").forEach((x) => (x.style.backgroundColor = ""));
            tr.style.backgroundColor = "#CCFFFF";
            renderDetail(rec);
          });

          tbody.appendChild(tr);
        });

        const ttl = res.data?.ttl || 0;
        totalPages = Math.max(1, Math.ceil(ttl / size));
        renderPagination();

        renderSummary(res.data?.summary || null, appt_date, doctor_id);
      })
      .catch((err) => {
        console.error(err);
        alert("載入掛號列表失敗");
      });
  }

  function renderPagination() {
    const box = document.getElementById("pagination-controls");
    if (!box) return;
    box.innerHTML = "";

    if (currentPage > totalPages) currentPage = totalPages;

    box.appendChild(pagBtn("firstpage_icon.png", 1));
    box.appendChild(pagBtn("previous_icon.png", Math.max(currentPage - 1, 1)));

    const start = Math.max(currentPage - 2, 1);
    const end = Math.min(currentPage + 2, totalPages);

    for (let i = start; i <= end; i++) {
      const a = document.createElement("a");
      a.href = "javascript:void(0)";
      a.textContent = String(i);
      a.style.margin = "0 5px";
      if (i === currentPage) a.classList.add("current-page");
      a.addEventListener("click", () => loadList(i));
      box.appendChild(a);
    }

    box.appendChild(pagBtn("nextpage_icon.png", Math.min(currentPage + 1, totalPages)));
    box.appendChild(pagBtn("endpage_icon.png", totalPages));

    const input = document.createElement("input");
    input.type = "number";
    input.min = "1";
    input.max = String(totalPages);
    input.value = String(currentPage);
    input.addEventListener("change", () => {
      const p = parseInt(input.value, 10);
      if (p >= 1 && p <= totalPages) loadList(p);
    });
    box.appendChild(input);
  }

  function pagBtn(icon, page) {
    const a = document.createElement("a");
    a.href = "javascript:void(0)";
    a.addEventListener("click", () => loadList(page));
    a.innerHTML = `<img src="${IMG_BASE}/${icon}" alt="icon" class="pagination-icon">`;
    return a;
  }

  function renderSummary(summary, appt_date, doctor_id) {
    setText("sumDate", appt_date || "-");
    const docSel = document.getElementById("filterDoctor");
    const docName = docSel?.selectedOptions?.[0]?.textContent || "-";
    setText("sumDoctor", doctor_id ? docName : "-");

    const s = summary || { total: 0, booked: 0, in_progress: 0, done: 0, canceled: 0 };
    setText("sumTotal", s.total ?? 0);
    setText("sumBooked", s.booked ?? 0);
    setText("sumInProgress", s.in_progress ?? 0);
    setText("sumDone", s.done ?? 0);
    setText("sumCanceled", s.canceled ?? 0);
  }

  function renderDetail(rec) {
    const box = document.getElementById("selectedDetail");
    if (!box) return;
    box.innerHTML = `
      <div><b>掛號編號：</b>${esc(rec.id)}</div>
      <div><b>病患：</b>${esc(rec.patient_id || "")} ${esc(rec.patient_cname || "")}</div>
      <div><b>醫師：</b>${esc(rec.doctor_cname || rec.doctor_id || "")}</div>
      <div><b>日期 / 號碼：</b>${esc(rec.appt_date || "")} / ${esc(rec.appt_no || "")}</div>
      <div><b>狀態：</b>${esc(rec.status || "")}</div>
      <div><b>類型：</b>${esc(rec.type || "")}</div>
      <div><b>診間：</b>${esc(rec.room_name || rec.room_id || "")}</div>
    `;
  }

  function setText(id, v) {
    const el = document.getElementById(id);
    if (el) el.textContent = String(v);
  }

  // -------- Modal --------
  function openAddModal() {
    resetForm(false);

    const d = document.getElementById("filterDate")?.value || "";
    if (d) document.getElementById("appt_date").value = d;

    loadDoctorsToModal(d).then(() => {
      document.getElementById("registeredModalTitle").textContent = "現場掛號";
      document.getElementById("type").value = "現場掛號";
      document.getElementById("status").value = "預約";
      document.getElementById("registeredModal").showModal();
    });
  }

  function openEditModal(rec) {
    resetForm(false);

    document.getElementById("reg_id").value = rec.id || "";
    document.getElementById("patient_id").value = rec.patient_id || "";
    document.getElementById("appt_date").value = rec.appt_date || "";
    document.getElementById("appt_no").value = rec.appt_no || "";
    document.getElementById("type").value = rec.type || "現場掛號";
    document.getElementById("status").value = rec.status || "預約";
    document.getElementById("room_id").value = rec.room_id || "";

    loadDoctorsToModal(rec.appt_date || "").then(() => {
      document.getElementById("doctor_id").value = rec.doctor_id || "";
      document.getElementById("registeredModalTitle").textContent = "修改掛號";
      document.getElementById("registeredModal").showModal();
    });
  }

  function closeModal() {
    document.getElementById("registeredModal")?.close();
  }

  function resetForm(resetNative) {
    const form = document.getElementById("registeredForm");
    if (resetNative && form) form.reset();

    const id = document.getElementById("reg_id");
    if (id) id.value = "";

    const suggest = document.getElementById("patient_suggest");
    if (suggest) {
      suggest.style.display = "none";
      suggest.innerHTML = "";
    }
  }

  function getNextNumber() {
    const doctor_id = document.getElementById("doctor_id")?.value || "";
    const appt_date = document.getElementById("appt_date")?.value || "";
    if (!doctor_id || !appt_date) {
      alert("請先選擇掛號醫師與掛號日期");
      return;
    }

    const params = new URLSearchParams({ mode: "next_no", doctor_id, appt_date });
    fetch(`${API_BASE}/registered/get_registered.php?${params.toString()}`)
      .then((r) => r.json())
      .then((res) => {
        if (!res.result) return alert(res.msg || "取得號碼失敗");
        document.getElementById("appt_no").value = res.data?.appt_no || "";
      })
      .catch((err) => {
        console.error(err);
        alert("取得號碼失敗");
      });
  }

  function submitForm(e) {
    e.preventDefault();

    const id = document.getElementById("reg_id").value.trim();
    const payload = {
      id: id || null,
      patient_id: document.getElementById("patient_id").value.trim(),
      doctor_id: document.getElementById("doctor_id").value,
      appt_date: document.getElementById("appt_date").value,
      appt_no: parseInt(document.getElementById("appt_no").value, 10),
      type: document.getElementById("type").value,
      status: document.getElementById("status").value,
      room_id: document.getElementById("room_id").value.trim() || null,
    };

    if (!payload.patient_id || !payload.doctor_id || !payload.appt_date || !payload.appt_no) {
      alert("請確認病患編號 / 醫師 / 日期 / 號碼皆已填寫");
      return;
    }

    const url = id ? `${API_BASE}/registered/update_registered.php` : `${API_BASE}/registered/create_registered.php`;
    fetch(url, {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify(payload),
    })
      .then((r) => r.json())
      .then((res) => {
        alert(res.msg || (res.result ? "儲存成功" : "儲存失敗"));
        if (res.result) {
          closeModal();
          loadList(1);
        }
      })
      .catch((err) => {
        console.error(err);
        alert("儲存失敗");
      });
  }

  function cancelSelectedRegistered() {
    const ids = Array.from(document.querySelectorAll('input[name="select_registered"]:checked')).map((x) => x.value);
    if (!ids.length) return alert("請先勾選要取消的掛號");
    if (!confirm(`確定要取消 ${ids.length} 筆掛號？`)) return;

    fetch(`${API_BASE}/registered/del_registered.php`, {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ ids }),
    })
      .then((r) => r.json())
      .then((res) => {
        alert(res.msg || (res.result ? "取消完成" : "取消失敗"));
        if (res.result) loadList(1);
      })
      .catch((err) => {
        console.error(err);
        alert("取消失敗");
      });
  }

  // -------- Patient suggest --------
  function onSuggestInput() {
    clearTimeout(patientSuggestTimer);
    patientSuggestTimer = setTimeout(() => {
      const kw = document.getElementById("patient_kw")?.value.trim() || "";
      const box = document.getElementById("patient_suggest");
      if (!box) return;

      if (!kw) {
        box.style.display = "none";
        box.innerHTML = "";
        return;
      }
      fetchSuggest(kw);
    }, 180);
  }

  function fetchSuggest(keyword) {
    const params = new URLSearchParams({ page: "1", size: "10", condition: keyword });
    fetch(`${API_BASE}/patients/get_patients.php?${params.toString()}`)
      .then((r) => r.json())
      .then((res) => {
        const box = document.getElementById("patient_suggest");
        if (!box) return;

        if (!res.result || !Array.isArray(res.data?.records) || !res.data.records.length) {
          box.innerHTML = `<div class="suggest-item">查無符合病患</div>`;
          box.style.display = "block";
          return;
        }

        box.innerHTML = res.data.records
          .map((p) => {
            const line = `${p.patient_id}｜${p.cname}｜${p.national_id || ""}｜${p.phone || ""}`;
            return `<div class="suggest-item" data-id="${esc(p.patient_id)}">${esc(line)}</div>`;
          })
          .join("");

        box.querySelectorAll(".suggest-item").forEach((item) => {
          item.addEventListener("click", () => {
            const pid = item.getAttribute("data-id");
            if (pid) document.getElementById("patient_id").value = pid;
            box.style.display = "none";
          });
        });

        box.style.display = "block";
      })
      .catch((err) => console.error("fetchSuggest error:", err));
  }

  function gotoPatients() {
    if (typeof window.loadContent === "function") {
      window.loadContent("book", "patients");
      closeModal();
      return;
    }
    alert("請回到「掛號管理 → 患者資料」新增病患後，再回來現場掛號。");
  }

  function esc(s) {
    return String(s ?? "")
      .replaceAll("&", "&amp;")
      .replaceAll("<", "&lt;")
      .replaceAll(">", "&gt;")
      .replaceAll('"', "&quot;")
      .replaceAll("'", "&#39;");
  }
})();

