/* for dashboard.html */

let currentPage = 1;  // 當前頁數
const pageSize = 10;  // 每頁最多顯示的資料數量
let totalPages = 1;   // 總頁數，根據總數據量計算出來
let currentCategoryFilter = "";
let currentSubcategoryFilter = "";
let currentFamilyFilter = "";
let currentPropertyFilter = "";
let currentTasteFilter = "";
let overdueTime = 0;

// 定義每個 img 的亮/暗模式檔案名稱

// 用於記錄當前被點擊的 img
let activeImgId = null;
let activeLblId = null;

// 放在同一支 dashboard.js 前面（或同檔內較上方）
const __loadedScripts = new Set();
function loadScriptOnce(src) {
	return new Promise((resolve, reject) => {
		if (__loadedScripts.has(src)) return resolve();
		const s = document.createElement('script');
		s.src = src;
		s.async = false; // 保順序
		s.defer = false;
		s.onload = () => { __loadedScripts.add(src); resolve(); };
		s.onerror = (e) => { console.error('loadScript error:', src, e); reject(e); };
		document.body.appendChild(s);
	});
}
/**********************/

// 路徑
const BASE_DIR = '/';
const API_BASE = BASE_DIR + 'api';
const IMG_BASE = BASE_DIR + 'img';
const HTML_BASE = BASE_DIR + 'dashboard';
const JS_BASE = BASE_DIR + 'js';

const imgMap = [
	{ id: 'createBtn', light: 'plus_dark.svg',      dark: 'plus_light.svg',     color: 'plus_color.svg' },     // 新增鈕
	{ id: 'deleteBtn', light: 'trashcan_dark.svg',  dark: 'trashcan_light.svg', color: 'trashcan_color.svg' }, // 刪除鈕
	{ id: 'resetPass', light: 'reset_dark.svg',     dark: 'reset_light.svg',    color: 'reset_color.svg' },    // 重設鈕
	{ id: 'settings',  light: 'gear_light.svg',     dark: 'gear_dark.svg',      color: 'gear_color.svg' },     // 設定鈕
	{ id: 'pwdBtn',    light: 'pwd_light.png',      dark: 'pwd_dark.png',       color: 'pwd_dark.png' },       // 變更密碼鈕
	{ id: 'logout',    light: 'logout_light.svg',   dark: 'logout_dark.svg',    color: 'logout_color.svg' },   // 登出鈕
	{ id: 'homeIcon',  light: 'logo_light.svg',     dark: 'logo_dark.svg',      color: 'logo_color.svg' },     // logo 鈕
	{ id: 'homeBtn',   light: 'home_light.svg',     dark: 'home_dark.svg',      color: 'home_color.svg' },     // 回首頁鈕
	{ id: 'prtBtn',    light: 'printer_light.svg',  dark: 'printer_dark.svg',   color: 'printer_color.svg' },  // 列印鈕
	{ id: 'bookBtn',   light: 'book_light.svg',     dark: 'book_dark.svg',      color: 'book_color.svg' },     // 掛號鈕
	{ id: 'opdBtn',    light: 'opd_light.svg',      dark: 'opd_dark.svg',       color: 'opd_color.svg' },      // 門診鈕
	{ id: 'basicBtn',  light: 'basic_light.svg',    dark: 'basic_dark.svg',     color: 'basic_color.svg' },    // 基本資料鈕
	{ id: 'wmsBtn',    light: 'basic_light.svg',    dark: 'basic_dark.svg',     color: 'basic_color.svg' },    // 物料管理鈕(Warehouse Management System)
	{ id: 'hrBtn',     light: 'hr_light.svg',       dark: 'hr_dark.svg',        color: 'hr_color.svg' },       // 人事鈕
	{ id: 'moneyBtn',  light: 'money_light.svg',    dark: 'money_dark.svg',     color: 'money_color.svg' },    // 會計鈕
	{ id: 'treatBtn',  light: 'treat_light.svg',    dark: 'treat_dark.svg',     color: 'treat_color.svg' },    // 診治
	{ id: 'originalBtn', light: 'old.svg',          dark: 'old.svg',            color: 'old.svg' }             // 回原版首頁
];

const labelMap = [
	{ id: 'basicTitle', imgId: 'basicBtn' },
	{ id: 'wmsTitle',   imgId: 'wmsBtn' },
	{ id: 'bookTitle',  imgId: 'bookIcon' },
	{ id: 'opdTitle',   imgId: 'opdIcon' },
	{ id: 'hrTitle',    imgId: 'hrBtn' },
	{ id: 'moneyTitle', imgId: 'moneyBtn' },
	{ id: 'treatTitle', imgId: 'treatBtn' }
];

const labelColor = { light: '#C97023', dark: '#60401b', color: '#000000' };

// 處理 hover 的狀態
function handleHover(image) {
	if (typeof image !== 'undefined' && image !== null && image.id) {
		console.log("handleHover Image ID: " + image.id);
		if (activeImgId !== image.id) {
			const imgElement = document.getElementById(image.id);
			if (imgElement) {
				imgElement.src = IMG_BASE + '/' + image.dark; // 切換成暗模式
			}
		}

		const label = labelMap.find(item => item.imgId === image.id);
		if (label && activeLblId !== label.id) {
			const labelEle = document.getElementById(label.id);
			if (labelEle) {
				labelEle.style.color = labelColor.dark;
			}
		}
	} else {
		console.log("handleHover: no such Image ID.");
	}
}

// 處理離開 hover 的狀態
function handleLeave(image) {
	if (typeof image !== 'undefined' && image !== null && image.id) {
		console.log("handleLeave Image ID: " + image.id);
		if (activeImgId !== image.id) {
			const imgElement = document.getElementById(image.id);
			if (imgElement) {
				imgElement.src = IMG_BASE + '/' + image.light; // 恢復亮模式
			}
		}

		const label = labelMap.find(item => item.imgId === image.id);
		if (label && activeLblId !== label.id) {
			const labelEle = document.getElementById(label.id);
			if (labelEle) {
				labelEle.style.color = labelColor.light;
			}
		}
	} else {
		console.log("handleLeave: no such Image ID.");
	}
}

// 處理點擊的狀態
function handleClick(image) {
	// 更新所有 img 的狀態
	imgMap.forEach(img => {
		const imgElement = document.getElementById(img.id);
		if (imgElement) {
			if (img.id === image.id) {
				imgElement.src = IMG_BASE + '/' + img.color; // 被點擊的維持暗模式
				activeImgId = img.id; // 設定為當前 active
			} else {
				imgElement.src = IMG_BASE + '/' + img.light; // 其他的恢復亮模式
			}
		}
	});

	labelMap.forEach(label => {
		const labelEle = document.getElementById(label.id);
		if (labelEle) {
			if (label.imgId === image.id) {
				labelEle.style.color = labelColor.color;
				activeLblId = label.id;
			} else {
				labelEle.style.color = labelColor.light;
			}
		}
	});
}

/*************************/

window.onload = function() {
	displayWelcomeMessage(); // 頁面加載時顯示歡迎訊息
};

document.addEventListener("DOMContentLoaded", function() {
	var dialogs = document.querySelectorAll('dialog');

	// 檢查頁面中的所有 <dialog>，如果不支援，則應用 polyfill
	dialogs.forEach(function(dialog) {
		if (typeof HTMLDialogElement !== 'function') {
			dialogPolyfill.registerDialog(dialog);
		}
	});

	// 初始化 img 的事件監聽器
	imgMap.forEach(image => {
		const imgElement = document.getElementById(image.id);

		if (imgElement) {
			// 滑鼠進入或觸摸開始
			imgElement.addEventListener('mouseenter', () => handleHover(image));
			imgElement.addEventListener('touchstart', () => handleHover(image));

			// 滑鼠移出或觸摸結束
			imgElement.addEventListener('mouseleave', () => handleLeave(image));
			imgElement.addEventListener('touchend', () => handleLeave(image));

			// 點擊事件
			imgElement.addEventListener('click', () => handleClick(image));
		}
	});

	// main-buttons label 綁定事件 (`bookTitle`, `opdTitle`)
	labelMap.forEach(label => {
		const element = document.getElementById(label.id);

		if (element) {
			const image = imgMap.find(item => item.id === label.imgId);

			// 滑鼠進入或觸摸開始
			element.addEventListener('mouseenter', () => handleHover(image));
			element.addEventListener('touchstart', () => handleHover(image));

			// 滑鼠移出或觸摸結束
			element.addEventListener('mouseleave', () => handleLeave(image));
			element.addEventListener('touchend', () => handleLeave(image));

			// 點擊事件
			element.addEventListener('click', () => handleClick(image));
		}
	});

});

// 設置自動登出時間 (以毫秒為單位)，這裡是 10 分鐘 (600000 毫秒)
const AUTO_LOGOUT_TIME = 10 * 60 * 1000; 

let logoutTimer;

// 定義一個重置計時器的函數
function resetLogoutTimer() {
	// 如果有一個已經在運行的計時器，先清除它
	if (logoutTimer) {
		//overdueTime = 0; // 測試時關閉逾時自動登出，正式版須 marked
		clearTimeout(logoutTimer);
	}

	// 設置新的計時器，10 分鐘後自動登出
	logoutTimer = setTimeout(() => {
		// 當 10 分鐘內無操作時，執行登出操作
		autoLogout();
	}, AUTO_LOGOUT_TIME);
}

// 登出
function logout(reason = '') {
	// 調用登出 API
	fetch(`${API_BASE}/logout.php`, {
		method: 'POST'
	}).then(response => {
		// 重定向到登入頁面
		window.location.href = BASE_DIR + 'login.html';
	}).catch(error => {
		console.error(reason+'登出失敗:', error);
		// 即使登出失敗，也將頁面導向登入頁面
		window.location.href = BASE_DIR + 'login.html';
	});
}

// 定義一個登出函數
function autoLogout() {
	alert('由於您 10 分鐘內未進行操作，將自動登出');
	logout('自動');
}

// 回首頁
function gohome() {
	// 重定向到登入頁面
	window.location.href = BASE_DIR + 'dashboard/dashboard.html';
}

// 回原版
function goOriginal() {
	// 重定向到登入頁面
	window.location.href = '/dashboard/dashboard.html';
}

// 監聽使用者的活動，並重置登出計時器
function setupActivityListeners() {
	// 監聽所有的常見使用者行為
	window.addEventListener('mousemove', resetLogoutTimer);
	window.addEventListener('mousedown', resetLogoutTimer); // 滑鼠點擊
	window.addEventListener('keydown', resetLogoutTimer); // 鍵盤輸入
	window.addEventListener('scroll', resetLogoutTimer);  // 滾動
	window.addEventListener('touchstart', resetLogoutTimer); // 觸摸事件（手機端）
}

// 初始化登出邏輯
function initializeAutoLogout() {
	setupActivityListeners(); // 設置監聽器
	resetLogoutTimer(); // 啟動計時器
}

// 當頁面加載後，初始化自動登出邏輯
window.onload = initializeAutoLogout;

function hideWelcomeMessage() {
	document.getElementById('welcome-message').style.display = 'none';
}

/* 載入設定管理相關的功能鈕 */
function setSubActiveButton(activeId) {
	// changed main-buttom background 
	document.querySelectorAll('.sub-buttons button').forEach(button => {
		button.classList.remove('active');
		button.style.backgroundColor = '#7777'; // 預設其他按鈕背景色
		button.style.color = '#fff'; // 預設其他按鈕背景色
	});
	document.getElementById(activeId).classList.add('active');
	document.getElementById(activeId).style.backgroundColor = '#fff'; // 當前按鈕背景色
	document.getElementById(activeId).style.color = '#000'; // 當前按鈕背景色

	handleDynamicContent(activeId);
	switch (activeId) {
		case "accounts": // systemBtn
			loadContent('system', activeId);
			break;
		case "dieases": // basicBtn
			loadContent('basic', activeId);
			break;
		case "efficacys": // basicBtn
			loadContent('basic', activeId);
			break;
		case "formulas": // basicBtn
			loadContent('basic', activeId);
			break;
		case "materials": // basicBtn
			loadContent('basic', activeId);
			break;
		case "symptoms": // basicBtn
			loadContent('basic', activeId);
			break;
		case "syndromes": // basicBtn
			loadContent('basic', activeId);
			break;
		case "inspections": // basicBtn
			loadContent('basic', activeId);
			break;
		case "olfactions": // basicBtn
			loadContent('basic', activeId);
			break;
		case "palpations": // basicBtn
			loadContent('basic', activeId);
			break;
		case "diagnostics": // basicBtn
			loadContent('basic', activeId);
			break;
		case "acupunctures": // basicBtn
			loadContent('basic', activeId);
			break;
		case "patients": // bookBtn
			loadContent('book', activeId);
			break;
		case "employees": // hrBtn
			loadContent('hr', activeId);
			break;
		case "opd": // opdBtn
			loadContent('outpatients', 'outpatients');
			break;
		case "mypatients": // opdBtn
			loadContent('outpatients', activeId);
			break;
		case "medicine": // wmsBtn
			loadContent('warehouse', activeId);
			break;
		case "registered": // bookBtn
			loadContent('book', activeId);
			break;
		case "scheduling": // bookBtn
			loadContent('book', activeId);
			break;
		default:
			loadDefault(activeId);
	}
}

/* 載入設定管理相關的功能鈕 */
function loadSystemBtn() {
	const subButtonsContainer = document.getElementById('sub-buttons');
	subButtonsContainer.innerHTML = ''; // 清空次要功能按鈕

	subButtonsContainer.innerHTML = `
		    <button class="sub-button" onclick="setSubActiveButton('accounts')" id="accounts">帳號管理</button>
		    <button class="sub-button" onclick="setSubActiveButton('service')" id="service">服務管理</button>
		    <button class="sub-button" onclick="setSubActiveButton('log')" id="log">記錄管理</button>
		    <button class="sub-button" onclick="setSubActiveButton('setpass')" id="setpass">修改密碼</button>
		`;

	// 顯示次要功能按鈕區
	subButtonsContainer.style.display = 'flex';
	setSubActiveButton('accounts');
}

/* 載入掛號管理相關的功能鈕 */
function loadBookBtn() {
	const subButtonsContainer = document.getElementById('sub-buttons');
	subButtonsContainer.innerHTML = ''; // 清空次要功能按鈕

	subButtonsContainer.innerHTML = `
		    <button class="sub-button" onclick="setSubActiveButton('registered')" id="registered">現場掛號</button>
		    <button class="sub-button" onclick="setSubActiveButton('appointment')" id="appointment">網路掛號</button>
		    <button class="sub-button" onclick="setSubActiveButton('patients')" id="patients">患者資料</button>
		    <button class="sub-button" onclick="setSubActiveButton('scheduling')" id="scheduling">排班管理</button>
		`;

	// 顯示次要功能按鈕區
	subButtonsContainer.style.display = 'flex';
	setSubActiveButton('registered');
}

/* 載入門診管理相關的功能鈕 */
function loadOutpatientBtn() {
	const subButtonsContainer = document.getElementById('sub-buttons');
	subButtonsContainer.innerHTML = ''; // 清空次要功能按鈕

	subButtonsContainer.innerHTML = `
					<button class="sub-button" onclick="setSubActiveButton('opd')" id="opd">門診</button>
					<button class="sub-button" onclick="setSubActiveButton('check')" id="check">檢查記錄</button>
					<button class="sub-button" onclick="setSubActiveButton('mypatients')" id="mypatients">我的病人</button>
					`;

	// 顯示次要功能按鈕區
	subButtonsContainer.style.display = 'flex';
	setSubActiveButton('opd');
}

/* 載入物料管理相關的功能鈕 */
function loadWmsBtn() {
	const subButtonsContainer = document.getElementById('sub-buttons');
	subButtonsContainer.innerHTML = ''; // 清空次要功能按鈕

	subButtonsContainer.innerHTML = `
					<button class="sub-button" onclick="setSubActiveButton('disinfect')" id="disinfect">消毒</button>
					<button class="sub-button" onclick="setSubActiveButton('equipment')" id="equipment">器材</button>
					<button class="sub-button" onclick="setSubActiveButton('medicine')" id="medicine">藥品</button>
					<button class="sub-button" onclick="setSubActiveButton('oopMedical')" id="oopMedical">自費購買</button>
					`;

	// 顯示次要功能按鈕區
	subButtonsContainer.style.display = 'flex';
	setSubActiveButton('disinfect');
}

/* 載入基本資料相關的功能鈕 */
function loadBasicBtn() {
	const subButtonsContainer = document.getElementById('sub-buttons');
	subButtonsContainer.innerHTML = ''; // 清空次要功能按鈕

	subButtonsContainer.innerHTML = `
					<button class="sub-button" onclick="setSubActiveButton('disinfect')" id="disinfect">消毒</button>
					<button class="sub-button" onclick="setSubActiveButton('equipment')" id="equipment">器材</button>
					<button class="sub-button" onclick="setSubActiveButton('medicine')" id="medicine">藥品</button>
					<button class="sub-button" onclick="setSubActiveButton('oopMedical')" id="oopMedical">自費購買</button>
					`;

	// 顯示次要功能按鈕區
	subButtonsContainer.style.display = 'flex';
	setSubActiveButton('disinfect');
}

/* 載入診治百科相關的功能鈕 */
function loadTreatBtn() {
	const subButtonsContainer = document.getElementById('sub-buttons');
	subButtonsContainer.innerHTML = ''; // 清空次要功能按鈕

	subButtonsContainer.innerHTML = `
					<button class="sub-button" onclick="setSubActiveButton('materials')" id="materials">藥材</button>
					<button class="sub-button" onclick="setSubActiveButton('formulas')" id="formulas">方劑</button>
					<button class="sub-button" onclick="setSubActiveButton('syndromes')" id="syndromes">証型資料</button>
					<button class="sub-button" onclick="setSubActiveButton('symptoms')" id="symptoms">症狀資料</button>
					<button class="sub-button" onclick="setSubActiveButton('dieases')" id="dieases">疾病資料</button>
					<button class="sub-button" onclick="setSubActiveButton('efficacys')" id="efficacys">功效資料</button>
					<button class="sub-button" onclick="setSubActiveButton('diagnostics')" id="diagnostics">四診</button>
					<button class="sub-button" onclick="setSubActiveButton('acupunctures')" id="acupunctures">針炙穴位</button>
					`;

	// 顯示次要功能按鈕區
	subButtonsContainer.style.display = 'flex';
	setSubActiveButton('materials');
}

/* 載入會計管理相關的功能鈕 */
function loadMoneyBtn() {
	const subButtonsContainer = document.getElementById('sub-buttons');
	subButtonsContainer.innerHTML = ''; // 清空次要功能按鈕

	subButtonsContainer.innerHTML = `
					<button class="sub-button" onclick="setSubActiveButton('staffSalary')" id="staffSalary">員工薪資</button>
					<button class="sub-button" onclick="setSubActiveButton('manufacturer')" id="manufacturer">廠商資料</button>
					<button class="sub-button" onclick="setSubActiveButton('accounting')" id="accounting">記帳</button>
					<button class="sub-button" onclick="setSubActiveButton('financial')" id="financial">銀行帳戶</button>
					<button class="sub-button" onclick="setSubActiveButton('opex')" id="opex">營運成本</button>
					<button class="sub-button" onclick="setSubActiveButton('is')" id="is">損益表</button>
					`;

	// 顯示次要功能按鈕區
	subButtonsContainer.style.display = 'flex';
	setSubActiveButton('staffSalary');
}

/* 載入人事管理相關的功能鈕 */
function loadHrBtn() {
	const subButtonsContainer = document.getElementById('sub-buttons');
	subButtonsContainer.innerHTML = ''; // 清空次要功能按鈕

	subButtonsContainer.innerHTML = `
					<button class="sub-button" onclick="setSubActiveButton('employees')" id="employees">員工個資</button>
					<button class="sub-button" onclick="setSubActiveButton('salary')" id="salary">薪資</button>
					<button class="sub-button" onclick="setSubActiveButton('sttendance')" id="sttendance">出勤</button>
					<button class="sub-button" onclick="setSubActiveButton('assessment')" id="assessment">考核</button>
					<button class="sub-button" onclick="setSubActiveButton('insurance')" id="insurance">保險</button>
					`;

	// 顯示次要功能按鈕區
	subButtonsContainer.style.display = 'flex';
	setSubActiveButton('employees');
}

/* 載入次要功能相關頁面 */
function handleDynamicContent(subId) {
	const divEle = document.getElementById('dynamic-content');
	divEle.style.backgroundColor = '#fff';
}

function loadDefault(activeId) {
	document.getElementById('dynamic-content').innerHTML = `
		<h3 style="margin: 0; align: center;">${activeId}</h3>
		`;
}

function loadContent(path, func) {
	let funcname; // 提升作用域
	console.log("path: "+path+", func: "+func);
	switch (func) {
		case "accounts": // systemBtn
			funcname = 'loadUsers';
			break;
		case "dieases": // basicBtn
			funcname = 'loadDieases';
			break;
		case "efficacys": // basicBtn
			funcname = 'loadEfficacys';
			break;
		case "formulas": // basicBtn
			funcname = 'loadFormulas';
			break;
		case "materials": // basicBtn
			funcname = 'loadMaterials';
			break;
		case "symptoms": // basicBtn
			funcname = 'loadSymptoms';
			break;
		case "syndromes": // basicBtn
			funcname = 'loadSyndromes';
			break;
		case "inspections": // basicBtn
			funcname = 'loadInspections';
			break;
		case "olfactions": // basicBtn
			funcname = 'loadOlfactions';
			break;
		case "palpations": // basicBtn
			funcname = 'loadPalpations';
			break;
		case "diagnostics": // basicBtn
			funcname = 'loadDiagnostics';
			break;
		case "acupunctures": // basicBtn
			funcname = 'loadAcupunctures';
			break;
		case "patients": // bookBtn
			funcname = 'loadPatients';
			break;
		case "employees": // hrBtn
			funcname = 'loadEmployees';
			break;
		case "outpatients": // opdBtn
			funcname = 'initOutpatients';
			break;
		case "mypatients": // opdBtn
			funcname = 'loadMypatients';
			break;
		case "medicine": // wmsBtn
			funcname = 'medicineInit';
			break;
		case "registered": // bookBtn
			funcname = 'loadRegistered';
			break;
		case "scheduling": // bookBtn
			funcname = 'loadScheduling';
			break;
		default:
			loadDefault(func);
			return;
	}
	const html = func === 'accounts' ? 'permissions' : func;
	const js = func === 'accounts' ? 'users' : func;

	// 載入對應的 HTML 和 JS
	fetch(`${HTML_BASE}/${path}/${html}.html`)
		.then(response => {
			if (!response.ok) {
				throw new Error(`HTTP error! status: ${response.status}`);
			}
			return response.text();
		})
		.then(html => {
			const dynamicContent = document.getElementById('dynamic-content');
			dynamicContent.innerHTML = html;
			dynamicContent.style.display = 'block';

			// 依序載入：先主 JS，再 guard（若沒有 guard，catch 後忽略即可）
			const mainJs = `${JS_BASE}/${js}.js?v=${Date.now()}`;
			let guardJs = '';

			if (func === "dieases" || func === "efficacys" || func === "formulas" ||
				func === "materials" || func === "symptoms" || func === "syndromes" ||
				func === "diagnostics" || func === "acupunctures") {
				guardJs = `${JS_BASE}/${js}-acl-guard.js?v=${Date.now()}`;
			}

			loadScriptOnce(mainJs)
				.then(() => guardJs ? loadScriptOnce(guardJs).catch(() => {/* 沒 guard 就算了 */}) : Promise.resolve())
				.then(() => {
					console.log("onLoad:", js, "/", funcname, 'typeof=', typeof window[funcname]);
					if (typeof window[funcname] === 'function') {
						window[funcname](); // 呼叫動態載入的函式（舊行為）
					} else {
						console.error(`${funcname} function not found in ${js}.js`);
					}
				})
				.catch(() => {
					console.error(`Error loading ${js}.js or its guard`);
				});
		})
		.catch(error => {
			console.error(`Error loading ${html}.html:`, error);
		});
}

// ===== ACL & 設定齒輪（最小侵入）======
(function(){
	function applySettingsGuard(){
		try{
			const role = (window.ACL && ACL.getRole) ? ACL.getRole() : parseInt(sessionStorage.getItem('role')||'0',10);
			const gear = document.getElementById('settings');
			const admin = (window.ACL && ACL.isAdmin) ? ACL.isAdmin(role) : (role === 255);
			if (gear && !admin) {
				gear.style.pointerEvents = 'none';
				gear.style.opacity = '0.5';
				gear.title = '僅管理者可使用';
				if ('disabled' in gear) gear.disabled = true;
				gear.removeAttribute('onclick');
			}
		}catch(e){}
	}
	function applyAcl(){
		try{ if (window.ACL && ACL.applyAclToDom) ACL.applyAclToDom(document); }catch(e){}
	}
	const init = ()=>{ applySettingsGuard(); applyAcl(); };
	if (document.readyState === 'loading') {
		document.addEventListener('DOMContentLoaded', init);
	} else {
		init();
	}
})();

// ===== 主功能權限（診治百科以外）比照 settings 禁用 — 強化：同時處理 *Btn 與 *Title =====
(function () {
	'use strict';

	function getRole() {
		try {
			if (window.ACL && typeof ACL.getRole === 'function') return ACL.getRole();
			return parseInt(sessionStorage.getItem('role') || '0', 10);
		} catch { return 0; }
	}

	var ROLE = (window.ACL && ACL.ROLE) ? ACL.ROLE : {
		READ_ONLY: 0, HR: 1, ACC: 2, WH: 4, MED: 8, ADMIN: 255
	};

	// 每個主功能 → 需要的權限 bit 與要一併禁用的 DOM id（*Btn + *Title）
	const FEATURE_GUARDS = [
		{ ids: ['hrBtn','hrTitle'],       mask: ROLE.HR,  name: '人事管理' },
		{ ids: ['moneyBtn','moneyTitle'], mask: ROLE.ACC, name: '會計管理' },
		{ ids: ['wmsBtn','wmsTitle'],     mask: ROLE.WH,  name: '物料管理' },
		{ ids: ['bookBtn','bookTitle'],   mask: ROLE.MED, name: '掛號管理' },
		{ ids: ['opdBtn','opdTitle'],     mask: ROLE.MED, name: '門診管理' },
		// treatBtn / treatTitle 不列入限制（診治百科照舊開放）
	];

	// 直接呼叫載入函式也要擋（名稱依你專案調整）
	const LOADER_REQ = {
		loadHRBtn: ROLE.HR,
		loadMoneyBtn: ROLE.ACC,
		loadWMSBtn: ROLE.WH,
		loadBookBtn: ROLE.MED,
		loadOutpatientBtn: ROLE.MED
	};

	function hasModule(mask) {
		const r = getRole();
		if (r === ROLE.ADMIN) return true;
		return !!(r & mask);
	}

	function disableLikeSettings(el, tip) {
		if (!el) return;
		if ('disabled' in el) el.disabled = true;
		el.setAttribute('aria-disabled','true');
		el.style.pointerEvents = 'none';
		el.style.opacity = '0.5';
		el.style.cursor = 'not-allowed';
		if (!el.title) el.title = tip || '僅有權限者可使用';
		// 捕獲階段硬擋任何點擊（即使外層有事件代理也擋得住）
		el.__mainACLClickGuard__ && el.removeEventListener('click', el.__mainACLClickGuard__, true);
		const handler = function(e){ e.preventDefault(); e.stopPropagation(); alert(tip || '您沒有此主功能的權限'); return false; };
		el.addEventListener('click', handler, true);
		el.__mainACLClickGuard__ = handler;
	}

	function guardElements(ids, mask, label) {
		if (hasModule(mask)) return;
		const tip = `您沒有「${label}」的權限`;
		ids.forEach(id => disableLikeSettings(document.getElementById(id), tip));
	}

	function guardLoader(fnName, mask, label) {
		const orig = window[fnName];
		if (typeof orig !== 'function') return;
		if (orig.__mainGuard__) return;
		const wrapped = function () {
			if (!hasModule(mask)) { alert(`您沒有「${label || '此主功能'}」的權限`); return; }
			return orig.apply(this, arguments);
		};
		wrapped.__mainGuard__ = true;
		window[fnName] = wrapped;
	}

	function applyMainFeatureGuards() {
		FEATURE_GUARDS.forEach(f => guardElements(f.ids, f.mask, f.name));
		Object.keys(LOADER_REQ).forEach(fn => guardLoader(fn, LOADER_REQ[fn], fn));
	}

	function init() { try { applyMainFeatureGuards(); } catch {} }

	if (document.readyState === 'loading') document.addEventListener('DOMContentLoaded', init);
	else init();

	// 若 navbar / 標題區會被重繪：監看 DOM，去抖後重套一次
	let t = null;
	try {
		const mo = new MutationObserver(() => { clearTimeout(t); t = setTimeout(applyMainFeatureGuards, 80); });
		mo.observe(document.body, { childList: true, subtree: true });
	} catch {}
})();


// ------------------------------------------------------------------

(function () {
	/********************************************************
	 * 0. 小工具：本地 yyyy-mm-dd
	 ********************************************************/
	function getLocalDateISO() {
		const now = new Date();
		const y = now.getFullYear();
		const m = String(now.getMonth() + 1).padStart(2, '0');
		const d = String(now.getDate()).padStart(2, '0');
		return `${y}-${m}-${d}`;
	}

	/********************************************************
	 * 1. 八字核心（放在自己的 namespace）
	 ********************************************************/
	const GAN = ['甲','乙','丙','丁','戊','己','庚','辛','壬','癸'];
	const ZHI = ['子','丑','寅','卯','辰','巳','午','未','申','酉','戌','亥'];
	const SOLAR_CACHE = {};

	function isBeforeLiChun(y,m,d) {
		return (m < 2) || (m === 2 && d < 4);
	}

	function getYearPillar(y,m,d) {
		const yy = isBeforeLiChun(y,m,d) ? y - 1 : y;
		return {
			gan: GAN[(yy - 4) % 10],
			zhi: ZHI[(yy - 4) % 12],
		};
	}

	async function fetchSolarTerms(y) {
		if (SOLAR_CACHE[y]) return SOLAR_CACHE[y];
		// 嘗試叫後端，叫不到就用固定值
		try {
			const res = await fetch(`/api/calendar/solar_terms.php?year=${y}`, { credentials: 'include' });
			const json = await res.json();
			if (json && json.result && json.data) {
				SOLAR_CACHE[y] = json.data;
				return json.data;
			}
		} catch (e) {}
		const data = {
			li_chun:    `${y}-02-04`,
			jing_zhe:   `${y}-03-06`,
			qing_ming:  `${y}-04-05`,
			li_xia:     `${y}-05-06`,
			mang_zhong: `${y}-06-06`,
			xiao_shu:   `${y}-07-07`,
			li_qiu:     `${y}-08-07`,
		};
		SOLAR_CACHE[y] = data;
		return data;
	}

	// 這裡有修你說的：芒種～小暑 = 午；小暑～立秋 = 未
	async function getMonthIndexByTerm(y,m,d) {
		const terms = await fetchSolarTerms(y);
		const curUTC = Date.UTC(y, m - 1, d);

		const liChun    = Date.parse(terms.li_chun);
		const jingZhe   = Date.parse(terms.jing_zhe);
		const qingMing  = Date.parse(terms.qing_ming);
		const liXia     = Date.parse(terms.li_xia);
		const mangZhong = Date.parse(terms.mang_zhong);
		const xiaoShu   = Date.parse(terms.xiao_shu);
		const liQiu     = Date.parse(terms.li_qiu);

		const md = m * 100 + d;
		if (md >= 1207 || md < 106) return 11; // 子月
		if (md >= 106 && md < 204)  return 12; // 丑月

		if (curUTC >= liChun && curUTC < jingZhe)   return 1;
		if (curUTC >= jingZhe && curUTC < qingMing) return 2;
		if (curUTC >= qingMing && curUTC < liXia)   return 3;
		if (curUTC >= liXia && curUTC < mangZhong)  return 4;
		if (curUTC >= mangZhong && curUTC < xiaoShu) return 5;
		if (curUTC >= xiaoShu && curUTC < liQiu)     return 6;

		if (md >= 807 && md < 908)   return 7;
		if (md >= 908 && md < 1008)  return 8;
		if (md >= 1008 && md < 1107) return 9;
		if (md >= 1107 && md < 1207) return 10;

		return 1;
	}

	function getMonthPillar(yearGan, monthIndex) {
		const yg = GAN.indexOf(yearGan);
		let first;
		if (yg === 0 || yg === 5)      first = 2;
		else if (yg === 1 || yg === 6) first = 4;
		else if (yg === 2 || yg === 7) first = 6;
		else if (yg === 3 || yg === 8) first = 8;
		else                           first = 0;
		return {
			gan: GAN[(first + (monthIndex - 1)) % 10],
			zhi: ZHI[(monthIndex + 1) % 12],
		};
	}

	// 日柱：1912-02-18 甲子，UTC 算天數
	function getDayPillar(y,m,d) {
		const baseUTC = Date.UTC(1912,1,18);
		const curUTC  = Date.UTC(y, m-1, d);
		const diff = Math.floor((curUTC - baseUTC) / 86400000);
		const idx = ((diff % 60) + 60) % 60;
		return {
			gan: GAN[idx % 10],
			zhi: ZHI[idx % 12],
		};
	}

	function getHourPillar(dayGan, hour) {
		const zhiIndex = Math.floor((hour + 1) / 2) % 12;
		const dayGanIdx = GAN.indexOf(dayGan);
		const gan = GAN[(dayGanIdx * 2 + zhiIndex) % 10];
		return { gan, zhi: ZHI[zhiIndex] };
	}

	async function calcBaziAccurateAsync(y, m, d, h, mi) {
		// 這裡不要直接回「請選日期」，改成回「日期格式錯誤」方便你看
		if (!y || !m || !d || isNaN(y) || isNaN(m) || isNaN(d)) {
			return '日期格式錯誤';
		}
		const yP = getYearPillar(y, m, d);
		const mIdx = await getMonthIndexByTerm(y, m, d);
		const mP = getMonthPillar(yP.gan, mIdx);
		const dP = getDayPillar(y, m, d);
		const hP = getHourPillar(dP.gan, h || 0);
		return `${yP.gan}${yP.zhi}年 ${mP.gan}${mP.zhi}月 ${dP.gan}${dP.zhi}日 ${hP.gan}${hP.zhi}時`;
	}

	// 給外面用（萬一你別處也要算）
	window.ClinicBazi = {
		calc: calcBaziAccurateAsync
	};

	/********************************************************
	 * 2. 萬年曆彈窗（小時寫死在 HTML，不做 options.length）
	 ********************************************************/
	function setupBaziModal() {
		// 浮動鈕
		if (!document.getElementById('floating-fnbar')) {
			const bar = document.createElement('div');
			bar.id = 'floating-fnbar';
			bar.innerHTML = '<button id="btn-bazi-calendar">萬年曆</button>';
			document.body.appendChild(bar);
		}

		// 彈窗
		if (!document.getElementById('bazi-modal')) {
			const modal = document.createElement('div');
			modal.id = 'bazi-modal';
			modal.className = 'hidden';
			modal.innerHTML = `
	<div class="bazi-box">
	  <div class="bazi-header">
	    <span>萬年曆八字換算</span>
	    <button id="bazi-close" class="close-btn">×</button>
	  </div>
	  <div class="bazi-body">
	    <label>西元年
	      <input type="date" id="bz-date">
	    </label>
	    <label>小時(24h)
	      <select id="bz-hour">
		${Array.from({length:24},(_,i)=>`<option value="${i}">${i}</option>`).join('')}
	      </select>
	    </label>
	    <label>分鐘
	      <input type="number" id="bz-minute" min="0" max="59" value="0">
	    </label>
	    <button id="bz-submit" class="primary-btn">換算</button>
	  </div>
	  <div class="bazi-result" id="bazi-result"></div>
	</div>
      `;
			document.body.appendChild(modal);
		}

		const openBtn = document.getElementById('btn-bazi-calendar');
		const modal   = document.getElementById('bazi-modal');
		const closeBtn= document.getElementById('bazi-close');

		if (openBtn) {
			openBtn.addEventListener('click', () => {
				const dateEl = document.getElementById('bz-date');
				const hourEl = document.getElementById('bz-hour');
				const minEl  = document.getElementById('bz-minute');
				const resEl  = document.getElementById('bazi-result');

				if (dateEl) {
					const iso = getLocalDateISO();
					dateEl.value = iso;
					if ('valueAsDate' in dateEl) dateEl.valueAsDate = new Date();
				}
				if (hourEl) hourEl.value = String(new Date().getHours());
				if (minEl)  minEl.value  = String(new Date().getMinutes()).padStart(2,'0');
				if (resEl)  resEl.textContent = '';

				modal.classList.remove('hidden');
			});
		}

		if (closeBtn) {
			closeBtn.addEventListener('click', () => modal.classList.add('hidden'));
		}
		if (modal) {
			modal.addEventListener('click', e => {
				if (e.target === modal) modal.classList.add('hidden');
			});
		}

		// 換算
		const submitBtn = document.getElementById('bz-submit');
		if (submitBtn) {
			submitBtn.addEventListener('click', async () => {
				// 這裡每次都重新抓，避免抓到已經被換掉的節點
				const dateEl = document.getElementById('bz-date');
				const hourEl = document.getElementById('bz-hour');
				const minEl  = document.getElementById('bz-minute');
				const resEl  = document.getElementById('bazi-result');

				if (dateEl && !dateEl.value) {
					const iso = getLocalDateISO();
					dateEl.value = iso;
					if ('valueAsDate' in dateEl) dateEl.valueAsDate = new Date();
				}

				const v = dateEl ? dateEl.value : '';
				if (!v) {
					if (resEl) resEl.textContent = '請選日期';
					return;
				}

				const parts = v.split('-');
				if (parts.length !== 3) {
					if (resEl) resEl.textContent = '日期格式錯誤';
					return;
				}

				const y  = parseInt(parts[0], 10);
				const m  = parseInt(parts[1], 10);
				const d  = parseInt(parts[2], 10);
				const hh = hourEl ? parseInt(hourEl.value || '0', 10) : 0;
				const mi = minEl  ? parseInt(minEl.value  || '0', 10) : 0;

				const text = await window.ClinicBazi.calc(y, m, d, hh, mi);
				if (resEl) resEl.textContent = text;
			});
		}
	}

	/********************************************************
	 * 3. 日期 hover / touch → 顯示八字（用同一份 calc）
	 ********************************************************/
	function setupDateHoverTip() {
		// 共用的 tip
		let tip = document.getElementById('bazi-tip');
		if (!tip) {
			tip = document.createElement('div');
			tip.id = 'bazi-tip';
			tip.style.display = 'none';
			document.body.appendChild(tip);
		}

		// 接受 6 種格式：
		// 1. 2025/11/01
		// 2. 2025-11-01
		// 3. 2025/11/01 12:33
		// 4. 2025-11-01 12:33
		// 5. 2025/11/01 12:33:44
		// 6. 2025-11-01 12:33:44
		const dateRegex = /(\d{4})[-\/](\d{1,2})[-\/](\d{1,2})(?:\s+(\d{1,2})(?::(\d{1,2})(?::(\d{1,2}))?)?)?/;

		async function toBazi(text) {
			const m = text.trim().match(dateRegex);
			if (!m) return null;
			const y  = parseInt(m[1], 10);
			const mm = parseInt(m[2], 10);
			const d  = parseInt(m[3], 10);
			const hh = m[4] ? parseInt(m[4], 10) : 0;
			const mi = m[5] ? parseInt(m[5], 10) : 0;
			// 用你前面已經建立好的 window.ClinicBazi
			return await window.ClinicBazi.calc(y, mm, d, hh, mi);
		}

		function showTip(msg, x, y) {
			tip.textContent = msg;
			tip.style.left = (x + 12) + 'px';
			tip.style.top  = (y + 12) + 'px';
			tip.style.position = 'fixed';
			tip.style.background = 'rgba(0,0,0,.85)';
			tip.style.color = '#fff';
			tip.style.padding = '6px 10px';
			tip.style.borderRadius = '6px';
			tip.style.fontSize = '12px';
			tip.style.pointerEvents = 'none';
			tip.style.zIndex = '99999';
			tip.style.display = 'block';
		}

		function hideTip() {
			tip.style.display = 'none';
		}

		// 讓 tip 跟著滑鼠
		document.addEventListener('mousemove', e => {
			if (tip.style.display === 'block') {
				tip.style.left = (e.clientX + 12) + 'px';
				tip.style.top  = (e.clientY + 12) + 'px';
			}
		});

		// ① 滑鼠 hover 到日期字串
		document.addEventListener('mouseover', async e => {
			const text = (e.target.textContent || '').trim();
			if (!dateRegex.test(text)) return;
			const bz = await toBazi(text);
			if (!bz) return;
			showTip(bz, e.clientX, e.clientY);
		});
		document.addEventListener('mouseout', hideTip);

		// ② 手機 touch 到日期字串
		document.addEventListener('touchstart', async e => {
			const t = e.touches[0];
			if (!t) return;
			const el = document.elementFromPoint(t.clientX, t.clientY);
			const text = el && el.textContent ? el.textContent.trim() : '';
			if (!dateRegex.test(text)) { hideTip(); return; }
			const bz = await toBazi(text);
			if (!bz) { hideTip(); return; }
			showTip(bz, t.clientX, t.clientY);
		});

		// ③ ✅ 新增：框選一段日期字串也顯示 tip
		document.addEventListener('selectionchange', async () => {
			const sel = window.getSelection ? window.getSelection() : document.getSelection();
			if (!sel) return;
			const txt = sel.toString().trim();
			if (!txt) {
				hideTip();
				return;
			}
			// 只處理像日期的選取
			if (!dateRegex.test(txt)) {
				hideTip();
				return;
			}

			// 算八字
			const bz = await toBazi(txt);
			if (!bz) {
				hideTip();
				return;
			}

			// 計算選取範圍的位置
			let x = 20, y = 20;
			try {
				if (sel.rangeCount > 0) {
					const range = sel.getRangeAt(0);
					const rect = range.getBoundingClientRect();
					// rect 可能為 0（例如跨多個元素），給個 fallback
					x = (rect.left || 20);
					y = (rect.top  || 20);
				}
			} catch (e) {
				// 失敗就用預設位置
			}

			showTip(bz, x, y);
		});
	}

	/********************************************************
	 * 4. 啟動
	 ********************************************************/
	document.addEventListener('DOMContentLoaded', function(){
		setupBaziModal();
		setupDateHoverTip();
	});

})();

/********************************************************
 * 設定密碼
 ********************************************************/
async function submitChangePassword() {
	const oldPwd = document.getElementById('oldPassword').value.trim();
	const newPwd = document.getElementById('newPassword').value.trim();
	const newPwd2 = document.getElementById('newPassword2').value.trim();

	if (!oldPwd || !newPwd || !newPwd2) {
		alert('請完整填寫所有欄位');
		return;
	}
	if (newPwd !== newPwd2) {
		alert('兩次新密碼不一致');
		return;
	}

	try {
		const res = await fetch('/api/user_management/change_password.php', {
			method: 'POST',
			credentials: 'include',
			headers: { 'Content-Type': 'application/json' },
			body: JSON.stringify({
				old_password: oldPwd,
				new_password: newPwd
			})
		});

		/* 先拿「純文字」 */
		const text = await res.text();

		/* 防呆：空回傳 */
		if (!text) {
			throw new Error('系統未回傳任何資料，請檢查 API');
		}

		/* 再手動 parse */
		let json;
		try {
			json = JSON.parse(text);
		} catch (e) {
			console.error('API Raw Response:', text);
			throw new Error('系統回傳格式錯誤（非 JSON）');
		}

		if (!json.result) {
			throw new Error(json.msg || '變更失敗');
		}

		/*
		const json = await res.json();
		if (!json.result) throw new Error(json.msg || '變更失敗');
		*/

		alert('密碼已修改完成，請重新登入');
		closeDialog('changePasswordDialog');
		logout(); // 安全起見，強制重新登入

	} catch (err) {
		alert(err.message);
	}
}

