/* =====================================================
   排班管理（月曆版） scheduling.js
   ===================================================== */

(() => {
	'use strict';

	let currentMonthDate;
	let schedulingData;
	let conflictMap;
	let shiftMap;
	let holidayMap;
	let editingId;
	let longPressTimer;

	let calendarGrid, calendarTitle;
	let scheduleDialog, scheduleForm;
	let work_date, staff_id, staff_role;
	let shift_id, start_time, end_time, remark, dlgTitle;

	window.loadScheduling = async function () {

		currentMonthDate = new Date();
		schedulingData = {};
		conflictMap = {};
		shiftMap = {};
		holidayMap = {};
		editingId = null;

		calendarGrid  = document.getElementById('calendarGrid');
		calendarTitle = document.getElementById('calendarTitle');

		scheduleDialog = document.getElementById('scheduleDialog');
		scheduleForm   = document.getElementById('scheduleForm');

		work_date  = document.getElementById('work_date');
		staff_id   = document.getElementById('staff_id');
		staff_role = document.getElementById('staff_role');
		shift_id   = document.getElementById('shift_id');
		start_time = document.getElementById('start_time');
		end_time   = document.getElementById('end_time');
		remark     = document.getElementById('remark');
		dlgTitle   = document.getElementById('dlgTitle');

		bindHeader();
		bindForm();
		loadStaff();
		await loadJobTitles();   // ← 新增
		loadShift();
		/* 人員 onchange */
		staff_id.onchange = () => {
			syncStaffRoleByStaffId(staff_id.value);
		};

		renderCalendar();
	};

	/* ===== Header ===== */
	function bindHeader() {
		document.getElementById('prevMonth').onclick = () => {
			currentMonthDate.setMonth(currentMonthDate.getMonth() - 1);
			renderCalendar();
		};
		document.getElementById('nextMonth').onclick = () => {
			currentMonthDate.setMonth(currentMonthDate.getMonth() + 1);
			renderCalendar();
		};
	}

	/* ===== 人員 ===== */
	function loadStaff() {
		fetch('/api/scheduling/get_employee_options.php')
			.then(r => r.json())
			.then(res => {
				staff_id.innerHTML = '<option value="">請選擇</option>';
				if (!res.result) return;
				res.data.records.forEach(e => {
					const o = document.createElement('option');
					o.value = e.employee_id;
					o.textContent = e.cname;
					staff_id.appendChild(o);
				});
			});
	}

	/* ===== 角色 ===== */
	function loadJobTitles() {
		staff_role.innerHTML = '<option value="0">請指定角色</option>';

		return fetch('/api/scheduling/get_jobtitles.php')
			.then(r => r.json())
			.then(res => {
				if (!res.result) return;

				const records = res.data?.records || [];
				records.forEach(j => {
					const opt = document.createElement('option');
					opt.value = String(j.jobtitle_id);
					opt.textContent = j.title;
					staff_role.appendChild(opt);
				});
			});
	}

	/* ===== 班別 ===== */
	function loadShift() {
		fetch('/api/scheduling/get_shift_settings.php')
			.then(r => r.json())
			.then(res => {
				shift_id.innerHTML = '<option value="">請選擇</option>';
				shiftMap = {};
				if (!res.result) return;
				res.data.records.forEach(s => {
					shiftMap[s.id] = s;
					const o = document.createElement('option');
					o.value = s.id;
					o.textContent = `${s.shift_name} (${s.start_time}-${s.end_time})`;
					shift_id.appendChild(o);
				});
			});

		shift_id.onchange = () => {
			const s = shiftMap[shift_id.value];
			if (!s) return;
			start_time.value = s.start_time;
			end_time.value   = s.end_time;
		};
	}

	/* ===== staff_id → staff_role 同步函式 ===== */
	function syncStaffRoleByStaffId(empId) {
		if (!empId) {
			staff_role.value = '0';
			return;
		}

		fetch(`/api/scheduling/get_employee_title.php?employee_id=${encodeURIComponent(empId)}`)
			.then(r => r.json())
			.then(res => {
				const rec = res.data?.records?.[0];
				const jobId = rec ? String(rec.jobtitle_id) : '0';

				// ⭐ 診斷用（你現在一定要看這個）
				console.log('[sync]', {
					empId,
					jobId,
					options: [...staff_role.options].map(o => o.value)
				});

				const opt = staff_role.querySelector(`option[value="${jobId}"]`);
				staff_role.value = opt ? jobId : '0';
			});
	}

	/* ===== 假日 ===== */
	function loadHoliday(year) {
		holidayMap = {};
		return fetch(`/data/holidays/holiday-${year}.txt`)
			.then(r => r.ok ? r.text() : '')
			.then(t => {
				t.split('\n').forEach(l => {
					const x = l.trim();
					if (!x) return;
					const [d, ...n] = x.split(' ');
					holidayMap[d] = n.join(' ');
				});
			});
	}

	/* ===== 月曆 ===== */
	function renderCalendar() {
		const y = currentMonthDate.getFullYear();
		const m = currentMonthDate.getMonth() + 1;

		calendarTitle.textContent =
			`${y}年${String(m).padStart(2, '0')}月 排班表`;

		calendarGrid.innerHTML = '';
		schedulingData = {};
		conflictMap = {};

		Promise.all([
			loadHoliday(y),
			loadScheduling(y, m)
		]).then(() => buildCalendar(y, m));
	}

	function buildCalendar(y, m) {
		const first = new Date(y, m - 1, 1).getDay();
		const days  = new Date(y, m, 0).getDate();
		const prev  = new Date(y, m - 1, 0).getDate();

		for (let i = 0; i < 42; i++) {
			let d, cur = true;
			if (i < first) {
				d = new Date(y, m - 2, prev - (first - i - 1));
				cur = false;
			} else if (i >= first + days) {
				d = new Date(y, m, i - (first + days) + 1);
				cur = false;
			} else {
				d = new Date(y, m - 1, i - first + 1);
			}
			calendarGrid.appendChild(createCell(d, cur));
		}
	}

	function createCell(d, cur) {
		const y = d.getFullYear();
		const m = String(d.getMonth() + 1).padStart(2, '0');
		const da = String(d.getDate()).padStart(2, '0');
		const ds = `${y}-${m}-${da}`;

		const c = document.createElement('div');
		c.className = 'calendar-cell';
		if (!cur) c.style.opacity = '0.4';
		if (d.getDay() === 0 || d.getDay() === 6) c.classList.add('weekend');
		if (holidayMap[ds]) {
			c.classList.add('holiday');
			const hl = document.createElement('div');
			hl.className = 'holiday-label';
			hl.textContent = holidayMap[ds];
			c.appendChild(hl);
		}

		const h = document.createElement('div');
		h.className = 'date-header';
		h.innerHTML = `
	    <span class="solar-date">${d.getDate()}</span>
	    <span class="lunar-date">${getLunar(d)}</span>
	`;
		c.appendChild(h);

		const list = document.createElement('div');
		list.className = 'schedule-list';
		(schedulingData[ds] || []).forEach(r => {
			list.appendChild(createItem(r));
		});
		c.appendChild(list);

		c.onclick = e => {
			if (e.target.closest('.schedule-item')) return;
			openAdd(ds);
		};

		return c;
	}

	function createItem(r) {
		const i = document.createElement('div');
		i.className = 'schedule-item';

		// ⭐ 依職稱加 class
		if (r.jobtitle_id) {
			i.classList.add(`role-${r.jobtitle_id}`);
		}

		i.textContent = `${r.shift_name} - ${r.staff_name}`;

		const key = `${r.work_date}_${r.staff_id}`;
		if (conflictMap[key] > 1) {
			i.classList.add('conflict');
			i.title = '同一人同日多筆排班';
		}

		i.onclick = e => {
			e.stopPropagation();
			openEdit(r);
		};

		i.oncontextmenu = e => {
			e.preventDefault();
			cancel(r.id);
		};

		i.ontouchstart = () => {
			longPressTimer = setTimeout(() => cancel(r.id), 600);
		};
		i.ontouchend = () => clearTimeout(longPressTimer);

		return i;
	}

	function openAdd(ds) {
		editingId = null;
		scheduleForm.reset();
		work_date.value = ds;
		dlgTitle.textContent = '新增排班';
		scheduleDialog.showModal();
	}

	function openEdit(r) {
		editingId = r.id;
		dlgTitle.textContent = '編輯排班';
		work_date.value  = r.work_date;
		staff_id.value   = r.staff_id;
		// staff_role.value = r.staff_role;
		staff_role.value = '0';
		syncStaffRoleByStaffId(r.staff_id);
		shift_id.value   = r.shift_id;
		start_time.value = r.start_time;
		end_time.value   = r.end_time;
		remark.value     = r.remark || '';
		scheduleDialog.showModal();
	}

	function bindForm() {
		scheduleForm.onsubmit = e => {
			e.preventDefault();
			const p = {
				work_date: work_date.value,
				staff_id: staff_id.value,
				staff_role: staff_role.value,
				shift_id: shift_id.value,
				start_time: start_time.value,
				end_time: end_time.value,
				remark: remark.value
			};
			let url = '/api/scheduling/create_scheduling.php';
			if (editingId) {
				p.id = editingId;
				url = '/api/scheduling/update_scheduling.php';
			}
			fetch(url, {
				method: 'POST',
				headers: {'Content-Type':'application/json'},
				body: JSON.stringify(p)
			}).then(r=>r.json()).then(res=>{
				if(!res.result) return alert(res.msg);
				scheduleDialog.close();
				renderCalendar();
			});
		};
	}

	function cancel(id) {
		if (!confirm('確定取消此排班？')) return;
		fetch('/api/scheduling/del_scheduling.php', {
			method:'POST',
			headers:{'Content-Type':'application/json'},
			body:JSON.stringify({id})
		}).then(r=>r.json()).then(res=>{
			if(!res.result) return alert(res.msg);
			renderCalendar();
		});
	}

	function loadScheduling(y,m){
		return fetch(`/api/scheduling/get_scheduling.php?month=${y}-${String(m).padStart(2,'0')}`)
			.then(r=>r.json())
			.then(res=>{
				if(!res.result) return;
				res.data.records.forEach(r=>{
					if(!schedulingData[r.work_date]) schedulingData[r.work_date]=[];
					schedulingData[r.work_date].push(r);
					const k = `${r.work_date}_${r.staff_id}`;
					conflictMap[k]=(conflictMap[k]||0)+1;
				});
			});
	}

	function getLunar(d){
		if(!window.solarlunar) return '';
		const i=solarlunar.solar2lunar(d.getFullYear(),d.getMonth()+1,d.getDate());
		return i.lDay===1?i.monthCn:i.dayCn;
	}

	/* ===== CSV ===== */
	window.exportCSV = function(){
		const y=currentMonthDate.getFullYear();
		const m=String(currentMonthDate.getMonth()+1).padStart(2,'0');
		location.href=`/api/scheduling/export_scheduling_csv.php?month=${y}-${m}`;
	};

	window.importCSV = function(input){
		const f=input.files[0];
		if(!f) return;
		const fd=new FormData();
		fd.append('file',f);
		fetch('/api/scheduling/import_scheduling_csv.php',{
			method:'POST',
			body:fd
		}).then(r=>r.json()).then(res=>{
			alert(res.msg);
			if(res.result) renderCalendar();
		});
	};

})();

