<?php
// api/get_practice_logs.php
// 取得當前登入使用者的中文輸入練習成績記錄（供「成績」浮窗使用）

session_start();

header('Content-Type: application/json; charset=utf-8');

// 與專案一致的 DB / session 引用
require_once __DIR__ . '/db_connect.php';
require_once __DIR__ . '/log_helper.php';

// 1. 檢查登入
if (!isset($_SESSION['user'])) {
    http_response_code(401);
    echo json_encode([
        'result' => false,
        'data'   => [],
        'msg'    => '未授權，請先登入'
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

$currentUser = $_SESSION['user'];
$currentUserId   = $currentUser['id'];
$currentUsername = $currentUser['username'] ?? '';
$currentGrade    = $currentUser['grade'] ?? null;

// 2. 讀取查詢區間參數（配合前端 typing.js）
$range = $_GET['range'] ?? 'today';  // today / week / month / custom
$start = $_GET['start'] ?? '';       // YYYY-MM-DD
$end   = $_GET['end'] ?? '';         // YYYY-MM-DD

try {
    // 🔸 本地時區：使用 db_connect.php 中設定的時區
    $localTzName = date_default_timezone_get();          // 例如 Asia/Taipei
    $tzLocal     = new DateTimeZone($localTzName);
    $tzUtc       = new DateTimeZone('UTC');

    // 2-1. 先用「本地時間」算區間
    switch ($range) {
        case 'week':
            // 本週一 ~ 本週日（依本地時間）
            $startLocal = new DateTime('today', $tzLocal);
            $weekday    = (int)$startLocal->format('N'); // 1=Mon..7=Sun
            $startLocal->modify('-' . ($weekday - 1) . ' day');
            $endLocal = clone $startLocal;
            $endLocal->modify('+6 day');
            break;

        case 'month':
            // 本月 1 號 ~ 本月最後一天（本地時間）
            $startLocal = new DateTime('first day of this month', $tzLocal);
            $endLocal   = new DateTime('last day of this month', $tzLocal);
            break;

        case 'custom':
            if (empty($start) || empty($end)) {
                echo json_encode([
                    'result' => false,
                    'data'   => [],
                    'msg'    => '自定區間需指定起訖日期'
                ], JSON_UNESCAPED_UNICODE);
                exit;
            }
            $startLocal = new DateTime($start, $tzLocal);
            $endLocal   = new DateTime($end,   $tzLocal);
            break;

        case 'today':
        default:
            $startLocal = new DateTime('today', $tzLocal);
            $endLocal   = new DateTime('today', $tzLocal);
            break;
    }

    // 一天區間定為 00:00:00 ~ 23:59:59（本地時間）
    $startLocal->setTime(0, 0, 0);
    $endLocal->setTime(23, 59, 59);

    // 2-2. 轉成 UTC，因為 DB 的 start_time / end_time 是 UTC
    $startUtc = clone $startLocal;
    $startUtc->setTimezone($tzUtc);
    $endUtc = clone $endLocal;
    $endUtc->setTimezone($tzUtc);

    $startStr = $startUtc->format('Y-m-d H:i:s');
    $endStr   = $endUtc->format('Y-m-d H:i:s');

    // 3. 查詢 DB（DB 內的 start_time / end_time 是 UTC）
    $pdo = get_db_connection();

    // ★ 表名與欄位請確認有照你實際建立的來：
    //    這裡假設是 typing_practice_logs，欄位：
    //    id, user_id, start_time, end_time, duration_seconds,
    //    practice_mode, grade, target_chars, target_seconds,
    //    total_chars, error_chars, correct_chars, wpm, finish_reason ...
    $sql = "
        SELECT
            start_time,
            end_time,
            duration_seconds,
            practice_mode,
            grade,
            CASE
                WHEN practice_mode = 'measured' THEN target_chars
                WHEN practice_mode = 'timed'    THEN target_seconds
                ELSE NULL
            END AS target_value,
            question_text,
            input_text,
            total_chars,
            error_chars,
            wpm
        FROM typing_practice_logs
        WHERE user_id = :uid
          AND start_time BETWEEN :st AND :ed
        ORDER BY start_time ASC
    ";

    $stmt = $pdo->prepare($sql);
    $stmt->execute([
        ':uid' => $currentUserId,
        ':st'  => $startStr,
        ':ed'  => $endStr,
    ]);

    $rowsUtc = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // 4. 將 DB 裡的 UTC 時間轉回「本地時間」後再回傳給前端
    $rows = [];
    foreach ($rowsUtc as $row) {
        // 轉 start_time
        $startLocalStr = '';
        if (!empty($row['start_time'])) {
            $dtStartUtc = new DateTime($row['start_time'], $tzUtc);
            $dtStartUtc->setTimezone($tzLocal);
            $startLocalStr = $dtStartUtc->format('Y-m-d H:i:s');
        }

        // 轉 end_time
        $endLocalStr = '';
        if (!empty($row['end_time'])) {
            $dtEndUtc = new DateTime($row['end_time'], $tzUtc);
            $dtEndUtc->setTimezone($tzLocal);
            $endLocalStr = $dtEndUtc->format('Y-m-d H:i:s');
        }

        $rows[] = [
            'start_time'       => $startLocalStr,                                  // ✅ 已轉成本地時間
            'end_time'         => $endLocalStr,                                    // ✅ 已轉成本地時間
            'duration_seconds' => isset($row['duration_seconds']) ? (int)$row['duration_seconds'] : 0,
            'practice_mode'    => $row['practice_mode'],
            'grade'            => isset($row['grade']) ? (int)$row['grade'] : null,
            'target_value'     => isset($row['target_value']) ? (int)$row['target_value'] : null,
            'question_text'    => $row['question_text'],
            'input_text'       => $row['input_text'],
            'total_chars'      => isset($row['total_chars']) ? (int)$row['total_chars'] : 0,
            'error_chars'      => isset($row['error_chars']) ? (int)$row['error_chars'] : 0,
            'wpm'              => isset($row['wpm']) ? (float)$row['wpm'] : 0.0,
        ];
    }

    echo json_encode([
        'result' => true,
        'data'   => $rows,
        'msg'    => ''
    ], JSON_UNESCAPED_UNICODE);

} catch (\Throwable $e) {
    http_response_code(500);
    echo json_encode([
        'result' => false,
        'data'   => [],
        'msg'    => '伺服器錯誤：' . $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
}

