<?php
// api/practice_log.php
// 接收「中文練習」成績並寫入 typing_practice_logs
session_start();
header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/db_connect.php';
require_once __DIR__ . '/log_helper.php';

// 1. 必須登入
if (!isset($_SESSION['user'])) {
    http_response_code(401);
    echo json_encode([
        'result'  => false,
        'message' => '未授權，請先登入'
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

// 2. 只接受 POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'result'  => false,
        'message' => '只接受 POST 請求'
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

// 3. 讀取 JSON
$raw  = file_get_contents('php://input');
$data = json_decode($raw, true);

// 除錯用：可以視需要註解掉
file_put_contents("/tmp/typing.log", "PRACTICE_RAW: $raw\n", FILE_APPEND);

if (!is_array($data)) {
    http_response_code(400);
    echo json_encode([
        'result'  => false,
        'message' => '請傳入 JSON 格式資料'
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

$user = $_SESSION['user'];

// ---- 取出前端傳入欄位 ----
$practiceMode  = $data['practice_mode'] ?? null;          // 'measured' / 'timed'
$targetChars   = isset($data['target_chars'])   ? (int)$data['target_chars']   : null;
$targetSeconds = isset($data['target_seconds']) ? (int)$data['target_seconds'] : null;

// 前端送的是 epoch 秒數（UNIX timestamp，無時區概念）
$startTs = isset($data['start_time']) ? (int)$data['start_time'] : null;
$endTs   = isset($data['end_time'])   ? (int)$data['end_time']   : null;

// ✅ 一律轉成 UTC 字串存入 DB
// gmdate 會以 UTC 為基準，不受 date_default_timezone_set 影響
if ($startTs) {
    $startTimeUtc = gmdate('Y-m-d H:i:s', $startTs);
} else {
    $startTimeUtc = gmdate('Y-m-d H:i:s');
}
if ($endTs) {
    $endTimeUtc = gmdate('Y-m-d H:i:s', $endTs);
} else {
    $endTimeUtc = $startTimeUtc;
}

$duration     = isset($data['duration_seconds']) ? max(0, (int)$data['duration_seconds']) : 0;
$questionText = $data['question_text'] ?? '';
$inputText    = $data['input_text'] ?? '';

$totalChars   = isset($data['total_chars'])   ? max(0, (int)$data['total_chars'])   : 0;
$errorChars   = isset($data['error_chars'])   ? max(0, (int)$data['error_chars'])   : 0;
$correctChars = isset($data['correct_chars']) ? max(0, (int)$data['correct_chars']) : max(0, $totalChars - $errorChars);

$wpm          = isset($data['wpm']) ? (float)$data['wpm'] : 0.0;
$finishReason = $data['finish_reason'] ?? 'unknown';

// 練習程度（年級）由前端傳 level
$level = isset($data['level']) ? (int)$data['level'] : null;

try {
    $pdo = get_db_connection();

    $stmt = $pdo->prepare("
        INSERT INTO typing_practice_logs
        (
            user_id,
            username,
            grade,
            practice_mode,
            target_chars,
            target_seconds,
            start_time,
            end_time,
            duration_seconds,
            question_text,
            input_text,
            total_chars,
            correct_chars,
            error_chars,
            wpm,
            finish_reason
        )
        VALUES
        (
            :user_id,
            :username,
            :grade,
            :practice_mode,
            :target_chars,
            :target_seconds,
            :start_time,
            :end_time,
            :duration_seconds,
            :question_text,
            :input_text,
            :total_chars,
            :correct_chars,
            :error_chars,
            :wpm,
            :finish_reason
        )
    ");

    $stmt->execute([
        ':user_id'          => $user['id'],
        ':username'         => $user['username'],
        // 原本是用 $user['grade']，現在改用前端傳來的 level
        ':grade'            => $level,
        ':practice_mode'    => $practiceMode,
        ':target_chars'     => $targetChars,
        ':target_seconds'   => $targetSeconds,
        ':start_time'       => $startTimeUtc,   // ✅ 存 UTC
        ':end_time'         => $endTimeUtc,     // ✅ 存 UTC
        ':duration_seconds' => $duration,
        ':question_text'    => $questionText,
        ':input_text'       => $inputText,
        ':total_chars'      => $totalChars,
        ':correct_chars'    => $correctChars,
        ':error_chars'      => $errorChars,
        ':wpm'              => $wpm,
        ':finish_reason'    => $finishReason,
    ]);

    $logId = $pdo->lastInsertId();

    // 寫入操作紀錄（可選）
    if (function_exists('log_operation')) {
        log_operation($pdo, $user['id'], $user['username'], 'PRACTICE_LOG_CREATED');
    }

    echo json_encode([
        'result'  => true,
        'log_id'  => $logId,
        'message' => '練習成績已記錄'
    ], JSON_UNESCAPED_UNICODE);

} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode([
        'result'  => false,
        'message' => '寫入練習成績失敗：' . $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
}

