document.addEventListener('DOMContentLoaded', () => {
    const accountInput = document.getElementById('account');
    const codeDisplay  = document.getElementById('code-display');
    const timerEl      = document.getElementById('timer');
    const codeInput    = document.getElementById('verification-code');
    const resetBtn     = document.getElementById('reset-code-btn');
    const passwordGroup= document.getElementById('password-group');
    const passwordInput= document.getElementById('password');
    const messageArea  = document.getElementById('message-area');
    const loginForm    = document.getElementById('login-form');
    const loginBtn     = document.getElementById('login-button');

    let timerId = null;
    let secondsLeft = 60;
    let inPasswordStep = false;

    function setMessage(msg, isError = false) {
        messageArea.textContent = msg || '';
        messageArea.className = isError ? 'error' : 'success';
    }

    function startTimer() {
        if (timerId) clearInterval(timerId);
        secondsLeft = 60;
        timerEl.textContent = `驗證碼將於 ${secondsLeft} 秒後失效`;
        timerId = setInterval(() => {
            secondsLeft--;
            if (secondsLeft <= 0) {
                clearInterval(timerId);
                timerId = null;
                timerEl.textContent = '驗證碼已失效，請按「重置驗證碼」重新取得';
                resetBtn.disabled = false;
            } else {
                timerEl.textContent = `驗證碼將於 ${secondsLeft} 秒後失效`;
            }
        }, 1000);
    }

async function fetchCode() {
    try {
        resetBtn.disabled = true;
        setMessage('');

        // ✅ 若你的 generate-code.php 不在同一層，這行路徑要一起改
        const res = await fetch('/api/generate-code.php', {
            cache: 'no-store'
        });

        if (!res.ok) {
            const txt = await res.text().catch(() => '');
            console.error('generate-code.php HTTP 錯誤：', res.status, txt);
            setMessage('無法取得驗證碼（HTTP ' + res.status + '）', true);
            return;
        }

        const text = await res.text();
        let data;
        try {
            data = JSON.parse(text);
        } catch (e) {
            console.error('generate-code.php 回傳的不是 JSON：', text);
            setMessage('驗證碼回應格式錯誤', true);
            return;
        }

        if (!data.code) {
            console.error('generate-code.php JSON 內沒有 code 欄位：', data);
            setMessage('驗證碼回應內容錯誤', true);
            return;
        }

        codeDisplay.textContent = data.code;
        startTimer();
    } catch (err) {
        console.error('fetchCode 發生錯誤：', err);
        setMessage('無法取得驗證碼（連線錯誤）', true);
    } finally {
        // 只有在驗證碼失效時才開啟重置按鈕，這裡先維持鎖定
        // resetBtn.disabled = false; // 若你想測試，可暫時打開
    }
}

/*
    async function fetchCode() {
        try {
            resetBtn.disabled = true;
            setMessage('');
            const res = await fetch('generate-code.php');
            if (!res.ok) throw new Error('無法取得驗證碼');
            const data = await res.json();
            codeDisplay.textContent = data.code || '.....';
            startTimer();
        } catch (err) {
            console.error(err);
            setMessage('無法取得驗證碼，請重新整理頁面', true);
        }
    }
*/

    resetBtn.addEventListener('click', () => {
        if (timerId) clearInterval(timerId);
        fetchCode();
    });

    // 第一階段：帳號 + 驗證碼
    async function doFirstStepLogin() {
        const account = accountInput.value.trim();
        const code    = codeInput.value.trim();

        if (!account || !code) {
            setMessage('請輸入帳號與 5 碼驗證碼', true);
            return;
        }
        if (code.length !== 5) {
            setMessage('驗證碼需為 5 碼數字', true);
            return;
        }

        loginBtn.disabled = true;
        loginBtn.textContent = '登入中...';

        try {
            const res = await fetch('/api/login.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ account, code })
            });

            const data = await res.json().catch(() => ({}));

            if (!res.ok || !data.success) {
                setMessage(data.message || '登入失敗', true);
                // 若是驗證碼錯誤或已失效，建議重新取得
                if (data.message && data.message.includes('驗證碼')) {
                    if (timerId) clearInterval(timerId);
                    resetBtn.disabled = false;
                }
                return;
            }

            // 學生：直接導向 dashboard
            if (!data.needPassword) {
                window.location.href = 'dashboard.html';
                return;
            }

            // 管理者／教師：進入第二階段密碼輸入
            inPasswordStep = true;
            passwordGroup.style.display = 'block';
            passwordInput.value = '';
            passwordInput.focus();

            // 第一階段完成後，不再允許改帳號與重取驗證碼
            accountInput.disabled = true;
            codeInput.disabled    = true;
            resetBtn.disabled     = true;

            if (timerId) {
                clearInterval(timerId);
                timerId = null;
            }
            timerEl.textContent = '驗證碼驗證成功，請輸入密碼完成登入';

            loginBtn.textContent = '送出密碼';

            setMessage(`帳號 ${data.username}（${data.role === 'admin' ? '管理者' : '教師'}），請輸入密碼`);
        } catch (err) {
            console.error(err);
            setMessage('登入過程發生錯誤，請稍後再試', true);
        } finally {
            loginBtn.disabled = false;
        }
    }

    // 第二階段：輸入密碼
    async function doPasswordLogin() {
        const pwd = passwordInput.value.trim();
        if (!pwd) {
            setMessage('請輸入密碼', true);
            return;
        }

        loginBtn.disabled = true;
        loginBtn.textContent = '驗證中...';

        try {
            const res = await fetch('/api/verify_password.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ password: pwd })
            });

            const data = await res.json().catch(() => ({}));

            if (!res.ok || !data.success) {
                setMessage(data.message || '密碼驗證失敗', true);
                return;
            }

            setMessage('登入成功，即將進入系統');
            window.location.href = 'dashboard.html';
        } catch (err) {
            console.error(err);
            setMessage('密碼驗證過程發生錯誤，請稍後再試', true);
        } finally {
            loginBtn.disabled = false;
            loginBtn.textContent = '送出密碼';
        }
    }

    loginForm.addEventListener('submit', (e) => {
        e.preventDefault();
        if (inPasswordStep) {
            doPasswordLogin();
        } else {
            doFirstStepLogin();
        }
    });

    // 進頁面就取得一次驗證碼
    fetchCode();
});

