// 題庫管理頁面的 JS 邏輯
(function() {
	// --- Global Variables & Constants ---
	const CURRENT_ROLE = window.currentUserRole;
	const CURRENT_USER_ID = window.currentUserId;
	const MAX_LENGTH = 1500;
	let ALL_QUESTION_TYPES = []; // Cache for question types {id, type_name}

	// --- DOM Elements ---
	const addEditModal = document.getElementById("add-edit-question-modal");
	const showAddBtn = document.getElementById("show-add-question-btn");
	const closeModalBtn = document.getElementById("close-question-modal");
	const form = document.getElementById("question-form");
	const formTitle = document.getElementById("form-title");
	const questionIdInput = document.getElementById("question-id");
	const nameInput = document.getElementById("question-name");
	const typeSelect = document.getElementById("question-type"); // Question type dropdown
	const contentTextarea = document.getElementById("question-content");
	const levelSelect = document.getElementById("question-level");
	const submitBtn = document.getElementById("submit-btn"); // Question submit
	const cancelBtn = document.getElementById("cancel-btn"); // Question form cancel
	const formMessage = document.getElementById("form-message"); // Inside add/edit question modal
	const charCountInfo = document.getElementById("char-count-info");
	const showTypeMaintenanceBtn = document.getElementById("show-type-maintenance-btn");
	const typeMaintenanceModal = document.getElementById("type-maintenance-modal");
	const closeTypeModalBtn = document.getElementById("close-type-modal");
	const typeForm = document.getElementById("type-form");
	const typeIdInput = document.getElementById("type-id");
	const typeNameInput = document.getElementById("type-name-input");
	const typeSubmitBtn = document.getElementById("type-submit-btn"); // Type submit
	const typeFormMessage = document.getElementById("type-form-message"); // Inside type modal
	const typeTableBody = document.getElementById("type-table-body"); // Target table body
	const typeListMessage = document.getElementById("type-list-message"); // Below type table
	const tableBody = document.getElementById("question-table-body");
	const tableMessage = document.getElementById("table-message"); // Below question table

	// --- Element Existence Check ---
	if (!addEditModal || !showAddBtn || !closeModalBtn || !form || !tableBody || !contentTextarea || !charCountInfo || !nameInput || !typeSelect ||
		!showTypeMaintenanceBtn || !typeMaintenanceModal || !closeTypeModalBtn || !typeForm || !typeTableBody || !typeSubmitBtn ) {
		console.error("Testbank management: Missing required DOM elements."); return;
	}

	// --- Utility Functions ---
	function showMessage(text, isError = false, target = formMessage) { if (!target) { console.warn("showMessage: Target element not found for message:", text); return; } target.textContent = text; target.className = isError ? 'form-message-area error' : 'form-message-area success'; if (target === typeListMessage || target === tableMessage || target === typeFormMessage) { target.style.whiteSpace = 'pre-wrap'; } }
	function escapeHTML(str) { if (str === null || str === undefined) return ''; return str.toString().replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/"/g, '&quot;').replace(/'/g, '&#39;'); }
		function formatDateTime(isoString) { if (!isoString) return 'N/A'; try { const date = new Date(isoString); return date.toLocaleString('sv-SE', { year: 'numeric', month: '2-digit', day: '2-digit', hour: '2-digit', minute: '2-digit' }); } catch (e) { return isoString; } }

		// --- Modal Control Functions ---
		function openModal(modalElement) { if (modalElement) modalElement.style.display = 'flex'; }
		function closeModal(modalElement) { if (modalElement) modalElement.style.display = 'none'; if (modalElement === addEditModal) resetForm(); if (modalElement === typeMaintenanceModal) resetTypeForm(); }

		// --- Core Data Loading and Rendering ---
		async function loadQuestions() { showMessage('', false, tableMessage); if (!tableBody) return; tableBody.innerHTML = '<tr><td colspan="8">載入中...</td></tr>'; console.log("[loadQuestions] Attempting to fetch questions..."); try { const response = await fetch('/api/testbank.php'); console.log("[loadQuestions] Fetch response status:", response.status); if (!response.ok) { const errText = await response.text(); console.error("[loadQuestions] API Error Response Text:", errText); let errMsg = `無法載入題庫列表 (HTTP ${response.status})`; try { errMsg = JSON.parse(errText).message || errMsg; } catch(e){} throw new Error(errMsg); } const contentType = response.headers.get("content-type"); if (!contentType || !contentType.includes("application/json")) { const errText = await response.text(); console.error("[loadQuestions] Non-JSON response received:", errText); throw new Error(`伺服器回應格式錯誤 (${contentType})`); } const questions = await response.json(); console.log("[loadQuestions] Received questions data (raw):", JSON.stringify(questions)); if (!Array.isArray(questions)) { throw new Error("API 回應的資料不是有效的陣列"); } renderTable(questions); console.log("[loadQuestions] Finished rendering table."); } catch (err) { console.error("[loadQuestions] Error during fetch or render:", err); tableBody.innerHTML = '<tr><td colspan="8" class="error">載入失敗，請檢查主控台錯誤訊息。</td></tr>'; showMessage('載入題庫列表失敗: ' + err.message, true, tableMessage); } }
		function renderTable(questions) { if (!tableBody) return; tableBody.innerHTML = ''; if (!Array.isArray(questions) || questions.length === 0) { tableBody.innerHTML = '<tr><td colspan="8">尚無任何題目</td></tr>'; return; } console.log("[renderTable] Rendering questions count:", questions.length); questions.forEach((q, index) => { try { if (typeof q !== 'object' || q === null) { throw new Error(`項目 ${index} 不是有效的物件`); } const tr = document.createElement('tr'); tr.dataset.question = JSON.stringify(q); const qName = q.name ?? ''; const qTypeName = q.type_name ?? '(未分類)'; const qContent = q.content ?? ''; const qLevel = q.level ?? 'N/A'; const qLength = q.length ?? 0; const qCreator = q.creator_username ?? q.created_by ?? 'N/A'; const qUpdateAt = q.update_at; const preview = qContent.length > 30 ? qContent.substring(0, 30) + '...' : qContent; tr.innerHTML = `<td>${escapeHTML(qName)}</td><td>${escapeHTML(qTypeName)}</td><td>${escapeHTML(preview)}</td><td>${qLevel}</td><td>${qLength}</td><td>${escapeHTML(qCreator)}</td><td>${formatDateTime(qUpdateAt)}</td><td class="actions"> <button class="edit-btn">編輯</button> <button class="delete-btn">刪除</button> </td>`; const editBtn = tr.querySelector('.edit-btn'); const deleteBtn = tr.querySelector('.delete-btn'); if (CURRENT_ROLE === 'teacher' && (!q.hasOwnProperty('created_by') || q.created_by !== CURRENT_USER_ID)) { editBtn?.remove(); deleteBtn?.remove(); } tableBody.appendChild(tr); } catch (renderError) { console.error(`[renderTable] Error rendering question row at index ${index}:`, q, renderError); const errTr = document.createElement('tr'); errTr.innerHTML = `<td colspan="8" class="error">渲染此列時發生錯誤: ${escapeHTML(renderError.message)}</td>`; tableBody.appendChild(errTr); } }); }

		// --- Question Form Handling ---
		function resetForm() { form.reset(); questionIdInput.value = ''; formTitle.textContent = '新增題目'; submitBtn.textContent = '儲存'; cancelBtn.style.display = 'none'; showMessage(''); updateCharCount(); typeSelect.value = ''; }
		function populateForm(q) { resetForm(); formTitle.textContent = '編輯題目'; submitBtn.textContent = '更新'; cancelBtn.style.display = 'inline-block'; questionIdInput.value = q.id; nameInput.value = q.name; typeSelect.value = q.type_id || ''; contentTextarea.value = q.content; levelSelect.value = q.level; updateCharCount(); }
		function updateCharCount() { const currentLength = contentTextarea.value.trim().length; charCountInfo.textContent = `字數: ${currentLength} / ${MAX_LENGTH}`; const nameIsFilled = nameInput.value.trim() !== ''; if (currentLength > MAX_LENGTH || !nameIsFilled) { charCountInfo.classList.toggle('error', currentLength > MAX_LENGTH); submitBtn.disabled = true; } else { charCountInfo.classList.remove('error'); submitBtn.disabled = false; } }
		nameInput.addEventListener('input', updateCharCount);
		async function handleFormSubmit(e) { e.preventDefault(); if (contentTextarea.value.trim().length > MAX_LENGTH) { showMessage(`題目內容不得超過 ${MAX_LENGTH} 字`, true); return; } if (nameInput.value.trim() === '') { showMessage('題目名稱不得為空', true); return; } showMessage(''); submitBtn.disabled = true; const id = questionIdInput.value; const isUpdate = !!id; const data = { id: id, name: nameInput.value, type_id: typeSelect.value || null, content: contentTextarea.value, level: levelSelect.value }; let url = '/api/testbank.php'; let method = isUpdate ? 'PUT' : 'POST'; try { const response = await fetch(url, { method: method, headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(data) }); const result = await response.json(); if (!response.ok) throw new Error(result.message || (isUpdate ? '更新失敗' : '新增失敗')); showMessage(isUpdate ? '更新成功' : '新增成功', false); setTimeout(() => { closeModal(addEditModal); loadQuestions(); }, 1000); } catch (err) { console.error("Error submitting question form:", err); showMessage(err.message, true); updateCharCount(); } }

		// (新) Handle deleting a question (Complete block)
		async function handleDelete(e) {
			const deleteButton = e.target;
			const tr = deleteButton.closest('tr');
			if (!tr || !tr.dataset.question) return;

			const q = JSON.parse(tr.dataset.question);

			if (!confirm(`確定要刪除題目 "${escapeHTML(q.name || '未命名')}" 嗎？`)) {
				return;
			}

			deleteButton.disabled = true;
			let finalErrorMessage = '刪除失敗 (未知錯誤)';

			try {
				const response = await fetch(`/api/testbank.php?id=${q.id}`, { method: 'DELETE' });

				// Robust response handling
				let responseBody = null;
				let responseOk = response.ok;
				try { responseBody = await response.json(); }
				catch (jsonError) {
					console.warn("Failed to parse delete response as JSON, trying text.", jsonError);
					try {
						if (typeof response.text === 'function') { responseBody = await response.text(); }
						else { responseBody = `伺服器錯誤 (Status: ${response.status} ${response.statusText})`; }
					} catch(textError){
						console.error("Failed to get delete response text either.", textError);
						responseBody = `伺服器錯誤 (Status: ${response.status} ${response.statusText})`;
					}
				}

				if (!responseOk) {
					let apiMessage = (typeof responseBody === 'string') ? responseBody : (responseBody?.message || finalErrorMessage);
					// Check for "in use" error specifically for questions
					if (apiMessage && (apiMessage.includes('已被用於測驗中') || apiMessage.includes('in use'))) {
						finalErrorMessage = '無法刪除：此題目已被用於測驗中。';
					} else {
						finalErrorMessage = apiMessage;
					}
					throw new Error(finalErrorMessage);
				}

				// --- Success ---
				alert(`題目 "${escapeHTML(q.name || '未命名')}" 刪除成功`);
				tr.remove(); // Remove row on success

			} catch (err) {
				// --- Error Handling ---
				console.error("Error deleting question:", err);
				// Show error in the main area below the table
				showMessage('刪除失敗：' + err.message, true, tableMessage);
				deleteButton.disabled = false; // Re-enable button on error
			}
			// No finally block needed
		}
		// Handle clicks within the question table
		function handleTableClick(e) { if (e.target.classList.contains('edit-btn')) { const tr = e.target.closest('tr'); if (tr && tr.dataset.question) { const q = JSON.parse(tr.dataset.question); populateForm(q); openModal(addEditModal); } } if (e.target.classList.contains('delete-btn')) { handleDelete(e); } }


		// --- Type Maintenance Functions ---
		// Load types
		async function loadTypes() { showMessage('', false, typeListMessage); console.log("[loadTypes] Attempting to fetch types..."); try { const response = await fetch('/api/question_types.php'); console.log("[loadTypes] Fetch response status:", response.status); if (!response.ok) { const errText = await response.text(); console.error("[loadTypes] API Error Response Text:", errText); let errMsg = `無法載入類型列表 (HTTP ${response.status})`; try { errMsg = JSON.parse(errText).message || errMsg; } catch(e){} throw new Error(errMsg); } const contentType = response.headers.get("content-type"); if (!contentType || !contentType.includes("application/json")) { const errText = await response.text(); console.error("[loadTypes] Non-JSON response received:", errText); throw new Error(`伺服器回應格式錯誤 (${contentType})`); } ALL_QUESTION_TYPES = await response.json(); if (!Array.isArray(ALL_QUESTION_TYPES)) { console.error("[loadTypes] API did not return a valid array:", ALL_QUESTION_TYPES); throw new Error("API 回應的類型資料格式不正確"); } console.log("[loadTypes] Received types data:", ALL_QUESTION_TYPES); renderTypeList(ALL_QUESTION_TYPES); populateQuestionTypeDropdown(ALL_QUESTION_TYPES); console.log("[loadTypes] Finished rendering type list and dropdown."); } catch (err) { console.error("[loadTypes] Error during fetch or render:", err); showMessage('載入類型列表失敗: ' + err.message, true, typeListMessage); throw err; } }
		// Render the type list TABLE
		function renderTypeList(types) { if (!typeTableBody) return; typeTableBody.innerHTML = ''; if (!Array.isArray(types) || types.length === 0) { typeTableBody.innerHTML = '<tr><td colspan="3">尚無自訂類型</td></tr>'; return; } console.log("[renderTypeList] Rendering types count:", types.length); types.forEach((type, index) => { if (typeof type !== 'object' || type === null || typeof type.id === 'undefined' || typeof type.type_name === 'undefined') { console.error(`[renderTypeList] Invalid type data at index ${index}:`, type); const errTr = document.createElement('tr'); errTr.innerHTML = `<td colspan="3" class="error">錯誤：無效的類型資料</td>`; typeTableBody.appendChild(errTr); return; } const tr = document.createElement('tr'); tr.dataset.type = JSON.stringify(type); tr.innerHTML = `<td>${type.id}</td><td>${escapeHTML(type.type_name)}</td><td class="actions"> <button class="edit-btn small-btn">編輯</button> <button class="delete-btn small-btn">刪除</button> </td>`; typeTableBody.appendChild(tr); }); }
		// Populate the question type dropdown
		function populateQuestionTypeDropdown(types) { if (!typeSelect) return; const currentVal = typeSelect.value; const firstOption = typeSelect.options.length > 0 && typeSelect.options[0].value === '' ? typeSelect.options[0] : null; typeSelect.innerHTML = ''; if (firstOption) { typeSelect.appendChild(firstOption); } else { typeSelect.add(new Option('(未分類)', '')); } if (Array.isArray(types)) { types.forEach(type => { if (typeof type === 'object' && type !== null && type.hasOwnProperty('id') && type.hasOwnProperty('type_name')) { typeSelect.add(new Option(type.type_name, type.id)); } else { console.warn("Invalid type object found while populating dropdown:", type); } }); } else { console.error("Invalid 'types' data received for dropdown:", types); } typeSelect.value = currentVal; if (typeSelect.selectedIndex <= 0 && currentVal !== '') { typeSelect.value = ''; } }
		// Reset the Type Maintenance form
		function resetTypeForm() { typeForm.reset(); typeIdInput.value = ''; typeNameInput.placeholder = '輸入類型名稱...'; typeSubmitBtn.textContent = '+'; typeSubmitBtn.disabled = false; showMessage('', false, typeFormMessage); }
		// Populate the Type Maintenance form for editing
		function populateTypeForm(type) { resetTypeForm(); typeIdInput.value = type.id; typeNameInput.value = type.type_name; typeNameInput.placeholder = ''; typeSubmitBtn.textContent = '確定'; }
		// Handle submission of the Type Maintenance form (Add or Update)
		async function handleTypeFormSubmit(e) { e.preventDefault(); const typeName = typeNameInput.value.trim(); if (!typeName) { showMessage('類型名稱不得為空', true, typeFormMessage); return; } showMessage('', false, typeFormMessage); typeSubmitBtn.disabled = true; const id = typeIdInput.value; const isUpdate = !!id; const data = { type_name: typeName }; let url = '/api/question_types.php'; let method = 'POST'; if (isUpdate) { data.id = id; method = 'PUT'; } try { const response = await fetch(url, { method: method, headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(data) }); const result = await response.json(); if (!response.ok) throw new Error(result.message || (isUpdate ? '更新失敗' : '新增失敗')); showMessage(isUpdate ? '更新成功' : '新增成功', false, typeFormMessage); resetTypeForm(); await loadTypes(); } catch (err) { console.error("Error submitting type form:", err); showMessage(err.message, true, typeFormMessage); } finally { typeSubmitBtn.disabled = false; } }
		// Handle deleting a type - Reload list on failure
		async function handleTypeDelete(buttonElement) {
			const tr = buttonElement.closest('tr'); if (!tr || !tr.dataset.type) return;
			const type = JSON.parse(tr.dataset.type);
			if (!confirm(`確定要刪除類型 "${escapeHTML(type.type_name)}" 嗎？\n(注意：如果尚有題目使用此類型，將無法刪除)`)) { return; }
			buttonElement.disabled = true; let finalErrorMessage = '刪除失敗 (未知錯誤)';
			try {
				const response = await fetch(`/api/question_types.php?id=${type.id}`, { method: 'DELETE' });
				let responseBody = null; let responseOk = response.ok;
				try { responseBody = await response.json(); } catch (e) { try { responseBody = await response.text(); } catch (e2) { responseBody = `伺服器錯誤 ${response.status}`; } }
				if (!responseOk) {
					let apiMessage = (typeof responseBody === 'string') ? responseBody : (responseBody?.message || finalErrorMessage);
					if (apiMessage && (apiMessage.includes('尚有題目使用此類型') || apiMessage.includes('in use'))) { finalErrorMessage = `無法刪除：尚有題目使用類型 "${escapeHTML(type.type_name)}"。`; }
					else { finalErrorMessage = apiMessage; }
					throw new Error(finalErrorMessage);
				}
				showMessage(`類型 "${escapeHTML(type.type_name)}" 刪除成功`, false, typeListMessage); tr.remove();
				await loadTypes(); // Reload dropdown on success
			} catch (err) {
				console.error("Error deleting type:", err); alert('刪除失敗：' + err.message);
				buttonElement.disabled = false; // Re-enable button on error
				// Reload types to ensure list consistency after failed delete
				console.log("[handleTypeDelete] Reloading types after error...");
				try { await loadTypes(); } catch (loadErr) { console.error("[handleTypeDelete] Error reloading types after delete error:", loadErr); showMessage("刪除失敗且列表刷新錯誤，請稍後重試。", true, typeListMessage); }
			}
		}
		// Handle clicks within the type list TABLE
		function handleTypeListClick(e) {
			const editButton = e.target.closest('.edit-btn.small-btn');
			const deleteButton = e.target.closest('.delete-btn.small-btn');
			if (editButton) { const tr = editButton.closest('tr'); if (tr && tr.dataset.type) { const type = JSON.parse(tr.dataset.type); populateTypeForm(type); typeNameInput.focus(); } }
			else if (deleteButton) { handleTypeDelete(deleteButton); } // Pass the button element
		}


		// --- Event Listeners Setup ---
		form.addEventListener('submit', handleFormSubmit);
		tableBody.addEventListener('click', handleTableClick); // Question table
		showAddBtn.addEventListener('click', () => { resetForm(); openModal(addEditModal); });
		closeModalBtn.addEventListener('click', () => closeModal(addEditModal));
		cancelBtn.addEventListener('click', () => closeModal(addEditModal));
		addEditModal.addEventListener('click', (e) => { if (e.target === addEditModal) closeModal(addEditModal); });
		contentTextarea.addEventListener('input', updateCharCount);
		// nameInput listener added

		// Type Maintenance Modal & Form Listeners
		showTypeMaintenanceBtn.addEventListener('click', () => { resetTypeForm(); loadTypes(); openModal(typeMaintenanceModal); });
		closeTypeModalBtn.addEventListener('click', () => { closeModal(typeMaintenanceModal); loadTypes(); /* Reload dropdown */ });
		typeMaintenanceModal.addEventListener('click', (e) => { if (e.target === typeMaintenanceModal) { closeModal(typeMaintenanceModal); loadTypes(); /* Reload dropdown */ } });
		typeForm.addEventListener('submit', handleTypeFormSubmit);
		if (typeTableBody) { typeTableBody.addEventListener('click', handleTypeListClick); } // Type table

		// --- Initial Load ---
		async function init() {
			console.log("Initializing Testbank Management...");
			// === (新增) 檢查 session ===
			if (!window.currentUsername || typeof window.currentUsername !== 'string') {
				alert("登入逾時或尚未登入，請重新登入！");
				fetch('/api/logout.php')
					.then(() => location.href = '/login.html')
					.catch(() => location.href = '/login.html');
				return; // 不再往下初始化
			}

			try {
				await loadTypes();
				console.log("[init] Finished loadTypes successfully.");
				await loadQuestions();
				console.log("Testbank Initialization complete.");
			} catch (initError) {
				console.error("Error during Testbank Initialization:", initError);
				if (tableMessage) showMessage("初始化題庫管理失敗，請重新整理頁面或檢查主控台。\n錯誤：" + initError.message, true, tableMessage);
			}
		}
		init();

	})();
