// 中文練習頁面的 JS 邏輯（配合 dashboard/typing-content.html）
// 功能：計時 / 限時練習 + 練習成績記錄到 api/practice_log.php + 個人成績視窗
(function () {
	// --- 1. 抓 DOM 元件 ---
	const modeMeasuredRadio   = document.getElementById('mode-measured');
	const modeTimedRadio      = document.getElementById('mode-timed');
	const levelSelect         = document.getElementById('level-select');
	const measuredSettings    = document.getElementById('measured-settings');
	const timedSettings       = document.getElementById('timed-settings');
	const charCountSelect     = document.getElementById('char-count-select');
	const timeDurationSelect  = document.getElementById('time-duration-select');
	const fontNoZhuyinRadio   = document.getElementById('font-no-zhuyin');
	const fontZhuyinRadio     = document.getElementById('font-zhuyin');
	const stopTestBtn         = document.getElementById('stop-test-btn');

	const timerDisplay        = document.getElementById('timer-display');
	const charsTypedDisplay   = document.getElementById('chars-typed-display');
	const errorsDisplay       = document.getElementById('errors-display');
	const speedDisplay        = document.getElementById('speed-display');
	const charsTargetDisplay  = document.getElementById('chars-target-display');
	const statusMessage       = document.getElementById('status-message');

	const textDisplayWindow   = document.getElementById('text-display-window'); // 題目顯示區
	const typingDisplay       = document.getElementById('typing-display');      // 已輸入內容顯示區
	const typingInput         = document.getElementById('typing-input');        // 真正輸入框

	const container           = document.querySelector('.typing-practice-container');

	// 成績視窗相關
	const showScoreBtn     = document.getElementById('show-score-btn');
	const scoreDialog      = document.getElementById('score-dialog');
	const scoreCloseBtn    = document.getElementById('score-close-btn');
	const scoreUserLabel   = document.getElementById('score-user-label');
	const scoreRangeRadios = document.querySelectorAll('input[name="score-range"]');
	const scoreStartInput  = document.getElementById('score-start-date');
	const scoreEndInput    = document.getElementById('score-end-date');
	const scoreRefreshBtn  = document.getElementById('score-refresh-btn');
	const scoreTableBody   = document.querySelector('#score-table tbody');
	const scoreChartCanvas = document.getElementById('score-chart');
	const scoreChartCtx    = scoreChartCanvas ? scoreChartCanvas.getContext('2d') : null;

	// 成績明細視窗（請在 HTML 中準備好對應 dialog）
	const scoreDetailDialog = document.getElementById('score-detail-dialog');
	const scoreDetailBody   = scoreDetailDialog ? scoreDetailDialog.querySelector('.score-detail-body') : null;

	if (!modeMeasuredRadio || !modeTimedRadio || !levelSelect || !charCountSelect ||
		!timeDurationSelect || !timerDisplay || !charsTypedDisplay || !errorsDisplay ||
		!speedDisplay || !statusMessage || !textDisplayWindow || !typingDisplay ||
		!typingInput || !stopTestBtn || !container) {

		console.error('Typing JS: 必要元素遺失，無法啟動練習頁面。');
		if (container) {
			container.innerHTML = '<h2 class="error">練習頁面載入錯誤，缺少必要元素。</h2>';
		} else {
			alert('練習頁面載入錯誤...');
		}
		return;
	}

	// --- 2. 狀態變數 ---
	let practiceMode    = 'measured'; // 'measured' 計字數 / 'timed' 限時
	let targetCount     = 50;         // 計字數目標
	let targetTime      = 30;         // 限時秒數
	let targetLevel     = levelSelect.value || '3';

	let useZhuyinFont   = false;

	let questionLines   = []; // 題目內容（已依標點拆行）
	let currentLineIndex = 0; // 題目目前所在行（配合 text-display-window 的 highlight）
	let typedLines      = []; // 使用者已完成的各行內容

	// 成績統計：以「不含 Enter」的字數為準
	let totalCharsTyped   = 0; // 總輸入字數（不含換行）
	let totalErrors       = 0; // 總錯誤字數
	let totalCorrectChars = 0; // 總正確字數

	let startTime       = null; // ms timestamp
	let timeElapsed     = 0;    // 已經過秒數
	let timerInterval   = null;

	let practiceActive     = false; // 是否正在練習
	let practicePrepared   = false; // 是否已設定好，等待點擊輸入框開始
	let practiceFinished   = false; // 是否已結束本輪練習
	let isLoadingQuestions = false;
	let isComposing        = false; // IME 組字中
	let finishReason       = 'unknown';

	const EDIT_KEYS = [
		'Backspace', 'Delete', 'ArrowLeft', 'ArrowRight', 'Home', 'End'
	];
	const IGNORE_KEYS = [
		'Shift', 'Control', 'Alt', 'Meta',
		'CapsLock', 'NumLock', 'ScrollLock',
		'Tab', 'ContextMenu',
		'ArrowUp', 'ArrowDown',
		'PageUp', 'PageDown', 'Insert',
		'Escape', 'F1', 'F2', 'F3', 'F4', 'F5', 'F6', 'F7', 'F8', 'F9', 'F10', 'F11', 'F12'
	];

	// 檢查登入狀態，不存在時登出並導回登入頁
	function checkUsername() {
		if (!window.currentUsername || typeof window.currentUsername !== 'string') {
			alert('登入逾時或尚未登入，請重新登入。');
			try {
				fetch('/api/logout.php').catch(() => {});
			} finally {
				window.location.href = '/login.html';
			}
			return false;
		}
		return true;
	}

	// --- 3. 工具函式（共用） ---
	function showStatusMessage(text, isError = false) {
		statusMessage.textContent = text;
		statusMessage.classList.toggle('error', !!isError);
	}

	// 題目依標點拆行，避免單行過長
	function splitLinesByPunctuation(rawLines) {
		const results = [];
		const punctPattern = /([，。！？；、,.!?;])/;

		rawLines.forEach(line => {
			if (!line) return;
			let buffer = '';
			for (const ch of line) {
				buffer += ch;
				if (punctPattern.test(ch)) {
					results.push(buffer.trim());
					buffer = '';
				}
			}
			if (buffer.trim()) {
				results.push(buffer.trim());
			}
		});

		return results;
	}

	/*
	function getUserFullText() {
		const full = [...typedLines];
		if (typingInput.value) full.push(typingInput.value);
		return full.join('\n');
	}
	*/

	function getUserFullText() {
                // 把「已完成的行」＋「目前輸入中的行」組成 typing-display 的內容
                const currentLine = typingInput.value || '';
                if (!typedLines.length) return currentLine;

                if (currentLine) {
                        return typedLines.join('\n') + '\n' + currentLine;
                }
                return typedLines.join('\n');
        }

	function updateTimerDisplay() {
		const m = Math.floor(timeElapsed / 60);
		const s = timeElapsed % 60;
		timerDisplay.textContent =
			String(m).padStart(2, '0') + ':' + String(s).padStart(2, '0');
	}

	function updateSpeedDisplay() {
		// 以分鐘為單位的打字速度（不含 Enter）
		const minutes = timeElapsed / 60;
		let wpm = 0;
		if (minutes > 0) {
			wpm = totalCharsTyped / minutes;
		}
		speedDisplay.textContent = wpm.toFixed(1);
	}

	function resetScores() {
		totalCharsTyped   = 0;
		totalErrors       = 0;
		totalCorrectChars = 0;

		charsTypedDisplay.textContent = '0';
		errorsDisplay.textContent     = '0';
		speedDisplay.textContent      = '0.0';
		timeElapsed                   = 0;
		updateTimerDisplay();
	}

	function updateModeVisibility() {
		if (practiceMode === 'measured') {
			measuredSettings.style.display = '';
			timedSettings.style.display    = 'none';
			charsTargetDisplay.style.display = '';
			charsTargetDisplay.textContent   = ' / ' + targetCount;
		} else {
			measuredSettings.style.display = 'none';
			timedSettings.style.display    = '';
			charsTargetDisplay.style.display = 'none';
		}
	}

	function readSettings() {
		practiceMode = modeMeasuredRadio.checked ? 'measured' : 'timed';
		targetLevel  = levelSelect.value || '3';

		if (practiceMode === 'measured') {
			targetCount = parseInt(charCountSelect.value, 10) || 50;
		} else {
			targetTime  = parseInt(timeDurationSelect.value, 10) || 30;
		}

		updateModeVisibility();
	}

	function updateTypingInputAvailability(prepared) {
		practicePrepared = prepared;

		if (!prepared) {
			typingInput.disabled = true;
			showStatusMessage('請設定練習模式與條件，然後點擊下方輸入框開始。', false);
		} else {
			typingInput.disabled = false;
			showStatusMessage('請點選下方輸入框，開始載入題目並練習。', false);
		}
	}

	function updateFont() {
		useZhuyinFont = fontZhuyinRadio.checked;
		if (useZhuyinFont) {
			// 啟用注音字型（套用到題目與輸入顯示區）
			textDisplayWindow.classList.add('zhuyin-font');
			typingDisplay.classList.add('zhuyin-font');
		} else {
			// 回復一般字型
			textDisplayWindow.classList.remove('zhuyin-font');
			typingDisplay.classList.remove('zhuyin-font');
		}
	}

	// 題目顯示：依 questionLines & currentLineIndex 建立行並 highlight
	function renderTextDisplay() {
		textDisplayWindow.innerHTML = '';

		if (!questionLines.length) {
			const span = document.createElement('span');
			span.className = 'upcoming-text';
			span.textContent = '尚未載入題目，請選擇模式並開始練習。';
			textDisplayWindow.appendChild(span);
			return;
		}

		// 顯示全部行，並自動讓目前行保持在可視範圍中
		questionLines.forEach((text, i) => {
			const lineDiv = document.createElement('div');
			lineDiv.classList.add('line');
			lineDiv.textContent = text || '';

			if (i < currentLineIndex) {
				lineDiv.classList.add('typed-correct');
			} else if (i === currentLineIndex) {
				lineDiv.classList.add('current');
			} else {
				lineDiv.classList.add('upcoming');
			}
			textDisplayWindow.appendChild(lineDiv);
		});

		const currentLineElem = textDisplayWindow.querySelector('.line.current');
		if (currentLineElem) {
			const lineTop    = currentLineElem.offsetTop;
			const lineHeight = currentLineElem.offsetHeight;
			const viewTop    = textDisplayWindow.scrollTop;
			const viewBottom = viewTop + textDisplayWindow.clientHeight;

			if (lineTop < viewTop || (lineTop + lineHeight) > viewBottom) {
				textDisplayWindow.scrollTop =
					lineTop - (textDisplayWindow.clientHeight - lineHeight) / 2;
			}
		}
	}

	function syncTypingDisplay() {
		const text = getUserFullText();
		typingDisplay.textContent = text || '';
	}

	// 成績統計：不將 Enter(\n) 納入統計，只算實際中文字
	function updateStatsFromUserText() {
		const userTextRaw = getUserFullText();
		const expectedRaw = questionLines.join('\n');

		const userText     = userTextRaw.replace(/\n/g, '');
		const expectedText = expectedRaw.replace(/\n/g, '');

		const total = userText.length;

		let correct = 0;
		let errors  = 0;

		for (let i = 0; i < total; i++) {
			const u = userText[i];
			const e = expectedText[i];

			if (e === undefined || e === null) {
				// 題目沒有這個位置的字，多打算錯
				errors++;
			} else if (u !== e) {
				errors++;
			} else {
				correct++;
			}
		}

		totalCharsTyped   = total;
		totalCorrectChars = correct;
		totalErrors       = errors;

		charsTypedDisplay.textContent = String(totalCharsTyped);
		errorsDisplay.textContent     = String(totalErrors);
		updateSpeedDisplay();

		// 計字數模式自動結束：以「實際輸入字數」判斷
		if (!practiceFinished && practiceActive &&
			practiceMode === 'measured' &&
			totalCharsTyped >= targetCount) {

			finishPractice('auto-target');
		}
	}

	// --- 4. 狀態控制 ---
	function setIdleState(finished = false) {
		practiceActive     = false;
		isLoadingQuestions = false;

		if (timerInterval) {
			clearInterval(timerInterval);
			timerInterval = null;
		}

		if (!finished) {
			resetScores();
		} else {
			const finalWpm = speedDisplay.textContent;
			showStatusMessage(
				`練習完成！速度: ${finalWpm} 字/分（共 ${totalCharsTyped} 字，錯 ${totalErrors} 字）。`,
				false
			);
		}

		stopTestBtn.disabled = true;
		if (showScoreBtn) showScoreBtn.disabled = false;
		modeMeasuredRadio.disabled  = false;
		modeTimedRadio.disabled     = false;
		levelSelect.disabled        = false;
		charCountSelect.disabled    = false;
		timeDurationSelect.disabled = false;
		fontNoZhuyinRadio.disabled  = false;
		fontZhuyinRadio.disabled    = false;

		// 清掉目前輸入內容與畫面
		typingInput.value = '';
		syncTypingDisplay();

		// 結束後主動 blur，一定會在下一次 click 重新觸發 focusin → startPracticeIfPrepared
		typingInput.blur();

		// 根據目前設定狀態，決定輸入框是否可用
		readSettings();
		updateTypingInputAvailability(false);

		practiceFinished = finished;

		// 若為正常結束，鎖住輸入框 10 秒後再允許重新開始
		if (finished) {
			typingInput.disabled = true;
			setTimeout(() => {
				if (!practiceActive && !isLoadingQuestions) {
					//typingInput.disabled = false;
					updateTypingInputAvailability(true);
				}
			}, 10000);
		}
	}

	function preparePractice() {
		readSettings();
		resetScores();
		practiceFinished = false;
		updateTypingInputAvailability(true);
	}

	// --- 5. 題目載入 ---
	async function fetchQuestions(minLength) {
		showStatusMessage('載入題目中...', false);

		const url = `/api/get_questions.php?level=${encodeURIComponent(targetLevel)}&targetLength=${encodeURIComponent(minLength)}`;
		const res = await fetch(url);
		if (!res.ok) {
			throw new Error(`API 錯誤 (${res.status})`);
		}

		const raw = await res.json();
		if (!Array.isArray(raw)) {
			throw new Error('API 回應格式錯誤（預期為陣列）');
		}
		if (!raw.length) {
			throw new Error('題庫沒有符合條件的題目。');
		}

		// 依標點拆成多行，讓題目顯示更易讀
		const lines = splitLinesByPunctuation(raw);
		if (!lines.length) {
			throw new Error('題庫資料格式不正確。');
		}

		questionLines    = lines;
		currentLineIndex = 0;
		renderTextDisplay();
	}

	function startTimer() {
		if (timerInterval || !practiceActive) return;

		timerInterval = setInterval(() => {
			timeElapsed += 1;
			updateTimerDisplay();
			updateSpeedDisplay();

			if (practiceMode === 'timed' && timeElapsed >= targetTime && !practiceFinished) {
				// 時間結束前，先把目前內容統計一次
				updateStatsFromUserText();
				finishPractice('auto-timeout');
			}
		}, 1000);
	}

	// --- 6. 練習流程 ---
	async function startPracticeIfPrepared() {
		// 僅在「使用者點到輸入框」且已完成設定時才開始
		if (!practicePrepared || practiceActive || isLoadingQuestions) return;
		if (!checkUsername()) return;

		practiceActive       = false;
		practiceFinished     = false;
		isLoadingQuestions   = true;
		finishReason         = 'unknown';

		showStatusMessage('載入題目中...', false);
		textDisplayWindow.innerHTML = '<div class="line upcoming">載入題目中...</div>';

		stopTestBtn.disabled        = true;
		if (showScoreBtn) showScoreBtn.disabled = true;
		modeMeasuredRadio.disabled  = true;
		modeTimedRadio.disabled     = true;
		levelSelect.disabled        = true;
		charCountSelect.disabled    = true;
		timeDurationSelect.disabled = true;
		fontNoZhuyinRadio.disabled  = true;
		fontZhuyinRadio.disabled    = true;
		typingInput.disabled        = true;

		try {
			// 前端給一個較大的最小字數，實際規則由後端符合你之前的說明
			const minLength = (practiceMode === 'measured')
				? (targetCount + 50)
				: Math.max(500, Math.floor(targetTime / 60 * 200) + 50);

			await fetchQuestions(minLength);

			if (!questionLines.length) {
				throw new Error('無法載入題目。');
			}

			// 題目載入成功，開始計時與輸入
			practiceActive     = true;
			isLoadingQuestions = false;
			startTime          = Date.now();
			timeElapsed        = 0;
			timerDisplay.textContent = '00:00';

			renderTextDisplay();
			syncTypingDisplay();

			typingInput.disabled = false;
			typingInput.value    = '';
			syncTypingDisplay();
			typingInput.focus();

			stopTestBtn.disabled = false;
			startTimer();
		} catch (e) {
			console.error('Typing JS: 載入題目失敗', e);
			showStatusMessage('載入題目失敗：' + e.message, true);
			setIdleState(false);
		} finally {
			isLoadingQuestions = false;
		}
	}

	function handleKeyDown(event) {
		// 只看 practiceActive / isLoadingQuestions
		if (!practiceActive || isLoadingQuestions) return;

		if (IGNORE_KEYS.includes(event.key)) {
			return;
		}

		// 方向鍵 / 刪除鍵：讓瀏覽器先修改 input.value，再同步到 display＋統計
		if (EDIT_KEYS.includes(event.key)) {
			setTimeout(() => {
				if (!isComposing && practiceActive) {
					syncTypingDisplay();
					updateStatsFromUserText();
				}
			}, 0);
		}

		if (event.key === 'Enter') {
			event.preventDefault();

			if (isComposing) {
				// 組字中不處理 Enter 換行
				return;
			}

			const currentLine  = typingInput.value || '';
			const expectedLine = questionLines[currentLineIndex] || '';

			// 將這一行視為「已完成的一行」，加入 typedLines
			typedLines.push(currentLine);
			typingInput.value = '';

			syncTypingDisplay();
			updateStatsFromUserText();

			// 題目行向下移一行（用於 highlight）
			if (expectedLine !== undefined) {
				currentLineIndex++;
				renderTextDisplay();
			}

			// 若題目已經沒行了：計字數模式直接結束；限時模式則只禁止繼續輸入
			if (currentLineIndex >= questionLines.length) {
				if (practiceMode === 'measured') {
					finishPractice('auto-no-more-questions');
				} else {
					typingInput.disabled = true;
					showStatusMessage('題目已全部輸入完畢，請等待時間結束。', false);
				}
			}
		}
	}

	function handleInput(event) {
		if (!practiceActive || isLoadingQuestions) return;
		if (isComposing || event.isComposing) return;

		syncTypingDisplay();
		updateStatsFromUserText();
	}

	// --- 7. 練習結束 & 成績寫入 ---
	async function savePracticeResult(reason, stats) {
		try {
			const payload = {
				practice_mode:  practiceMode,
				target_chars:   practiceMode === 'measured' ? targetCount : null,
				target_seconds: practiceMode === 'timed'    ? targetTime  : null,
				start_time:     stats.startEpoch,
				end_time:       stats.endEpoch,
				duration_seconds: stats.durationSeconds,
				question_text:  questionLines.join('\n'),
				input_text:     getUserFullText(),
				total_chars:    totalCharsTyped,
				error_chars:    totalErrors,
				wpm:            parseFloat(speedDisplay.textContent) || 0,
				grade:          parseInt(targetLevel, 10) || null,
				level:          parseInt(targetLevel, 10) || null
			};

			const res = await fetch('/api/practice_log.php', {
				method: 'POST',
				headers: { 'Content-Type': 'application/json' },
				body: JSON.stringify(payload)
			});
			if (!res.ok) {
				throw new Error(`HTTP ${res.status}`);
			}
			const data = await res.json();
			if (!data.result) {
				throw new Error(data.msg || '記錄成績失敗');
			}
		} catch (e) {
			console.error('Typing JS: 成績記錄時發生錯誤', e);
			showStatusMessage('練習結束，但成績記錄時發生錯誤：' + e.message, true);
		}
	}

	function finishPractice(reason = 'unknown') {
		if (practiceFinished) return;

		practiceFinished = true;
		finishReason     = reason;

		practiceActive = false;

		if (timerInterval) {
			clearInterval(timerInterval);
			timerInterval = null;
		}

		// 若最後一行尚未按 Enter，也要納入 typedLines
		const lastLine = typingInput.value.trim();
		if (lastLine) {
			typedLines.push(lastLine);
		}

		// 清空 typing-input，避免重複記錄
		typingInput.value = '';

		// 最後一刻再同步一次，確保 typing-display / 成績 是最新的
		syncTypingDisplay();
		updateStatsFromUserText();

		const nowEpoch   = Math.floor(Date.now() / 1000);
		const startEpoch = startTime ? Math.floor(startTime / 1000) : nowEpoch;
		const duration   = Math.max(timeElapsed, nowEpoch - startEpoch);

		const finalChars   = totalCharsTyped;
		const finalErrors  = totalErrors;
		const correctChars = totalCorrectChars;

		const stats = {
			startEpoch,
			endEpoch: nowEpoch,
			durationSeconds: duration,
			totalChars: finalChars,
			errorChars: finalErrors,
			correctChars,
			reason
		};

		setIdleState(true);
		savePracticeResult(reason, stats);
	}

	// --- 8. 成績視窗相關 ---

	async function loadScoreData() {
		if (!checkUsername()) {
			// init 一開始就檢查一次，避免未登入直接操作
			return;
		}
		if (!scoreTableBody) return;
		if (!checkUsername()) return;

		const range = getSelectedScoreRange();

		if (range !== 'custom') {
			applyScoreRangePreset();
		}

		const start = scoreStartInput && scoreStartInput.value ? scoreStartInput.value : '';
		const end   = scoreEndInput && scoreEndInput.value ? scoreEndInput.value : '';

		const params = new URLSearchParams({
			range,
			start,
			end
		});

		try {
			scoreTableBody.innerHTML = '<tr><td colspan="9" class="no-data">讀取中...</td></tr>';
			renderScoreChart([]);

			const res = await fetch('/api/get_practice_logs.php?' + params.toString());
			if (!res.ok) throw new Error('HTTP ' + res.status);

			const data = await res.json();
			if (!data.result) throw new Error(data.msg || '取得成績失敗');

			const rows = data.data || [];
			renderScoreTable(rows);
			renderScoreChart(rows);
		} catch (e) {
			console.error('loadScoreData error', e);
			if (scoreTableBody) {
				scoreTableBody.innerHTML =
					`<tr><td colspan="9" class="no-data">讀取失敗：${e.message}</td></tr>`;
			}
			renderScoreChart([]);
		}
	}

	function openScoreDialog() {
		if (!checkUsername()) {
			// init 一開始就檢查一次，避免未登入直接操作
			return;
		}
		if (!scoreDialog) return;
		if (!checkUsername()) return;

		// 帳號標籤（若有 global 資訊就帶入）
		if (scoreUserLabel) {
			const name =
				window.currentUserName ||
				window.currentUsername ||
				window.currentUserAccount ||
				'';
			const role =
				window.currentUserRoleLabel ||
				window.currentUserRoleName ||
				'';
			if (name) {
				scoreUserLabel.textContent = role
					? `帳號：${name}（${role}）`
					: `帳號：${name}`;
			} else {
				scoreUserLabel.textContent = '帳號：';
			}
		}

		const range = getSelectedScoreRange();
		if (range !== 'custom') {
			applyScoreRangePreset();
		}

		scoreDialog.showModal();
		loadScoreData();
	}

	function getSelectedScoreRange() {
		let v = 'today';
		scoreRangeRadios.forEach(r => {
			if (r.checked) v = r.value;
		});
		return v;
	}

	function applyScoreRangePreset() {
		if (!scoreStartInput || !scoreEndInput) return;

		const today = new Date();
		const end   = today;
		let start   = new Date(today);

		const range = getSelectedScoreRange();

		if (range === 'today') {
			// 當天：start = end = 今天
		} else if (range === 'week') {
			// 當週：往前推 6 天
			start.setDate(end.getDate() - 6);
		} else if (range === 'month') {
			// 當月：往前推 29 天 (簡化)
			start.setDate(end.getDate() - 29);
		} else {
			return;
		}

		const toYMD = d =>
			d.getFullYear() + '-' +
				String(d.getMonth() + 1).padStart(2, '0') + '-' +
				String(d.getDate()).padStart(2, '0');

		scoreStartInput.value = toYMD(start);
		scoreEndInput.value   = toYMD(end);
	}

	function renderScoreTable(rows) {
		if (!checkUsername()) {
			// init 一開始就檢查一次，避免未登入直接操作
			return;
		}
		if (!scoreTableBody) return;
		if (!rows || !rows.length) {
			scoreTableBody.innerHTML = '<tr><td colspan="9" class="no-data">此區間尚無資料</td></tr>';
			return;
		}

		scoreTableBody.innerHTML = '';
		rows.forEach(r => {
			const tr = document.createElement('tr');

			const tdStart = document.createElement('td');
			tdStart.textContent = r.start_time || '';
			tr.appendChild(tdStart);

			const tdEnd = document.createElement('td');
			tdEnd.textContent = r.end_time || '';
			tr.appendChild(tdEnd);

			const tdDur = document.createElement('td');
			tdDur.textContent = r.duration_seconds != null ? r.duration_seconds : '';
			tr.appendChild(tdDur);

			const tdMode = document.createElement('td');
			if (r.practice_mode === 'measured') {
				tdMode.textContent = '計時';
			} else if (r.practice_mode === 'timed') {
				tdMode.textContent = '限時';
			} else {
				tdMode.textContent = r.practice_mode || '';
			}
			tr.appendChild(tdMode);

			const tdLevel = document.createElement('td');
			tdLevel.textContent = r.grade != null ? r.grade : '';
			tr.appendChild(tdLevel);

			const tdTarget = document.createElement('td');
			tdTarget.textContent = r.target_value != null ? r.target_value : '';
			tr.appendChild(tdTarget);

			const tdInput = document.createElement('td');
			tdInput.textContent = r.total_chars != null ? r.total_chars : '';
			tr.appendChild(tdInput);

			const tdErr = document.createElement('td');
			tdErr.textContent = r.error_chars != null ? r.error_chars : '';
			tr.appendChild(tdErr);

			const tdWpm = document.createElement('td');
			tdWpm.textContent = r.wpm != null ? parseFloat(r.wpm).toFixed(2) : '';
			tr.appendChild(tdWpm);

			// 點擊列開啟明細視窗
			if (scoreDetailDialog && scoreDetailBody) {
				tr.addEventListener('click', () => {
					openScoreDetail(r);
				});
				tr.style.cursor = 'pointer';
			}

			scoreTableBody.appendChild(tr);
		});
	}

	function openScoreDetail(row) {
		// 1. 檢查登入狀態
		if (!checkUsername()) return;

		// 2. 找到已經寫在 HTML 裡的 dialog / body
		let dialog = scoreDetailDialog || document.getElementById('score-detail-dialog');
		if (!dialog) return;

		let body =
			scoreDetailBody ||
			dialog.querySelector('.score-detail-body') ||
			dialog.querySelector('.score-modal-body'); // 舊 class 名稱也順便支援

		if (!body) return;

		// === 小工具：把文字 A、文字 B 做逐字比對，錯的字加上 .diff-error ===
		function renderDiffText(container, selfTextRaw, otherTextRaw) {
			const selfText  = (selfTextRaw  || '').replace(/\r\n/g, '\n');
			const otherText = (otherTextRaw || '').replace(/\r\n/g, '\n');

			container.innerHTML = '';
			const len = selfText.length;

			for (let i = 0; i < len; i++) {
				const ch      = selfText[i];
				const otherCh = i < otherText.length ? otherText[i] : undefined;

				if (ch === '\n') {
					container.appendChild(document.createElement('br'));
					continue;
				}

				const span = document.createElement('span');
				span.textContent = ch;

				// 「別人的同一位置沒有字」或「字不同」都算錯
				if (otherCh === undefined || otherCh !== ch) {
					span.classList.add('diff-error');
				}

				container.appendChild(span);
			}
		}

		// === 整理上半部摘要資訊 ===
		const items = [];

		items.push(['開始時間', row.start_time || '']);
		items.push(['結束時間', row.end_time || '']);
		if (row.duration_seconds != null) {
			items.push(['耗時(秒)', row.duration_seconds]);
		}

		let modeLabel = row.practice_mode || '';
		if (row.practice_mode === 'measured') modeLabel = '計字數';
		if (row.practice_mode === 'timed')    modeLabel = '限時';
		items.push(['模式', modeLabel]);

		if (row.grade != null) {
			items.push(['年級', row.grade]);
		}
		if (row.target_value != null) {
			items.push(['字數 / 時間', row.target_value]);
		}
		if (row.total_chars != null) {
			items.push(['輸入字數', row.total_chars]);
		}
		if (row.error_chars != null) {
			items.push(['錯誤字數', row.error_chars]);
		}
		if (row.wpm != null) {
			items.push(['速度(字/分)', parseFloat(row.wpm).toFixed(2)]);
		}

		// === 清空舊內容，開始重畫 ===
		body.innerHTML = '';

		// 上半部：摘要表格
		const table = document.createElement('table');
		table.className = 'score-detail-table';
		const tbody = document.createElement('tbody');

		items.forEach(([label, value]) => {
			const tr = document.createElement('tr');
			const tdLabel = document.createElement('td');
			const tdValue = document.createElement('td');
			tdLabel.textContent = label;
			tdValue.textContent = value;
			tr.appendChild(tdLabel);
			tr.appendChild(tdValue);
			tbody.appendChild(tr);
		});

		table.appendChild(tbody);
		body.appendChild(table);

		// 取得題目 & 輸入內容（可能為空，但也要顯示框）
		const questionText = row.question_text || '';
		const inputText    = row.input_text    || '';

		// 題目框（一定要有）
		const titleQ = document.createElement('div');
		titleQ.textContent = '題目 (question_text)：';
		titleQ.style.marginTop = '8px';
		titleQ.style.fontWeight = '600';
		body.appendChild(titleQ);

		const boxQ = document.createElement('div');
		boxQ.className = 'score-detail-text';
		// 題目這一欄：用「輸入內容」來判斷哪裡沒打對 → 題目被標出「哪些字沒被正確輸入」
		renderDiffText(boxQ, questionText, inputText);
		body.appendChild(boxQ);

		// 輸入內容框（一定要有）
		const titleI = document.createElement('div');
		titleI.textContent = '輸入內容 (input_text)：';
		titleI.style.marginTop = '8px';
		titleI.style.fontWeight = '600';
		body.appendChild(titleI);

		const boxI = document.createElement('div');
		boxI.className = 'score-detail-text';
		// 輸入內容這一欄：用「題目」來判斷 → 哪些是多打／打錯
		renderDiffText(boxI, inputText, questionText);
		body.appendChild(boxI);

		// 顯示 dialog
		if (typeof dialog.showModal === 'function') {
			dialog.showModal();
		} else {
			// 不支援 dialog 時，用 alert 做 fallback（防呆而已）
			const basicText = items.map(([l, v]) => l + '：' + v).join('\n');
			const qText = questionText ? '\n\n[題目]\n' + questionText : '\n\n[題目]\n(無資料)';
			const iText = inputText    ? '\n\n[輸入內容]\n' + inputText : '\n\n[輸入內容]\n(無資料)';
			alert(basicText + qText + iText);
		}
	}

	function renderScoreChart(rows) {
		if (!scoreChartCtx || !scoreChartCanvas) return;

		const ctx = scoreChartCtx;
		const canvas = scoreChartCanvas;
		const w = canvas.width;
		const h = canvas.height;

		ctx.clearRect(0, 0, w, h);

		if (!rows || !rows.length) {
			ctx.font = '14px sans-serif';
			ctx.textAlign = 'center';
			ctx.textBaseline = 'middle';
			ctx.fillText('尚無資料', w / 2, h / 2);
			return;
		}

		// 時間格式：MM-dd HH:mm:ss
		function formatTimeLabel(ts) {
			if (!ts) return '';
			const d = new Date(ts);
			if (isNaN(d.getTime())) return ts; // 解析失敗就用原字串

			const mm = String(d.getMonth() + 1).padStart(2, '0');
			const dd = String(d.getDate()).padStart(2, '0');
			const HH = String(d.getHours()).padStart(2, '0');
			const MM = String(d.getMinutes()).padStart(2, '0');
			const SS = String(d.getSeconds()).padStart(2, '0');
			return `${mm}-${dd} ${HH}:${MM}:${SS}`;
		}

		const times  = rows.map(r => formatTimeLabel(r.start_time || ''));
		const speeds = rows.map(r => parseFloat(r.wpm || 0));
		const errors = rows.map(r => parseInt(r.error_chars || 0, 10));

		const maxSpeed = Math.max(...speeds, 10);
		const maxError = Math.max(...errors, 1);

		const paddingLeft   = 60;
		const paddingRight  = 60;
		const paddingTop    = 30;
		const paddingBottom = 50;

		const innerW = w - paddingLeft - paddingRight;
		const innerH = h - paddingTop - paddingBottom;

		function xPos(i) {
			if (rows.length === 1) return paddingLeft + innerW / 2;
			return paddingLeft + (innerW * i / (rows.length - 1));
		}

		function ySpeed(v) {
			return paddingTop + innerH * (1 - v / maxSpeed);
		}

		function yError(v) {
			return paddingTop + innerH * (1 - v / maxError);
		}

		// 座標軸
		ctx.strokeStyle = '#444';
		ctx.lineWidth   = 1;
		ctx.beginPath();
		ctx.moveTo(paddingLeft, paddingTop);
		ctx.lineTo(paddingLeft, paddingTop + innerH);
		ctx.lineTo(paddingLeft + innerW, paddingTop + innerH);
		ctx.stroke();

		ctx.font = '12px sans-serif';
		ctx.textBaseline = 'middle';

		// 左軸（速度）單位
		ctx.textAlign = 'right';
		ctx.fillText('速度(字/分)', paddingLeft - 8, paddingTop);

		// 右軸（錯誤）單位
		ctx.textAlign = 'left';
		ctx.fillText('錯誤字數(次)', paddingLeft + innerW + 8, paddingTop);

		// X 軸單位標示
		ctx.textAlign = 'center';
		ctx.textBaseline = 'bottom';
		ctx.fillText('時間 (MM-dd HH:mm:ss)', paddingLeft + innerW / 2, h - 2);

		// X 軸標籤（逐筆時間）
		ctx.textBaseline = 'top';
		times.forEach((t, i) => {
			const x = xPos(i);
			const y = paddingTop + innerH + 4;
			ctx.save();
			ctx.translate(x, y);
			ctx.rotate(-Math.PI / 6); // 斜 30 度避免重疊
			ctx.fillText(t, 0, 0);
			ctx.restore();
		});

		// 速度折線
		ctx.strokeStyle = '#007bff';
		ctx.fillStyle   = '#007bff';
		ctx.lineWidth   = 2;
		ctx.beginPath();
		speeds.forEach((v, i) => {
			const x = xPos(i);
			const y = ySpeed(v);
			if (i === 0) ctx.moveTo(x, y);
			else ctx.lineTo(x, y);
		});
		ctx.stroke();
		speeds.forEach((v, i) => {
			const x = xPos(i);
			const y = ySpeed(v);
			ctx.beginPath();
			ctx.arc(x, y, 3, 0, Math.PI * 2);
			ctx.fill();
		});

		// 錯誤字數折線（右軸）
		ctx.setLineDash([4, 4]);
		ctx.strokeStyle = '#ff5555';
		ctx.fillStyle   = '#ff5555';
		ctx.beginPath();
		errors.forEach((v, i) => {
			const x = xPos(i);
			const y = yError(v);
			if (i === 0) ctx.moveTo(x, y);
			else ctx.lineTo(x, y);
		});
		ctx.stroke();
		ctx.setLineDash([]);
		errors.forEach((v, i) => {
			const x = xPos(i);
			const y = yError(v);
			ctx.beginPath();
			ctx.arc(x, y, 3, 0, Math.PI * 2);
			ctx.fill();
		});
	}

	// --- 9. 初始化與事件綁定 ---
	function init() {
		if (!checkUsername()) {
			// init 一開始就檢查一次，避免未登入直接操作
			return;
		}

		// 若 dashboard 有傳年級，則帶入；否則看 select 預設值
		if (typeof window.currentUserGrade === 'number' &&
			window.currentUserGrade >= 3 && window.currentUserGrade <= 6 &&
			Array.from(levelSelect.options).some(o => o.value === String(window.currentUserGrade))) {

			levelSelect.value = String(window.currentUserGrade);
		}

		readSettings();
		updateFont();
		resetScores();

		// 初始狀態：依照目前下拉選單 / 單選按鈕是否有值，決定輸入框可不可點
		updateTypingInputAvailability(true);
		practiceFinished = false;

		modeMeasuredRadio.addEventListener('change', preparePractice);
		modeTimedRadio.addEventListener('change', preparePractice);
		levelSelect.addEventListener('change', preparePractice);
		charCountSelect.addEventListener('change', preparePractice);
		timeDurationSelect.addEventListener('change', preparePractice);

		fontNoZhuyinRadio.addEventListener('change', updateFont);
		fontZhuyinRadio.addEventListener('change', updateFont);

		typingInput.addEventListener('focusin', startPracticeIfPrepared);
		typingInput.addEventListener('keydown', handleKeyDown);
		typingInput.addEventListener('input', handleInput);

		stopTestBtn.addEventListener('click', () => {
			if (practiceActive) {
				finishPractice('manual-stop');
			}
		});

		typingInput.addEventListener('compositionstart', () => {
			isComposing = true;
		});

		typingInput.addEventListener('compositionend', (event) => {
			isComposing = false;
			if (!practiceActive || isLoadingQuestions) return;
			syncTypingDisplay();
			updateStatsFromUserText();
		});

		// 禁用 copy / paste / context menu
		typingInput.addEventListener('paste', e => e.preventDefault());
		typingInput.addEventListener('copy', e => e.preventDefault());
		typingInput.addEventListener('cut', e => e.preventDefault());
		typingInput.addEventListener('contextmenu', e => e.preventDefault());

		// 成績視窗事件綁定
		if (showScoreBtn && scoreDialog) {
			showScoreBtn.addEventListener('click', openScoreDialog);
		}
		if (scoreCloseBtn && scoreDialog) {
			scoreCloseBtn.addEventListener('click', () => scoreDialog.close());
		}
		if (scoreDetailDialog) {
			const closeBtn = scoreDetailDialog.querySelector('.score-detail-close');
			if (closeBtn) {
				closeBtn.addEventListener('click', () => scoreDetailDialog.close());
			}
		}
		if (scoreRangeRadios && scoreRangeRadios.length) {
			scoreRangeRadios.forEach(r => {
				r.addEventListener('change', () => {
					if (r.value !== 'custom') {
						applyScoreRangePreset();
					}
					if (scoreDialog && scoreDialog.open) {
						loadScoreData();
					}
				});
			});
		}
		if (scoreRefreshBtn) {
			scoreRefreshBtn.addEventListener('click', () => {
				if (scoreDialog && scoreDialog.open) {
					loadScoreData();
				}
			});
		}
	}

	// 重要：這裡直接呼叫 init()，不要用 DOMContentLoaded
	// 因為本頁是由 dashboard 動態載入，script 載入時 HTML 已經在 DOM 內。
	init();
})();

