<?php
session_start();
include 'db_connect.php';
include 'log_helper.php';

header('Content-Type: application/json; charset=utf-8');

if (!isset($_SESSION['user'])) {
    http_response_code(401);
    echo json_encode(['message' => '未授權，請先登入']);
    return;
}

$currentUser = $_SESSION['user'];
$role        = $currentUser['role'];
$method      = $_SERVER['REQUEST_METHOD'];
$action      = $_REQUEST['action'] ?? null;
$pdo         = get_db_connection();

/*
|--------------------------------------------------------------------------
| 1. GET: fetch=classlist  — Admin + Teacher 都可使用
|--------------------------------------------------------------------------
*/
if ($method === 'GET' && isset($_GET['fetch']) && $_GET['fetch'] === 'classlist') {

    if (!in_array($role, ['admin', 'teacher'], true)) {
        http_response_code(403);
        echo json_encode(['message' => '權限不足']);
        return;
    }

    $stmt = $pdo->query("SELECT id, class_name FROM classes ORDER BY class_name");
    $classes = $stmt->fetchAll(PDO::FETCH_ASSOC);

    echo json_encode($classes, JSON_UNESCAPED_UNICODE);
    return;
}

/*
|--------------------------------------------------------------------------
| 2. 教師無法進行任何 CRUD（POST/PUT/DELETE/export/batch）
|--------------------------------------------------------------------------
*/
if ($role === 'teacher') {
    http_response_code(403);
    echo json_encode(['message' => '教師無法管理班級資料']);
    return;
}

/*
|--------------------------------------------------------------------------
| 3. 學生無法使用任何班級 API
|--------------------------------------------------------------------------
*/
if ($role === 'student') {
    http_response_code(403);
    echo json_encode(['message' => '學生無法存取班級資料']);
    return;
}

/*
|--------------------------------------------------------------------------
| 只有 admin 可以走到這裡（CRUD 功能）
|--------------------------------------------------------------------------
*/

try {

    // ========================
    // (A) 批次建立班級 action=batch_create
    // ========================
    if ($method === 'POST' && $action === 'batch_create') {

        if (!isset($_FILES['class_file']) || $_FILES['class_file']['error'] !== UPLOAD_ERR_OK) {
            http_response_code(400);
            echo json_encode(['message' => '檔案上傳失敗']);
            return;
        }

        $tmp  = $_FILES['class_file']['tmp_name'];
        $name = $_FILES['class_file']['name'];
        $ext  = strtolower(pathinfo($name, PATHINFO_EXTENSION));

        $createdCount = 0;
        $errors = [];

        if ($ext === 'csv') {
            if (($handle = fopen($tmp, 'r')) === false) {
                http_response_code(400);
                echo json_encode(['message' => 'CSV 無法開啟']);
                return;
            }

            $pdo->beginTransaction();
            $header = fgetcsv($handle);

            if (!$header || !in_array('class_name', $header)) {
                fclose($handle);
                http_response_code(400);
                echo json_encode(['message' => "CSV 缺少必要欄位 class_name"]);
                return;
            }

            $line = 1;
            while (($row = fgetcsv($handle)) !== false) {
                $line++;
                if (empty(array_filter($row))) continue;

                if (count($header) != count($row)) {
                    $errors[] = "第 $line 行：欄位數量不符";
                    continue;
                }

                $item = array_combine($header, $row);
                $name = trim($item['class_name']);
                $grade = $item['grade_level'] ?? null;

                if ($name === '') {
                    $errors[] = "第 $line 行：缺少班級名稱";
                    continue;
                }

                $stmt = $pdo->prepare("INSERT INTO classes (class_name, grade_level) VALUES (?, ?)");
                $stmt->execute([
                    $name,
                    ($grade !== null && $grade !== '') ? (int)$grade : null
                ]);

                $createdCount++;
            }

            fclose($handle);
            $pdo->commit();

            log_operation($pdo, $currentUser['id'], $currentUser['username'], 'BATCH_CREATE_CLASSES');

            echo json_encode(['success' => true, 'createdCount' => $createdCount, 'errors' => $errors]);
            return;
        }

        http_response_code(400);
        echo json_encode(['message' => '僅支援 CSV']);
        return;
    }

    // ========================
    // (B) 匯出班級 CSV action=export
    // ========================
    if ($method === 'GET' && $action === 'export') {
        $filename = "classes-export-" . date('Ymd') . ".csv";

        header("Content-Type: text/csv; charset=utf-8");
        header("Content-Disposition: attachment; filename={$filename}");

        $out = fopen("php://output", "w");
        fprintf($out, chr(0xEF) . chr(0xBB) . chr(0xBF));

        fputcsv($out, ['id','class_name','grade_level']);

        $stmt = $pdo->query("SELECT id, class_name, grade_level FROM classes ORDER BY grade_level, class_name");
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            fputcsv($out, $row);
        }

        fclose($out);
        return;
    }

    // ========================
    // (C) 一般 CRUD
    // ========================
    switch ($method) {

        case 'GET':
            $stmt = $pdo->query("SELECT id, class_name, grade_level FROM classes ORDER BY grade_level, class_name");
            echo json_encode($stmt->fetchAll(PDO::FETCH_ASSOC));
            return;

        case 'POST':
            $data = json_decode(file_get_contents('php://input'), true) ?? [];
            if (empty($data['class_name'])) {
                http_response_code(400);
                echo json_encode(['message' => '班級名稱為必填']);
                return;
            }

            $stmt = $pdo->prepare("INSERT INTO classes (class_name, grade_level) VALUES (?,?)");
            $stmt->execute([
                $data['class_name'],
                (!empty($data['grade_level'])) ? (int)$data['grade_level'] : null
            ]);

            echo json_encode(['success'=>true]);
            return;

        case 'PUT':
            $data = json_decode(file_get_contents('php://input'), true) ?? [];
            if (empty($data['id']) || empty($data['class_name'])) {
                http_response_code(400);
                echo json_encode(['message'=>'缺少資料']);
                return;
            }

            $stmt = $pdo->prepare("UPDATE classes SET class_name=?, grade_level=? WHERE id=?");
            $stmt->execute([
                $data['class_name'],
                (!empty($data['grade_level'])) ? (int)$data['grade_level'] : null,
                $data['id']
            ]);

            echo json_encode(['success'=>true]);
            return;

        case 'DELETE':
            $id = $_GET['id'] ?? null;
            if (!$id) {
                http_response_code(400);
                echo json_encode(['message'=>'缺少 ID']);
                return;
            }

            // 檢查是否有學生使用該班級
            $check = $pdo->prepare("SELECT COUNT(*) FROM users WHERE JSON_CONTAINS(class, CAST(? AS CHAR))");
            $check->execute([$id]);

            if ($check->fetchColumn() > 0) {
                http_response_code(400);
                echo json_encode(['message'=>'無法刪除：仍有學生屬於此班級']);
                return;
            }

            $stmt = $pdo->prepare("DELETE FROM classes WHERE id=?");
            $stmt->execute([$id]);

            echo json_encode(['success'=>true]);
            return;

        default:
            http_response_code(405);
            echo json_encode(['message'=>'方法不允許']);
            return;
    }

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['message'=>'錯誤：'.$e->getMessage()]);
    return;
}
?>

