<?php
session_start();
include 'db_connect.php';
include 'log_helper.php';

header('Content-Type: application/json');

// --- Access Control ---
if (!isset($_SESSION['user']) || !in_array($_SESSION['user']['role'], ['admin', 'teacher'])) { http_response_code(403); echo json_encode(['message' => '權限不足：僅管理者或教師可存取']); exit; }
$currentUser = $_SESSION['user'];
$method = $_SERVER['REQUEST_METHOD'];
$pdo = get_db_connection();
$action = $_REQUEST['action'] ?? null;

// --- Helper: Validate Question Data (Removed type name check) ---
function validateQuestion(&$data, $isUpdate = false) {
    if (empty($data['name'])) { return ['valid' => false, 'message' => '題目名稱不得為空']; }
    if (mb_strlen($data['name'], 'UTF-8') > 255) { return ['valid' => false, 'message' => '題目名稱不得超過 255 字']; }
    if (empty($data['content'])) { return ['valid' => false, 'message' => '題目內容不得為空']; }
    $data['length'] = mb_strlen(trim($data['content']), 'UTF-8');
    if ($data['length'] > 1500) { return ['valid' => false, 'message' => '題目內容不得超過 1500 字 (目前 ' . $data['length'] . ' 字)']; }
    if (empty($data['level']) || !in_array($data['level'], [3, 4, 5, 6])) { return ['valid' => false, 'message' => '程度/年級必須是 3 到 6']; }
    // type_id validation happens implicitly via FK or check before insert/update if needed
    $data['level'] = (int)$data['level'];
    return ['valid' => true];
}

try {
    switch ($method) {
        // --- GET --- (Join to get type_name)
        case 'GET':
             $sql = "
                SELECT
                    tb.id, tb.name, tb.type_id, qt.type_name, -- Select both ID and Name
                    tb.content, tb.length, tb.level,
                    tb.created_by, tb.created_at, tb.update_at, u.username as creator_username
                FROM test_bank tb
                JOIN users u ON tb.created_by = u.id
                LEFT JOIN question_types qt ON tb.type_id = qt.id -- LEFT JOIN in case type_id is NULL
            ";
             $sql .= " ORDER BY tb.update_at DESC";
             $stmt = $pdo->query($sql);
             echo json_encode($stmt->fetchAll());
            break;

        // --- POST --- (Store type_id)
        case 'POST':
            $data = json_decode(file_get_contents('php://input'), true);
            $validation = validateQuestion($data);
            $typeId = isset($data['type_id']) && $data['type_id'] !== '' ? (int)$data['type_id'] : null; // Get type ID, ensure it's int or null
            $logDetails = ['name' => $data['name'] ?? null, 'type_id' => $typeId, 'level' => $data['level'] ?? null, 'length' => $data['length'] ?? 0];

            if (!$validation['valid']) {
                log_operation($pdo, $currentUser['id'], $currentUser['username'], 'QUESTION_CREATE_FAIL', json_encode(array_merge($logDetails, ['error' => $validation['message']])));
                http_response_code(400); echo json_encode(['message' => $validation['message']]); exit;
            }

            // Insert type_id
            $sql = "INSERT INTO test_bank (name, type_id, content, length, level, created_by) VALUES (?, ?, ?, ?, ?, ?)";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                trim($data['name']),
                $typeId, // Store the ID
                trim($data['content']),
                $data['length'],
                $data['level'],
                $currentUser['id']
            ]);
            $newId = $pdo->lastInsertId();

            log_operation($pdo, $currentUser['id'], $currentUser['username'], 'QUESTION_CREATE_SUCCESS', json_encode(array_merge($logDetails, ['id' => $newId])));
            // Return the newly created question data including type name
            $stmt = $pdo->prepare("SELECT tb.*, u.username as creator_username, qt.type_name FROM test_bank tb JOIN users u ON tb.created_by = u.id LEFT JOIN question_types qt ON tb.type_id = qt.id WHERE tb.id = ?");
            $stmt->execute([$newId]);
            echo json_encode($stmt->fetch());
            break;

        // --- PUT --- (Store type_id)
        case 'PUT':
            $data = json_decode(file_get_contents('php://input'), true);
            $targetId = $data['id'] ?? null;
            if (empty($targetId)) { http_response_code(400); echo json_encode(['message' => '缺少題目 ID']); exit; }

            $stmt = $pdo->prepare("SELECT * FROM test_bank WHERE id = ?"); $stmt->execute([$targetId]); $originalQuestion = $stmt->fetch();
            if (!$originalQuestion) { http_response_code(404); echo json_encode(['message' => '找不到指定的題目']); exit; }

            // Permission Check
            if ($currentUser['role'] == 'teacher' && $originalQuestion['created_by'] != $currentUser['id']) {
                log_operation($pdo, $currentUser['id'], $currentUser['username'], 'QUESTION_UPDATE_FAIL', json_encode(['id' => $targetId, 'error' => 'Permission denied']));
                http_response_code(403); echo json_encode(['message' => '權限不足：您只能修改自己建立的題目']); exit;
            }

            $validation = validateQuestion($data, true);
            $typeId = isset($data['type_id']) && $data['type_id'] !== '' ? (int)$data['type_id'] : null; // Get type ID
            $logDetails = ['id' => $targetId, 'name' => $data['name'] ?? null, 'type_id' => $typeId, 'level' => $data['level'] ?? null, 'length' => $data['length'] ?? 0];

            if (!$validation['valid']) {
                log_operation($pdo, $currentUser['id'], $currentUser['username'], 'QUESTION_UPDATE_FAIL', json_encode(array_merge($logDetails, ['error' => $validation['message']])));
                http_response_code(400); echo json_encode(['message' => $validation['message']]); exit;
            }

            // Update type_id
            $sql = "UPDATE test_bank SET name = ?, type_id = ?, content = ?, length = ?, level = ? WHERE id = ?";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                trim($data['name']),
                $typeId, // Store the ID
                trim($data['content']),
                $data['length'],
                $data['level'],
                $targetId
            ]);

            log_operation($pdo, $currentUser['id'], $currentUser['username'], 'QUESTION_UPDATE_SUCCESS', json_encode($logDetails));
            // Return updated question data including type name
            $stmt = $pdo->prepare("SELECT tb.*, u.username as creator_username, qt.type_name FROM test_bank tb JOIN users u ON tb.created_by = u.id LEFT JOIN question_types qt ON tb.type_id = qt.id WHERE tb.id = ?");
            $stmt->execute([$targetId]);
            echo json_encode($stmt->fetch());
            break;

        // --- DELETE --- (Check uses type_id)
        case 'DELETE':
            $targetId = $_GET['id'] ?? null;
            if (empty($targetId)) { http_response_code(400); echo json_encode(['message' => '缺少題目 ID']); exit; }

            $stmt = $pdo->prepare("SELECT * FROM test_bank WHERE id = ?"); $stmt->execute([$targetId]); $originalQuestion = $stmt->fetch();
            if (!$originalQuestion) { http_response_code(404); echo json_encode(['message' => '找不到指定的題目']); exit; }
            $logDetails = ['id' => $targetId, 'name' => $originalQuestion['name'] ?? 'N/A'];

            // Permission Check
            if ($currentUser['role'] == 'teacher' && $originalQuestion['created_by'] != $currentUser['id']) {
                log_operation($pdo, $currentUser['id'], $currentUser['username'], 'QUESTION_DELETE_FAIL', json_encode(array_merge($logDetails, ['error' => 'Permission denied'])));
                http_response_code(403); echo json_encode(['message' => '權限不足：您只能刪除自己建立的題目']); exit;
            }

            // Check if question is used in tests
            $checkSql = "SELECT COUNT(*) FROM test_questions WHERE question_id = ?"; $checkStmt = $pdo->prepare($checkSql); $checkStmt->execute([$targetId]);
            if ($checkStmt->fetchColumn() > 0) {
                 log_operation($pdo, $currentUser['id'], $currentUser['username'], 'QUESTION_DELETE_FAIL', json_encode(array_merge($logDetails, ['error' => 'Question in use'])));
                 http_response_code(400); echo json_encode(['message' => '無法刪除：此題目已被用於測驗中。']); exit;
            }

            // Delete
            $sql = "DELETE FROM test_bank WHERE id = ?"; $stmt = $pdo->prepare($sql); $stmt->execute([$targetId]);

            if ($stmt->rowCount() > 0) {
                log_operation($pdo, $currentUser['id'], $currentUser['username'], 'QUESTION_DELETE_SUCCESS', json_encode($logDetails));
                echo json_encode(['success' => true]);
            } else {
                 log_operation($pdo, $currentUser['id'], $currentUser['username'], 'QUESTION_DELETE_FAIL', json_encode(array_merge($logDetails, ['error' => 'Not found or delete failed'])));
                 http_response_code(404); echo json_encode(['message' => '刪除失敗：找不到指定的題目 ID 或刪除時發生錯誤']);
            }
            break;

        default:
             http_response_code(405); echo json_encode(['message' => '方法不允許']); break;
    }
} catch (\PDOException $e) {
    // Log DB errors
    log_operation($pdo, $currentUser['id'], $currentUser['username'], 'TESTBANK_DB_ERROR', json_encode(['error' => $e->getMessage(), 'code' => $e->getCode()]));
    http_response_code(500); echo json_encode(['message' => '資料庫錯誤: ' . $e->getMessage()]);
} catch (\Exception $e) {
     // Log other errors
    log_operation($pdo, $currentUser['id'], $currentUser['username'], 'TESTBANK_ERROR', json_encode(['error' => $e->getMessage()]));
    http_response_code(400); echo json_encode(['message' => '處理錯誤: ' . $e->getMessage()]);
}

