<?php
/**
 * users.php
 * 帳號管理 API
 *
 * 功能：
 * - GET    /api/users.php                  取得目前登入者可見的帳號列表
 * - GET    /api/users.php?id=123           取得單一帳號
 * - GET    /api/users.php?action=export    匯出 CSV
 * - POST   /api/users.php                  新增帳號
 * - PUT    /api/users.php?id=123           更新帳號（不可改 username / role）
 * - DELETE /api/users.php?id=123           刪除帳號
 *
 * - POST   /api/users.php?action=change_password
 *          變更自己帳號的密碼（僅 admin / teacher）
 *
 * - POST   /api/users.php?action=reset_password
 *          管理者重設他人密碼（預設＝帳號），規則：
 *          - 只允許 admin 呼叫
 *          - 任何人都不得重設 typing 密碼
 *          - 不可重設自己密碼（請用 change_password）
 *          - 若目標是 admin，只有 typing 可重設其密碼
 */

session_start();
header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/db_connect.php';
if (file_exists(__DIR__ . '/log_helper.php')) {
    require_once __DIR__ . '/log_helper.php';
}

/**
 * 簡單輸出 JSON 的 helper
 */
function json_response($data, int $status = 200)
{
    http_response_code($status);
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
}

/**
 * 取得目前登入者
 */
if (!isset($_SESSION['user']) || !is_array($_SESSION['user'])) {
    json_response(['message' => '未授權，請先登入'], 401);
    return;
}

$currentUser = $_SESSION['user'];
$currentUserId = (int)$currentUser['id'];
$currentRole = $currentUser['role'] ?? 'student';
$currentUsername = $currentUser['username'] ?? '';

$method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
$action = $_GET['action'] ?? null;

try {
    $pdo = get_db_connection();

    /**
     * 取得教師所屬班級 ID 陣列
     */
    function get_teacher_class_ids(PDO $pdo, int $userId): array
    {
        // 依你的實際資料表調整；這裡假設有 teacher_classes(user_id, class_id)
        try {
            $stmt = $pdo->prepare("SELECT class_id FROM teacher_classes WHERE user_id = ?");
            $stmt->execute([$userId]);
            $rows = $stmt->fetchAll(PDO::FETCH_COLUMN);
            return array_map('intval', $rows ?: []);
        } catch (Throwable $e) {
            // 若沒有 teacher_classes 這張表，回傳空陣列
            return [];
        }
    }

    /**
     * 取得目前登入者可見的使用者列表
     */
    function get_visible_users(PDO $pdo, array $currentUser, array $teacherClassIDs): array
    {
        $role = $currentUser['role'] ?? 'student';
        $userId = (int)$currentUser['id'];
        $username = $currentUser['username'] ?? '';

        if ($role === 'admin') {
            // 管理者：可以看到所有帳號
            $stmt = $pdo->query("SELECT id, username, name, class, role, grade, is_active 
                                 FROM users ORDER BY id ASC");
            $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
            foreach ($rows as &$r) {
                $r['is_active'] = (bool)$r['is_active'];
            }
            return $rows;
        }

        if ($role === 'teacher') {
            // 教師：可見自己 + 自己班級學生 + 未分班學生
            $ids = $teacherClassIDs;
            $placeholders = '';
            $params = [];

            $sql = "SELECT id, username, name, class, role, grade, is_active FROM users WHERE ";
            $whereParts = [];

            // 自己
            $whereParts[] = "id = ?";
            $params[] = $userId;

            // 學生（自己的班級 或 未分班）
            // 用 class JSON 判斷
            $sqlBaseStudent = "role = 'student'";

            // 未分班 (class IS NULL 或 '[]')
            $whereStudent = "($sqlBaseStudent AND (class IS NULL OR class = '[]'))";

            if (!empty($ids)) {
                // 在自己班級的學生
                $placeholders = implode(',', array_fill(0, count($ids), '?'));
                $whereStudent = "($sqlBaseStudent AND (
                    class IS NULL OR class = '[]' OR JSON_OVERLAPS(class, JSON_ARRAY($placeholders))
                ))";
                $params = array_merge($params, $ids);
            }

            $whereParts[] = $whereStudent;

            $sql .= implode(' OR ', $whereParts) . " ORDER BY id ASC";

            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);
            $rows = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
            foreach ($rows as &$r) {
                $r['is_active'] = (bool)$r['is_active'];
            }
            return $rows;
        }

        // 學生：只看自己
        $stmt = $pdo->prepare("SELECT id, username, name, class, role, grade, is_active 
                               FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$row) return [];
        $row['is_active'] = (bool)$row['is_active'];
        return [$row];
    }

    /**
     * 記錄操作
     */
    function log_operation_safe(PDO $pdo, int $userId, string $username, string $op)
    {
        if (function_exists('log_operation')) {
            try {
                log_operation($pdo, $userId, $username, $op);
            } catch (Throwable $e) {
                // 忽略 log 寫入錯誤
            }
        }
    }

    // 先算一次 teacher 的班級
    $teacherClassIDs = [];
    if ($currentRole === 'teacher') {
        $teacherClassIDs = get_teacher_class_ids($pdo, $currentUserId);
    }

    // =========================================================
    // 特殊 action: change_password
    // =========================================================
    if ($method === 'POST' && $action === 'change_password') {

        $data = json_decode(file_get_contents('php://input'), true) ?? [];
        $currentPassword = trim($data['current_password'] ?? '');
        $newPassword     = trim($data['new_password'] ?? '');

        if ($currentPassword === '' || $newPassword === '') {
            json_response(['success' => false, 'message' => '請輸入目前密碼與新密碼'], 400);
            return;
        }

        // 只有 admin / teacher 有密碼機制
        if (!in_array($currentRole, ['admin', 'teacher'], true)) {
            json_response(['success' => false, 'message' => '此角色不需設定密碼'], 403);
            return;
        }

        $stmt = $pdo->prepare("SELECT id, username, role, password FROM users WHERE id = ?");
        $stmt->execute([$currentUserId]);
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$row) {
            json_response(['success' => false, 'message' => '帳號不存在'], 404);
            return;
        }

        if (empty($row['password'])) {
            // 理論上 upgrade 時就應已填入 password，
            // 若為空，要求管理者先重設一次
            json_response([
                'success' => false,
                'message' => '尚未設定密碼，請先聯絡管理者重設一次密碼'
            ], 400);
            return;
        }

        if (!password_verify($currentPassword, $row['password'])) {
            json_response(['success' => false, 'message' => '目前密碼錯誤'], 400);
            return;
        }

        if (strlen($newPassword) < 4) {
            json_response(['success' => false, 'message' => '新密碼長度至少需 4 碼'], 400);
            return;
        }

        $hash = password_hash($newPassword, PASSWORD_DEFAULT);
        $upd  = $pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
        $upd->execute([$hash, $currentUserId]);

        log_operation_safe($pdo, $currentUserId, $currentUsername, 'USER_CHANGE_PASSWORD');

        json_response(['success' => true, 'message' => '密碼已變更完成']);
        return;
    }

    // =========================================================
    // 特殊 action: reset_password（管理者重設他人密碼）
    // =========================================================
    if ($method === 'POST' && $action === 'reset_password') {

        $data = json_decode(file_get_contents('php://input'), true) ?? [];
        $targetId = (int)($data['id'] ?? 0);

        if (!$targetId) {
            json_response(['message' => '缺少目標帳號 ID'], 400);
            return;
        }

        if ($currentRole !== 'admin') {
            json_response(['message' => '僅管理者可以重設密碼'], 403);
            return;
        }

        $stmt = $pdo->prepare("SELECT id, username, role FROM users WHERE id = ?");
        $stmt->execute([$targetId]);
        $targetUser = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$targetUser) {
            json_response(['message' => '目標帳號不存在'], 404);
            return;
        }

        // ✅ typing 帳號不可被任何人還原密碼
        if ($targetUser['username'] === 'typing') {
            json_response(['message' => 'typing 帳號不得還原密碼'], 403);
            return;
        }

        // ✅ 不可用 reset_password 還原自己的密碼（請用 change_password）
        if ((int)$targetUser['id'] === $currentUserId) {
            json_response(['message' => '請使用「變更我的密碼」功能，不可還原自己的密碼'], 403);
            return;
        }

        // 密碼只針對 admin / teacher
        if (!in_array($targetUser['role'], ['admin', 'teacher'], true)) {
            json_response(['message' => '此帳號類型不使用密碼機制'], 400);
            return;
        }

        // ✅ 僅 typing 可重設「其他 admin」的密碼
        if ($targetUser['role'] === 'admin' && $currentUsername !== 'typing') {
            json_response(['message' => '僅 typing 可重設其他管理者的密碼'], 403);
            return;
        }

        $hash = password_hash($targetUser['username'], PASSWORD_DEFAULT);
        $upd  = $pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
        $upd->execute([$hash, $targetUser['id']]);

        log_operation_safe($pdo, $currentUserId, $currentUsername, 'USER_RESET_PASSWORD');

        json_response(['success' => true, 'message' => '密碼已重設（預設為帳號）']);
        return;
    }

    // =========================================================
    // 特殊 action: export（匯出 CSV）
    // =========================================================
    if ($method === 'GET' && $action === 'export') {

        // 匯出僅限 admin / teacher
        if (!in_array($currentRole, ['admin', 'teacher'], true)) {
            json_response(['message' => '權限不足'], 403);
            return;
        }

        $users = get_visible_users($pdo, $currentUser, $teacherClassIDs);

        // 改 header 為 CSV
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="users_' . date('Ymd_His') . '.csv"');

        $fp = fopen('php://output', 'w');
        // BOM for Excel
        fprintf($fp, chr(0xEF) . chr(0xBB) . chr(0xBF));

        // 標題列
        fputcsv($fp, ['id', 'username', 'name', 'class', 'role', 'grade', 'is_active']);

        foreach ($users as $u) {
            fputcsv($fp, [
                $u['id'],
                $u['username'],
                $u['name'],
                $u['class'],
                $u['role'],
                $u['grade'],
                $u['is_active'] ? 1 : 0,
            ]);
        }
        return;
    }

    // =========================================================
    // 一般 CRUD
    // =========================================================

    switch ($method) {

        // ===========================
        // GET：取得列表或單筆
        // ===========================
        case 'GET': {
            $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

            if ($id > 0) {
                // 單筆
                $stmt = $pdo->prepare("SELECT id, username, name, class, role, grade, is_active 
                                       FROM users WHERE id = ?");
                $stmt->execute([$id]);
                $user = $stmt->fetch(PDO::FETCH_ASSOC);

                if (!$user) {
                    json_response(['message' => '帳號不存在'], 404);
                    return;
                }

                // 權限檢查：可見性
                $canView = false;

                if ($currentRole === 'admin') {
                    $canView = true;
                } elseif ($currentRole === 'teacher') {

                    if ((int)$user['id'] === $currentUserId) {
                        $canView = true;
                    } elseif ($user['role'] === 'student') {

                        $classIds = [];
                        if (!empty($user['class'])) {
                            $tmp = json_decode($user['class'], true);
                            if (is_array($tmp)) $classIds = $tmp;
                        }
                        $isUnassigned = empty($classIds);
                        $inMyClass    = false;

                        if (!$isUnassigned && !empty($teacherClassIDs)) {
                            foreach ($classIds as $cid) {
                                if (in_array((int)$cid, $teacherClassIDs, true)) {
                                    $inMyClass = true;
                                    break;
                                }
                            }
                        }
                        if ($isUnassigned || $inMyClass) {
                            $canView = true;
                        }
                    }

                } else { // student
                    if ((int)$user['id'] === $currentUserId) {
                        $canView = true;
                    }
                }

                if (!$canView) {
                    json_response(['message' => '權限不足'], 403);
                    return;
                }

                $user['is_active'] = (bool)$user['is_active'];
                json_response($user);
                return;
            }

            // 列表
            $rows = get_visible_users($pdo, $currentUser, $teacherClassIDs);
            json_response($rows);
            return;
        }

        // ===========================
        // POST：新增帳號
        // ===========================
        case 'POST': {

            if ($action !== null) {
                json_response(['message' => '無效的操作'], 400);
                return;
            }

            $data = json_decode(file_get_contents('php://input'), true);
            if (!is_array($data)) {
                json_response(['message' => 'Invalid JSON'], 400);
                return;
            }

            $newRole = $data['role'] ?? null;
            if (!in_array($newRole, ['admin', 'teacher', 'student'], true)) {
                json_response(['message' => '角色不正確'], 400);
                return;
            }

            if (empty($data['username'])) {
                json_response(['message' => '「帳號」為必要欄位'], 400);
                return;
            }
            if (empty($data['name'])) {
                json_response(['message' => '「姓名」為必要欄位'], 400);
                return;
            }

            // 建立權限：
            if ($newRole === 'admin') {
                // 僅 typing 可建立管理者
                if ($currentUsername !== 'typing') {
                    json_response(['message' => '僅 typing 可建立管理者帳號'], 403);
                    return;
                }
            } elseif ($newRole === 'teacher') {
                // 僅 admin 可建立教師
                if ($currentRole !== 'admin') {
                    json_response(['message' => '僅管理者可建立教師帳號'], 403);
                    return;
                }
            } else { // student
                if (!in_array($currentRole, ['admin', 'teacher'], true)) {
                    json_response(['message' => '權限不足：僅管理者或教師可建立學生帳號'], 403);
                    return;
                }
            }

            // 處理班級
            $class_json  = null;
            $class_data  = $data['class'] ?? [];
            $class_array = is_array($class_data)
                ? $class_data
                : ((string)$class_data === '' ? [] : [$class_data]);

            if ($newRole === 'student') {
                if (count($class_array) > 1) {
                    json_response(['message' => '學生僅能指定一個班級'], 400);
                    return;
                }
            } else {
                // 管理者 / 教師不需班級
                $class_array = [];
            }

            if (!empty($class_array)) {
                $class_array = array_map('intval', $class_array);
                $class_json  = json_encode($class_array);
            }

            // 教師建立學生時，只能指派到自己班級或未分班
            if ($currentRole === 'teacher' && $newRole === 'student' && !empty($class_array)) {
                $targetClass = (int)$class_array[0];
                if (!in_array($targetClass, $teacherClassIDs, true)) {
                    json_response(['message' => '您僅能將學生指派到您所屬班級'], 403);
                    return;
                }
            }

            // 年級
            $grade = null;
            if ($newRole === 'student') {
                if (isset($data['grade']) && $data['grade'] !== '') {
                    $grade = (int)$data['grade'];
                }
            }

            $is_active = isset($data['is_active']) ? (bool)$data['is_active'] : true;

            // 密碼：管理者 / 教師才有
            $passwordHash = null;
            if (in_array($newRole, ['admin', 'teacher'], true)) {
                $plainPwd = trim($data['password'] ?? '');
                if ($plainPwd === '') {
                    $plainPwd = $data['username'];
                }
                $passwordHash = password_hash($plainPwd, PASSWORD_DEFAULT);
            }

            $stmt = $pdo->prepare(
                "INSERT INTO users (username, name, class, role, password, grade, is_active)
                 VALUES (?, ?, ?, ?, ?, ?, ?)"
            );
            $stmt->execute([
                $data['username'],
                $data['name'],
                $class_json,
                $newRole,
                $passwordHash,
                $grade,
                $is_active ? 1 : 0,
            ]);

            $newId = (int)$pdo->lastInsertId();
            log_operation_safe($pdo, $currentUserId, $currentUsername, 'USER_CREATE');

            json_response(['id' => $newId, 'message' => 'User created successfully']);
            return;
        }

        // ===========================
        // PUT：更新帳號
        // ===========================
        case 'PUT': {

            $targetId = isset($_GET['id']) ? (int)$_GET['id'] : 0;
            if ($targetId <= 0) {
                json_response(['message' => '缺少 id'], 400);
                return;
            }

            $data = json_decode(file_get_contents('php://input'), true) ?? [];

            $stmt = $pdo->prepare(
                "SELECT id, username, name, class, role, grade, is_active 
                 FROM users WHERE id = ?"
            );
            $stmt->execute([$targetId]);
            $targetUser = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$targetUser) {
                json_response(['message' => '帳號不存在'], 404);
                return;
            }

            // ✅ typing 僅能由 typing 本人編輯
            if ($targetUser['username'] === 'typing' && $currentUsername !== 'typing') {
                json_response(['message' => 'typing 帳號僅能由本人編輯'], 403);
                return;
            }

            $newName     = $data['name'] ?? $targetUser['name'];
            $newIsActive = isset($data['is_active']) ? (bool)$data['is_active'] : (bool)$targetUser['is_active'];
            $newGrade    = $targetUser['grade'];

            // 班級
            $class_json  = $targetUser['class'];
            $class_data  = $data['class'] ?? null;
            $class_array = null;

            if ($class_data !== null) {
                $class_array = is_array($class_data)
                    ? $class_data
                    : ((string)$class_data === '' ? [] : [$class_data]);

                $class_array = array_map('intval', $class_array);

                if ($targetUser['role'] === 'student' && count($class_array) > 1) {
                    json_response(['message' => '學生僅能指定一個班級'], 400);
                    return;
                }

                $class_json = !empty($class_array) ? json_encode($class_array) : null;
            }

            // 年級只對學生有意義
            if ($targetUser['role'] === 'student' && isset($data['grade'])) {
                $newGrade = (int)$data['grade'];
            }

            // 權限檢查
            if ($currentRole === 'teacher') {

                if ($targetUser['role'] !== 'student') {
                    json_response(['message' => '教師僅能修改學生'], 403);
                    return;
                }

                // 教師只能修改自己班級或未分班的學生
                $tClass = [];
                if (!empty($targetUser['class'])) {
                    $tmp = json_decode($targetUser['class'], true);
                    if (is_array($tmp)) $tClass = $tmp;
                }

                $isUnassigned = empty($tClass);
                $inMyClass    = false;

                if (!$isUnassigned && !empty($teacherClassIDs)) {
                    foreach ($tClass as $cid) {
                        if (in_array((int)$cid, $teacherClassIDs, true)) {
                            $inMyClass = true;
                            break;
                        }
                    }
                }

                $newAssigned = null;
                if ($class_array !== null && !empty($class_array)) {
                    $newAssigned = (int)$class_array[0];
                }

                if ($isUnassigned) {
                    if ($newAssigned !== null && !in_array($newAssigned, $teacherClassIDs, true)) {
                        json_response(['message' => '您僅能將學生指派到您所屬的班級'], 403);
                        return;
                    }
                } elseif ($inMyClass) {
                    if ($newAssigned !== null && !in_array($newAssigned, $teacherClassIDs, true)) {
                        json_response(['message' => '您僅能將學生留在您所屬的班級'], 403);
                        return;
                    }
                } else {
                    json_response(['message' => '您無權修改此學生'], 403);
                    return;
                }

            } elseif ($currentRole === 'admin') {

                // 管理者帳號只能由 typing 修改
                if ($targetUser['role'] === 'admin' && $currentUsername !== 'typing') {
                    json_response(['message' => '僅 typing 可修改管理者帳號'], 403);
                    return;
                }

            } else {
                // student 無權限
                json_response(['message' => '權限不足'], 403);
                return;
            }

            $upd = $pdo->prepare(
                "UPDATE users SET name = ?, class = ?, grade = ?, is_active = ? WHERE id = ?"
            );
            $upd->execute([
                $newName,
                $class_json,
                ($targetUser['role'] === 'student' ? $newGrade : null),
                $newIsActive ? 1 : 0,
                $targetId,
            ]);

            log_operation_safe($pdo, $currentUserId, $currentUsername, 'USER_UPDATE');

            json_response(['message' => 'User updated successfully']);
            return;
        }

        // ===========================
        // DELETE：刪除帳號
        // ===========================
        case 'DELETE': {

            $targetId = isset($_GET['id']) ? (int)$_GET['id'] : 0;
            if ($targetId <= 0) {
                json_response(['message' => '缺少 id'], 400);
                return;
            }

            $stmt = $pdo->prepare("SELECT id, username, role, class FROM users WHERE id = ?");
            $stmt->execute([$targetId]);
            $targetUser = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$targetUser) {
                json_response(['message' => '帳號不存在'], 404);
                return;
            }

            // typing 不可被刪除
            if ($targetUser['username'] === 'typing') {
                json_response(['message' => 'typing 帳號不得刪除'], 403);
                return;
            }

            if ($currentRole === 'teacher') {

                if ($targetUser['role'] !== 'student') {
                    json_response(['message' => '教師僅能刪除學生'], 403);
                    return;
                }

                $tClass = [];
                if (!empty($targetUser['class'])) {
                    $tmp = json_decode($targetUser['class'], true);
                    if (is_array($tmp)) $tClass = $tmp;
                }

                $isUnassigned = empty($tClass);
                $inMyClass    = false;

                if (!$isUnassigned && !empty($teacherClassIDs)) {
                    foreach ($tClass as $cid) {
                        if (in_array((int)$cid, $teacherClassIDs, true)) {
                            $inMyClass = true;
                            break;
                        }
                    }
                }

                if (!($isUnassigned || $inMyClass)) {
                    json_response(['message' => '您無權刪除此學生'], 403);
                    return;
                }

            } elseif ($currentRole === 'admin') {

                // 管理者帳號只能由 typing 刪除
                if ($targetUser['role'] === 'admin' && $currentUsername !== 'typing') {
                    json_response(['message' => '僅 typing 可刪除管理者帳號'], 403);
                    return;
                }

            } else {
                json_response(['message' => '權限不足'], 403);
                return;
            }

            $del = $pdo->prepare("DELETE FROM users WHERE id = ?");
            $del->execute([$targetId]);

            log_operation_safe($pdo, $currentUserId, $currentUsername, 'USER_DELETE');

            json_response(['message' => 'User deleted successfully']);
            return;
        }

        // ===========================
        // 其他 HTTP 方法
        // ===========================
        default:
            json_response(['message' => '方法不允許'], 405);
            return;
    }

} catch (PDOException $e) {
    if (stripos($e->getCode(), '23000') !== false) {
        json_response(['message' => '錯誤：帳號 (username) 已存在'], 500);
    } else {
        json_response(['message' => '資料庫錯誤：' . $e->getMessage()], 500);
    }
    return;
} catch (Throwable $e) {
    json_response(['message' => '處理錯誤：' . $e->getMessage()], 400);
    return;
}

