<?php
session_start();
include 'db_connect.php';
include 'log_helper.php';

header('Content-Type: application/json; charset=utf-8');

$input    = json_decode(file_get_contents('php://input'), true);
$password = $input['password'] ?? '';

// 必須先通過第一階段（login.php）才會有 pending_user_id
if (empty($_SESSION['pending_user_id'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => '尚未通過驗證碼，請重新登入']);
    exit;
}

if (trim($password) === '') {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => '請輸入密碼']);
    exit;
}

try {
    $pdo = get_db_connection();

    $pendingId = (int)$_SESSION['pending_user_id'];

    $stmt = $pdo->prepare(
        "SELECT id, username, name, role, grade, class, is_active, password 
         FROM users 
         WHERE id = ?"
    );
    $stmt->execute([$pendingId]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$user) {
        unset($_SESSION['pending_user_id']);
        http_response_code(404);
        echo json_encode(['success' => false, 'message' => '帳號不存在']);
        exit;
    }

    if (!$user['is_active']) {
        unset($_SESSION['pending_user_id']);
        http_response_code(403);
        echo json_encode(['success' => false, 'message' => '此帳號已失效，無法登入']);
        exit;
    }

    if (!in_array($user['role'], ['admin','teacher'], true)) {
        // 理論上不會發生，但防呆
        unset($_SESSION['pending_user_id']);
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => '此帳號不需密碼登入，請重新操作']);
        exit;
    }

    if (empty($user['password'])) {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => '此帳號尚未設定密碼，請聯絡系統管理者']);
        exit;
    }

    if (!password_verify($password, $user['password'])) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => '密碼錯誤']);
        exit;
    }

    // 密碼正確，正式登入並移除 pending
    unset($_SESSION['pending_user_id']);

    $_SESSION['user'] = [
        'id'       => $user['id'],
        'username' => $user['username'],
        'name'     => $user['name'],
        'role'     => $user['role'],
        'grade'    => $user['grade'],
        'class'    => $user['class'],
    ];

    log_operation($pdo, $user['id'], $user['username'], 'USER_LOGIN');

    echo json_encode([
        'success' => true,
        'role'    => $user['role'],
        'message' => '登入成功'
    ]);
} catch (\PDOException $e) {
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => '資料庫錯誤：' . $e->getMessage()]);
}

