<?php
// api/typing_practice.php
// 記錄打字練習成績 + 報表（個人、班級、年級、各年級）
// 使用 get_db_connection()；不使用 exit，統一回傳輸出

declare(strict_types=1);
header('Content-Type: application/json; charset=utf-8');

require_once __DIR__ . '/auth.php';
require_once __DIR__ . '/db_connect.php';

function run() {
  $method = strtoupper($_SERVER['REQUEST_METHOD'] ?? 'GET');
  $action = isset($_GET['action']) ? trim((string)$_GET['action']) : '';

  if (!auth_is_logged_in()) return [401, ['success'=>false,'message'=>'未登入']];
  $u = auth_current_user();
  $userId   = (int)($u['id'] ?? 0);
  $username = (string)($u['username'] ?? ($u['account'] ?? ''));
  $className= isset($u['class_name']) ? (string)$u['class_name'] : (isset($u['class']) ? (string)$u['class'] : null);

  try {
    $pdo = get_db_connection();
  } catch (Throwable $e) {
    return [500, ['success'=>false,'message'=>'資料庫連線失敗：'.$e->getMessage()]];
  }

  if ($method === 'POST') {
    $raw = file_get_contents('php://input');
    $input = json_decode($raw ?: 'null', true);
    if (!$input) return [400, ['success'=>false,'message'=>'無法解析送出的資料']];

    $csrfErr = csrf_verify_from_request($input);
    if ($csrfErr) return $csrfErr;

    $mode   = ($input['mode'] ?? '') === 'timed' ? 'timed' : 'measured';
    $grade  = max(1, min(12, (int)($input['grade'] ?? 3)));
    $target = max(1, (int)($input['target_quota'] ?? 0));
    $startedAt = (string)($input['started_at'] ?? '');
    $endedAt   = (string)($input['ended_at']   ?? '');
    $qText = trim((string)($input['question_text'] ?? ''));
    $tText = trim((string)($input['typed_text']    ?? ''));
    $correct = max(0, (int)($input['correct_chars'] ?? 0));
    $wrong   = max(0, (int)($input['wrong_chars']   ?? 0));
    $endReason = in_array(($input['end_reason'] ?? ''), ['button','auto'], true) ? $input['end_reason'] : 'auto';

    if (!$startedAt || !$endedAt || !$qText) {
      return [400, ['success'=>false,'message'=>'欄位不足：時間或題目內容缺失']];
    }

    $durSec = max(1, (int)round((strtotime($endedAt) - strtotime($startedAt))));
    $wpm = $durSec > 0 ? round(($correct / $durSec) * 60, 2) : 0.0;

    try {
      $stmt = $pdo->prepare("
        INSERT INTO typing_practice_logs
          (user_id, username, class_name, grade, mode, target_quota, started_at, ended_at,
           duration_seconds, question_text, typed_text, correct_chars, wrong_chars, wpm, end_reason)
        VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)
      ");
      $stmt->execute([
        $userId, $username, $className, $grade, $mode, $target, $startedAt, $endedAt,
        $durSec, $qText, $tText, $correct, $wrong, $wpm, $endReason
      ]);
      return [200, ['success'=>true, 'id'=>(int)$pdo->lastInsertId(), 'wpm'=>$wpm]];
    } catch (Throwable $e) {
      return [500, ['success'=>false,'message'=>'伺服器錯誤：'.$e->getMessage()]];
    }
  }

  // 日期區間
  $start = $_GET['start'] ?? null; // YYYY-MM-DD
  $end   = $_GET['end']   ?? null; // YYYY-MM-DD
  if ($start && !$end) $end = $start;
  if ($start && $end) { $start .= ' 00:00:00'; $end .= ' 23:59:59'; }

  // 個人
  if ($method === 'GET' && $action === 'personal') {
    $sqlBase = "FROM typing_practice_logs WHERE user_id = :uid";
    $params = [':uid'=>$userId];
    if ($start && $end) { $sqlBase .= " AND started_at BETWEEN :s AND :e"; $params[':s']=$start; $params[':e']=$end; }

    $rows = $pdo->prepare("
      SELECT id, started_at, ended_at, mode, grade, target_quota,
             correct_chars, wrong_chars, wpm, end_reason
      $sqlBase ORDER BY started_at DESC
    ");
    $rows->execute($params);
    $list = $rows->fetchAll();

    $stat = $pdo->prepare("
      SELECT COUNT(*) AS cnt, ROUND(AVG(wpm),2) AS avg_wpm, MAX(wpm) AS max_wpm, MIN(wpm) AS min_wpm
      $sqlBase
    ");
    $stat->execute($params);
    $summary = $stat->fetch() ?: ['cnt'=>0,'avg_wpm'=>null,'max_wpm'=>null,'min_wpm'=>null];

    $seriesStmt = $pdo->prepare("
      SELECT UNIX_TIMESTAMP(started_at) AS ts, wpm
      $sqlBase ORDER BY started_at ASC
    ");
    $seriesStmt->execute($params);
    $series = $seriesStmt->fetchAll();

    return [200, ['success'=>true, 'list'=>$list, 'summary'=>$summary, 'series'=>$series]];
  }

  // 班級
  if ($method === 'GET' && $action === 'class_rank') {
    $class = $_GET['class'] ?? $className;
    if (!$class) return [400, ['success'=>false,'message'=>'查無班級資訊']];

    $sqlBase = "FROM typing_practice_logs WHERE class_name = :c";
    $params = [':c'=>$class];
    if ($start && $end) { $sqlBase .= " AND started_at BETWEEN :s AND :e"; $params[':s']=$start; $params[':e']=$end; }

    $stmt = $pdo->prepare("
      SELECT username, user_id, ROUND(AVG(wpm),2) AS avg_wpm, COUNT(*) AS cnt
      $sqlBase GROUP BY username, user_id ORDER BY avg_wpm DESC
    ");
    $stmt->execute($params);
    $rows = $stmt->fetchAll();

    $avgStmt = $pdo->prepare("SELECT ROUND(AVG(wpm),2) AS class_avg $sqlBase");
    $avgStmt->execute($params);
    $classAvg = $avgStmt->fetchColumn();

    foreach ($rows as $i=>$r) $rows[$i]['rank'] = $i+1;

    return [200, ['success'=>true, 'class'=>$class, 'average'=>$classAvg, 'rank'=>$rows]];
  }

  // 年級各班
  if ($method === 'GET' && $action === 'grade_classes') {
    $grade = max(1, min(12, (int)($_GET['grade'] ?? 0)));
    if ($grade <= 0) return [400, ['success'=>false,'message'=>'請提供年級']];

    $sqlBase = "FROM typing_practice_logs WHERE grade = :g";
    $params = [':g'=>$grade];
    if ($start && $end) { $sqlBase .= " AND started_at BETWEEN :s AND :e"; $params[':s']=$start; $params[':e']=$end; }

    $stmt = $pdo->prepare("
      SELECT class_name, ROUND(AVG(wpm),2) AS avg_wpm, COUNT(*) AS cnt
      $sqlBase GROUP BY class_name ORDER BY avg_wpm DESC
    ");
    $stmt->execute($params);
    $rows = $stmt->fetchAll();
    foreach ($rows as $i=>$r) $rows[$i]['rank'] = $i+1;

    return [200, ['success'=>true, 'grade'=>$grade, 'classes'=>$rows]];
  }

  // 各年級
  if ($method === 'GET' && $action === 'grades_overall') {
    $sqlBase = "FROM typing_practice_logs WHERE 1=1";
    $params = [];
    if ($start && $end) { $sqlBase .= " AND started_at BETWEEN :s AND :e"; $params[':s']=$start; $params[':e']=$end; }

    $stmt = $pdo->prepare("
      SELECT grade, ROUND(AVG(wpm),2) AS avg_wpm, COUNT(*) AS cnt
      $sqlBase GROUP BY grade ORDER BY grade ASC
    ");
    $stmt->execute($params);
    $rows = $stmt->fetchAll();

    $rank = $rows; usort($rank, fn($a,$b)=>($b['avg_wpm']<=>$a['avg_wpm']));
    foreach ($rank as $i=>$r) $rank[$i]['rank'] = $i+1;

    return [200, ['success'=>true, 'grades'=>$rows, 'rank_by_avg'=>$rank]];
  }

  return [405, ['success'=>false,'message'=>'不支援的請求']];
}

[$code, $body] = run();
http_response_code($code);
if ($body !== null) echo json_encode($body, JSON_UNESCAPED_UNICODE);

