// 打字練習頁 JS
(function() {
    // 獲取 DOM 元素
    const textContent = document.getElementById("text-content");
    const typingInput = document.getElementById("typing-input");
    const bopomofoSwitch = document.getElementById("bopomofo-switch");
    const stopButton = document.getElementById("stop-test-btn");
    const timerDisplay = document.getElementById("timer");
    const charCountDisplay = document.getElementById("char-count");
    const errorCountDisplay = document.getElementById("error-count");
    const wpmDisplay = document.getElementById("wpm");

    // 檢查元素是否存在
    if (!typingInput) return;

    // 模擬從題庫載入的題目
    const sampleText = [
        "白日依山盡，黃河入海流。",
        "欲窮千里目，更上一層樓。",
        "紅豆生南國，春來發幾枝？",
        "願君多采擷，此物最相思。"
    ];

    let currentLineIndex = 0;
    let totalChars = 0;
    let totalErrors = 0;
    let startTime = null;
    let timerInterval = null;

    // 1. 載入題目
    function loadTest() {
        textContent.innerHTML = ""; 
        sampleText.forEach((line, index) => {
            const lineEl = document.createElement("div");
            lineEl.className = "text-line";
            lineEl.textContent = line;
            if (index === 0) {
                lineEl.classList.add("current");
            }
            textContent.appendChild(lineEl);
        });
        typingInput.focus();
    }

    // 2. 禁用複製功能 [cite: 10]
    function disableCopyPaste(element) {
        element.addEventListener('contextmenu', e => e.preventDefault()); 
        element.addEventListener('keydown', e => {
            if (e.ctrlKey && (e.key === 'c' || e.key === 'v' || e.key === 'x')) {
                e.preventDefault();
            }
        });
        element.addEventListener('paste', e => e.preventDefault());
    }

    // 3. 切換注音字型 [cite: 40]
    bopomofoSwitch.addEventListener('change', (e) => {
        if (e.target.checked) {
            textContent.classList.add('bopomofo');
        } else {
            textContent.classList.remove('bopomofo');
        }
    });

    // 4. 處理輸入
    typingInput.addEventListener('keydown', (e) => {
        if (startTime === null && e.key.length === 1) { 
            startTimer();
        }
        if (e.key === 'Enter' && currentLineIndex < sampleText.length) {
            processLine();
        }
    });

    // 5. 綁定停止鈕
    stopButton.addEventListener('click', () => {
        finishTest(); // [cite: 51, 57, 60]
    });

    // 6. 處理換行邏輯
    function processLine() {
        const lines = textContent.querySelectorAll('.text-line');
        const currentLineEl = lines[currentLineIndex];
        const currentLineText = currentLineEl.textContent;
        const userInput = typingInput.value;

        let errors = 0;
        for (let i = 0; i < currentLineText.length; i++) {
            if (i >= userInput.length || userInput[i] !== currentLineText[i]) {
                errors++;
            }
        }
        totalChars += currentLineText.length; 
        totalErrors += errors;

        charCountDisplay.textContent = totalChars;
        errorCountDisplay.textContent = totalErrors;
        
        typingInput.value = ""; // 清空輸入框 [cite: 41]
        currentLineEl.classList.remove('current');
        currentLineIndex++;

        if (currentLineIndex < sampleText.length) {
            const nextLineEl = lines[currentLineIndex];
            nextLineEl.classList.add('current');
            const lineHeight = currentLineEl.offsetHeight; 
            textContent.style.transform = `translateY(-${currentLineIndex * lineHeight}px)`; // 自動上捲 [cite: 41]
        } else {
            finishTest();
        }
    }

    // 7. 計時器與 WPM 計算
    function startTimer() {
        startTime = Date.now();
        timerInterval = setInterval(() => {
            const elapsedTime = Date.now() - startTime;
            const seconds = Math.floor(elapsedTime / 1000);
            
            const displayMin = String(Math.floor(seconds / 60)).padStart(2, '0');
            const displaySec = String(seconds % 60).padStart(2, '0');
            timerDisplay.textContent = `${displayMin}:${displaySec}`; // [cite: 43]
            
            const minutes = seconds / 60;
            if (minutes > 0) {
                const correctChars = totalChars - totalErrors;
                const wpm = (correctChars / minutes).toFixed(1); // [cite: 46]
                wpmDisplay.textContent = wpm;
            }
        }, 1000);
    }

    // 8. 結束測驗
    function finishTest() {
        if (!timerInterval && startTime === null) return; // 未開始時按結束
        
        clearInterval(timerInterval);
        timerInterval = null; 
        typingInput.disabled = true;
        typingInput.placeholder = "測驗完成！";
        stopButton.disabled = true;
        stopButton.textContent = "已結束";
        
        console.log("測驗完成");
    }

    // 初始載入
    loadTest();
    disableCopyPaste(typingInput); // [cite: 10]

})();
