<?php
session_start();
include 'db_connect.php'; // Include your database connection
// (Optional: Include log_helper.php if you want unified logging)
// include 'log_helper.php';

// Enable error logging
ini_set('display_errors', 0); // Disable display for production
error_reporting(E_ALL);
// ini_set('log_errors', 1); // Ensure errors are logged
// ini_set('error_log', '/path/to/your/php-error.log'); // Specify log file path if needed

header('Content-Type: application/json');

// --- 1. Access Control (Optional but recommended) ---
if (!isset($_SESSION['user'])) {
    http_response_code(401);
    echo json_encode(['message' => '未授權，請先登入']);
    exit;
}
$currentUser = $_SESSION['user']; // Get user info if needed later

// --- 2. Get Parameters ---
$level = filter_input(INPUT_GET, 'level', FILTER_VALIDATE_INT, [
    'options' => ['min_range' => 3, 'max_range' => 6]
]);

// Default level if not provided or invalid
if ($level === false || $level === null) {
    $level = $currentUser['grade'] ?? 3; // Use user's grade or default to 3
    if ($level < 3 || $level > 6) { $level = 3; }
}

error_log("[get_questions] Requested level: {$level}"); // Log requested level

// How many questions to fetch
$limit = 10; // Fetch 10 random questions

try {
    $pdo = get_db_connection();

    // --- 3. Query Database (Attempt specific level first) ---
    $sql = "SELECT id, content FROM test_bank WHERE level = ? ORDER BY RAND() LIMIT ?";
    $stmt = $pdo->prepare($sql);
    $stmt->execute([$level, $limit]);
    $questions = $stmt->fetchAll(PDO::FETCH_ASSOC);
    $count = count($questions); // Get the count

    error_log("[get_questions] Found {$count} questions for level {$level}."); // Log count for specific level

    // (新) Fallback: If no questions found for the specific level, try fetching any level
    if ($count === 0) {
        error_log("[get_questions] No questions found for level {$level}. Attempting to fetch from any level...");
        $sqlFallback = "SELECT id, content FROM test_bank ORDER BY RAND() LIMIT ?";
        $stmtFallback = $pdo->prepare($sqlFallback);
        $stmtFallback->execute([$limit]);
        $questions = $stmtFallback->fetchAll(PDO::FETCH_ASSOC);
        $fallbackCount = count($questions);
        error_log("[get_questions] Fallback fetch found {$fallbackCount} questions from any level."); // Log fallback count
    }

    // --- 4. Return JSON ---
    // If questions is still empty here, it means the test_bank might be completely empty.
    echo json_encode($questions, JSON_UNESCAPED_UNICODE); // Ensure correct encoding

} catch (\PDOException $e) {
    error_log("[get_questions] PDOException: " . $e->getMessage()); // Log DB error
    http_response_code(500);
    echo json_encode(['message' => '無法取得題目資料庫資料，請稍後再試']);
} catch (\Exception $e) {
    error_log("[get_questions] General Exception: " . $e->getMessage()); // Log other errors
    http_response_code(500);
    echo json_encode(['message' => '取得題目時發生伺服器錯誤']);
}

