document.addEventListener("DOMContentLoaded", () => {
    // --- DOM Elements ---
    const mainContainer = document.getElementById("main-container");
    const nav = document.getElementById("main-nav");
    const usernameDisplay = document.getElementById("username-display");
    const logoutBtn = document.getElementById("logout-btn");

    // Store initial dashboard content to restore when navigating back
    const dashboardHomeContent = mainContainer.innerHTML;

    // --- Core Functions ---

    // 1. Check Login Status (includes storing user role/id/class)
    async function checkLoginStatus() {
        try {
            const response = await fetch('api/check-session.php');
            // If not logged in (e.g., 401 Unauthorized), redirect to index/login
            if (!response.ok) {
                window.location.href = 'index.html'; // Redirect to index (which redirects to login)
                return; // Stop further execution
            }
            const data = await response.json(); // Get user data from session

            // Store user data globally for other scripts (like users.js, testbank.js)
            window.currentUserId = data.id;
            window.currentUserRole = data.role;
            window.currentUsername = data.username;
            window.currentUserClass = data.class; // Stores class JSON string (e.g., "[1,2]" or "[1]" or null)

            // Display username in the header
            if (usernameDisplay) {
                 usernameDisplay.textContent = data.username;
            } else {
                 console.warn("Username display element not found.");
            }


            // Show/Hide Nav Links based on Role
            const adminLinks = document.querySelectorAll('.admin-link'); // Only for admin (e.g., Class Management)
            const adminTeacherLinks = document.querySelectorAll('.admin-teacher-link'); // For admin & teacher (e.g., Account, Test Bank)

            adminLinks.forEach(link => link.style.display = (data.role === 'admin') ? 'inline-block' : 'none');
            adminTeacherLinks.forEach(link => link.style.display = (data.role === 'admin' || data.role === 'teacher') ? 'inline-block' : 'none');

        } catch (err) {
            // Handle network errors or API failures during session check
            console.error("Session check failed, redirecting to login.", err);
            window.location.href = 'index.html'; // Redirect to index/login on error
        }
    }

    // 2. Load Dynamic Content based on nav link clicked
    async function loadContent(page) {
        // Clear previous dynamic script if it exists to prevent conflicts
        document.getElementById("dynamic-script")?.remove();

        // Highlight the currently active nav link
        nav.querySelectorAll('a').forEach(a => a.classList.remove('active'));
        nav.querySelector(`[data-page="${page}"]`)?.classList.add('active');

        try {
            let contentUrl; // URL of the HTML content file
            let scriptUrl;  // URL of the corresponding JS file

            // Determine which content and script to load based on the 'page' identifier
            switch (page) {
                case 'typing':
                    contentUrl = 'dashboard/typing-content.html';
                    scriptUrl = 'js/typing.js';
                    break;
                case 'admin-users':
                    contentUrl = 'dashboard/users-content.html';
                    scriptUrl = 'js/users.js';
                    break;
                case 'admin-classes':
                    contentUrl = 'dashboard/classes-content.html';
                    scriptUrl = 'js/classes.js';
                    break;
                case 'admin-testbank': // Added route for testbank
                    contentUrl = 'dashboard/testbank-content.html';
                    scriptUrl = 'js/testbank.js';
                    break;
                case 'dashboard':
                    // Restore the initial dashboard content (announcements)
                    mainContainer.innerHTML = dashboardHomeContent;
                    // Re-bind username after resetting innerHTML
                    const userDisplay = mainContainer.querySelector('#username-display');
                    if (userDisplay) userDisplay.textContent = window.currentUsername || '...';
                    return; // Done for dashboard home
                default:
                    // Handle placeholder pages like 'test', 'competition'
                    mainContainer.innerHTML = `<h2>${escapeHTML(page)} 頁面 (建置中)</h2>`;
                    return; // Done for placeholder pages
            }

            // Fetch the HTML content file
            const response = await fetch(contentUrl);
            if (!response.ok) throw new Error(`內容載入失敗: ${contentUrl} (${response.status} ${response.statusText})`);
            // Inject the fetched HTML into the main container
            mainContainer.innerHTML = await response.text();

            // If a corresponding script exists, load and append it to the body
            if (scriptUrl) {
                const script = document.createElement('script');
                script.src = scriptUrl;
                script.id = 'dynamic-script'; // Mark for removal on next navigation
                document.body.appendChild(script);
            }

        } catch (err) {
            // Display error message if content loading fails
            console.error("Error loading content:", err);
            mainContainer.innerHTML = `<h2>內容載入錯誤</h2><p>${escapeHTML(err.message)}</p>`;
        }
    }

    // 3. Handle Nav Clicks (Event Delegation on nav element)
    nav.addEventListener('click', (e) => {
        // Handle Logout button separately
        if (e.target.id === 'logout-btn') {
            e.preventDefault(); // Prevent default link behavior
            logout(); // Call logout function
            return;
        }
        // Handle SPA content switching for links with data-page attribute
        if (e.target.tagName === 'A' && e.target.dataset.page) {
            e.preventDefault(); // Prevent default link behavior
            const page = e.target.dataset.page; // Get the target page identifier
            loadContent(page); // Load the corresponding content
        }
    });

    // 4. Logout Function
    async function logout() {
        try {
            // Call the logout API endpoint
            await fetch('api/logout.php');
        } catch (err) {
            // Log error but proceed with redirect anyway
            console.error('Logout request failed', err);
        } finally {
            // Always redirect to index.html after attempting logout
            window.location.href = 'index.html'; // index.html handles redirect to login if needed
        }
    }

     // 5. HTML Escape Utility (Simple version)
     function escapeHTML(str) {
        if (str === null || str === undefined) return '';
        return str.toString()
                  .replace(/&/g, '&amp;')
                  .replace(/</g, '&lt;')
                  .replace(/>/g, '&gt;')
                  .replace(/"/g, '&quot;')
                  .replace(/'/g, '&#39;');
    }

    // --- Initial Load ---
    checkLoginStatus(); // Check login status and set up nav links visibility on page load
    // Activate the 'dashboard' nav link visually by default
    nav.querySelector(`[data-page="dashboard"]`)?.classList.add('active');
    // Note: The initial dashboard content (announcements) is already present in dashboard.html

});
