// 中文練習頁面的 JS 邏輯 (重寫以符合 DOCX 需求 + 修正)
(function() {
    // --- DOM Elements ---
    const modeMeasuredRadio = document.getElementById('mode-measured');
    const modeTimedRadio = document.getElementById('mode-timed');
    const levelSelect = document.getElementById('level-select');
    const measuredSettings = document.getElementById('measured-settings');
    const timedSettings = document.getElementById('timed-settings');
    const charCountSelect = document.getElementById('char-count-select');
    const timeDurationSelect = document.getElementById('time-duration-select');
    const fontNoZhuyinRadio = document.getElementById('font-no-zhuyin');
    const fontZhuyinRadio = document.getElementById('font-zhuyin');
    const stopTestBtn = document.getElementById('stop-test-btn');
    const timerDisplay = document.getElementById('timer-display');
    const charsTypedDisplay = document.getElementById('chars-typed-display');
    const charsTargetDisplay = document.getElementById('chars-target-display');
    const errorsDisplay = document.getElementById('errors-display');
    const speedDisplay = document.getElementById('speed-display');
    const statusMessage = document.getElementById('status-message');
    const textDisplayWindow = document.getElementById('text-display-window');
    const typingInput = document.getElementById('typing-input');

    // --- State Variables ---
    let practiceMode = 'measured'; let targetLevel = 3; let targetCount = 50; let targetTime = 10;
    let useZhuyinFont = false; let questions = []; let currentLineIndex = 0; let fullText = '';
    let totalCharsCompleted = 0; let displayTypedChars = 0; let totalErrors = 0;
    let timerInterval = null; let startTime = null; let timeElapsed = 0;
    let practiceActive = false; let practiceFinished = false; let isLoadingQuestions = false; let hasStarted = false;
    let currentLineHeight = 0;

    // --- Element Existence Check ---
    if (!modeMeasuredRadio || !typingInput || !textDisplayWindow || !statusMessage /* ... etc ... */ ) {
        console.error("Typing Practice: Missing required DOM elements.");
        const container = document.getElementById('main-container');
        if(container) { container.innerHTML = '<h2 class="error">練習頁面載入錯誤...</h2>'; }
        else { alert("練習頁面載入錯誤..."); }
        return;
    }

    // --- Utility Functions ---
    function escapeHTML(str) { if (str === null || str === undefined) return ''; return str.toString().replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/"/g, '&quot;').replace(/'/g, '&#39;'); }
    function showStatusMessage(text, isError = false) { if (!statusMessage) return; statusMessage.textContent = text; statusMessage.classList.toggle('error', isError); }

    // --- Initialization ---
    function initializePractice() { console.log("Initializing Practice..."); resetState(); setupEventListeners(); if(window.currentUserGrade && window.currentUserGrade >= 3 && window.currentUserGrade <= 6) { levelSelect.value = window.currentUserGrade; } else { levelSelect.value = 3; } handleSettingsChange(false); typingInput.disabled = false; showStatusMessage("請設定練習模式並將游標移至下方輸入框開始..."); textDisplayWindow.innerHTML = '<div class="line upcoming" style="top: 0px;">請設定練習模式並開始...</div>'; handleFontChange(); }

    // --- State Management ---
    function resetState() { practiceActive = false; practiceFinished = false; isLoadingQuestions = false; hasStarted = false; clearInterval(timerInterval); timerInterval = null; startTime = null; timeElapsed = 0; currentLineIndex = 0; totalCharsCompleted = 0; totalErrors = 0; displayTypedChars = 0; questionLines = []; fullText = ''; timerDisplay.textContent = '00:00'; charsTypedDisplay.textContent = '0'; errorsDisplay.textContent = '0'; speedDisplay.textContent = '0.0'; stopTestBtn.disabled = true; typingInput.value = ''; typingInput.disabled = false; showStatusMessage("請設定練習模式並將游標移至下方輸入框開始..."); textDisplayWindow.innerHTML = '<div class="line upcoming" style="top: 0px;">請設定練習模式並開始...</div>'; textDisplayWindow.style.transform = 'translateY(0px)'; currentLineHeight = 0; updateTargetDisplay(); }

    // --- Event Listeners ---
    function setupEventListeners() { modeMeasuredRadio.addEventListener('change', () => handleSettingsChange(true)); modeTimedRadio.addEventListener('change', () => handleSettingsChange(true)); levelSelect.addEventListener('change', () => handleSettingsChange(true)); charCountSelect.addEventListener('change', () => handleSettingsChange(true)); timeDurationSelect.addEventListener('change', () => handleSettingsChange(true)); fontNoZhuyinRadio.addEventListener('change', handleFontChange); fontZhuyinRadio.addEventListener('change', handleFontChange); typingInput.addEventListener('focusin', startPractice); typingInput.addEventListener('keydown', handleKeyDown); typingInput.addEventListener('input', updateDisplayCharCount); stopTestBtn.addEventListener('click', finishPractice); typingInput.addEventListener('paste', e => e.preventDefault()); typingInput.addEventListener('copy', e => e.preventDefault()); typingInput.addEventListener('cut', e => e.preventDefault()); typingInput.addEventListener('contextmenu', e => e.preventDefault()); }

    // --- Settings Update ---
    function handleSettingsChange(shouldReset = true) { practiceMode = document.querySelector('input[name="practice-mode"]:checked').value; targetLevel = parseInt(levelSelect.value, 10); targetCount = parseInt(charCountSelect.value, 10); targetTime = parseInt(timeDurationSelect.value, 10); console.log("Settings Changed:", { practiceMode, targetLevel, targetCount, targetTime }); updateSettingsUI(); if (shouldReset) { resetState(); } }
    function updateSettingsUI() { measuredSettings.style.display = (practiceMode === 'measured') ? 'flex' : 'none'; timedSettings.style.display = (practiceMode === 'timed') ? 'flex' : 'none'; updateTargetDisplay(); }
    function updateTargetDisplay() { charsTargetDisplay.textContent = (practiceMode === 'measured') ? ` / ${targetCount}` : ''; charsTargetDisplay.style.display = (practiceMode === 'measured') ? 'inline' : 'none'; }
    function handleFontChange() { useZhuyinFont = fontZhuyinRadio.checked; textDisplayWindow.classList.toggle('zhuyin-font', useZhuyinFont); currentLineHeight = 0; if(practiceActive || hasStarted) renderTextDisplay(); }

    // --- Practice Logic ---
    // Start Practice - Fetches questions FIRST
    async function startPractice() {
        if (practiceActive || practiceFinished || hasStarted || isLoadingQuestions) return;
        console.log("Focus detected, attempting to start practice...");
        hasStarted = true; showStatusMessage('載入題目中...', false);
        textDisplayWindow.innerHTML = '<div class="line upcoming" style="top: 0px;">載入題目中...</div>';
        typingInput.disabled = true; isLoadingQuestions = true; stopTestBtn.disabled = true;

        try {
            await fetchMoreQuestions(); // Fetch initial questions

            // (修正) Check questionLines instead of fullText directly for initial load
            if (questionLines.length === 0) {
                // Throw specific error based on fetchMoreQuestions internal state if possible
                throw new Error('無法載入題目，請確認該程度題庫內容。'); // More specific message
            }

            // --- Questions Loaded Successfully ---
            console.log("Questions loaded, Starting Practice Now...");
            isLoadingQuestions = false; practiceActive = true; practiceFinished = false;
            stopTestBtn.disabled = false; typingInput.disabled = false;

            showStatusMessage('練習中...', false);
            renderTextDisplay(); // Render initial lines and highlight
            startTime = Date.now(); timeElapsed = 0;
            startTimer(); // Start the timer
            typingInput.focus(); // Explicitly focus after enabling

        } catch (error) {
            console.error("Failed to start practice:", error);
            // Display the specific error message thrown by fetchMoreQuestions or the check above
            showStatusMessage(`開始練習失敗：${error.message}`, true);
            practiceActive = false; practiceFinished = true; isLoadingQuestions = false;
            typingInput.disabled = true; stopTestBtn.disabled = true;
            hasStarted = false; // Allow retry on next focus
        }
    }

    // Start/Manage Timer interval
    function startTimer() { console.log("startTimer called."); if (!practiceActive) { console.warn("startTimer called but practice not active."); return; } clearInterval(timerInterval); timeElapsed = 0; updateTimerDisplay(); updateSpeedDisplay(); timerInterval = setInterval(() => { if (!practiceActive) { clearInterval(timerInterval); return; } timeElapsed = Math.floor((Date.now() - startTime) / 1000); updateTimerDisplay(); updateSpeedDisplay(); if (practiceMode === 'timed' && timeElapsed >= targetTime) { finishPractice(); } }, 1000); }
    // Update Timer Display
    function updateTimerDisplay() { let displayTime = (practiceMode === 'timed') ? Math.max(0, targetTime - timeElapsed) : timeElapsed; const prefix = (practiceMode === 'timed') ? '剩餘 ' : ''; const minutes = String(Math.floor(displayTime / 60)).padStart(2, '0'); const seconds = String(displayTime % 60).padStart(2, '0'); timerDisplay.textContent = prefix + `${minutes}:${seconds}`; }
    // Update Speed Display
     function updateSpeedDisplay() { const minutes = timeElapsed / 60; let wpm = 0; if (minutes > 0) { const correctChars = Math.max(0, totalCharsCompleted - totalErrors); wpm = (correctChars / minutes).toFixed(1); } speedDisplay.textContent = wpm; }
     // Update Character Count Display (Real-time)
     function updateDisplayCharCount() { if (practiceActive) { const currentInputLength = typingInput.value.length; displayTypedChars = totalCharsCompleted + currentInputLength; charsTypedDisplay.textContent = displayTypedChars; } else if (!hasStarted) { charsTypedDisplay.textContent = '0'; } }

     // Handle KeyDown - Process Enter key for line submission
     async function handleKeyDown(event) {
          if (!practiceActive || practiceFinished || isLoadingQuestions) return;
          if (event.key === 'Enter') {
              event.preventDefault(); console.log("Enter pressed, processing line:", currentLineIndex);
              const inputLine = typingInput.value; const expectedLine = questionLines[currentLineIndex];
              if (expectedLine === undefined) { console.warn("Enter pressed but no expected line found, waiting for fetch?"); return; }
              let lineErrors = 0; for (let i = 0; i < expectedLine.length; i++) { if (i >= inputLine.length || inputLine[i] !== expectedLine[i]) { lineErrors++; } } if (inputLine.length > expectedLine.length) { lineErrors += (inputLine.length - expectedLine.length); }
              totalCharsCompleted += expectedLine.length; totalErrors += lineErrors;
              console.log(`Line ${currentLineIndex}: Expected='${expectedLine}', Input='${inputLine}', Errors=${lineErrors}, TotalChars=${totalCharsCompleted}, TotalErrors=${totalErrors}`);
              charsTypedDisplay.textContent = totalCharsCompleted; errorsDisplay.textContent = totalErrors; updateSpeedDisplay();
              currentLineIndex++; typingInput.value = ''; displayTypedChars = totalCharsCompleted; renderTextDisplay();
              if (practiceMode === 'measured' && totalCharsCompleted >= targetCount) { finishPractice(); return; }
              if (currentLineIndex >= questionLines.length - 2 && !isLoadingQuestions) { fetchMoreQuestions().catch(err => { console.error("Background fetch failed:", err); }); }
              if (currentLineIndex >= questionLines.length && !isLoadingQuestions) { console.warn("Reached end of available lines."); showStatusMessage("等待題目...", false); isLoadingQuestions = true; setTimeout(async () => { if(practiceActive && currentLineIndex >= questionLines.length) { try { await fetchMoreQuestions(); } catch(e){} if(currentLineIndex >= questionLines.length) { if(practiceMode === 'timed') showStatusMessage('題目似乎已載完，計時繼續...', false); else { console.log("No more lines available in measured mode. Finishing."); finishPractice(); } } else { renderTextDisplay(); showStatusMessage('練習中...', false); } } isLoadingQuestions = false; }, 100); }
          }
     }

    // Render Text Display - Line based with highlighting and scrolling
     function renderTextDisplay() {
         if (!textDisplayWindow) return; textDisplayWindow.innerHTML = '';
         if (!Array.isArray(questionLines) || questionLines.length === 0) { textDisplayWindow.innerHTML = `<div class="line upcoming" style="top: 0px;">${isLoadingQuestions ? '載入題目中...' : '(練習結束或無題目)'}</div>`; textDisplayWindow.style.transform = 'translateY(0px)'; return; }
         const displayStartLine = 0; const displayEndLine = questionLines.length;
         if (currentLineHeight <= 0 && questionLines.length > 0) {
             const tempDiv = document.createElement('div'); tempDiv.classList.add('line'); tempDiv.style.visibility = 'hidden'; tempDiv.style.position = 'absolute'; tempDiv.style.top = '0'; tempDiv.style.left = '0'; tempDiv.textContent = questionLines[0] || '測'; textDisplayWindow.appendChild(tempDiv); currentLineHeight = tempDiv.offsetHeight; textDisplayWindow.removeChild(tempDiv); console.log("Calculated line height:", currentLineHeight); if (currentLineHeight <= 0) currentLineHeight = 35; // Fallback
         }
         for (let i = displayStartLine; i < displayEndLine; i++) {
             const lineDiv = document.createElement('div'); lineDiv.classList.add('line'); lineDiv.textContent = questionLines[i] || '';
             lineDiv.style.top = `${i * currentLineHeight}px`; // Position absolutely
             if (i < currentLineIndex) { lineDiv.classList.add('typed-correct'); }
             else if (i === currentLineIndex) { lineDiv.classList.add('current'); lineDiv.style.fontWeight = 'bold'; }
             else { lineDiv.classList.add('upcoming'); }
             textDisplayWindow.appendChild(lineDiv);
         }
          if (currentLineHeight > 0) {
              const linesToShowAbove = 1;
              const targetScrollTop = Math.max(0, (currentLineIndex - linesToShowAbove) * currentLineHeight);
              textDisplayWindow.style.transform = `translateY(-${targetScrollTop}px)`;
          } else { textDisplayWindow.style.transform = 'translateY(0px)'; }
     }


    // Fetch More Questions from API - Adds to questionLines
    async function fetchMoreQuestions() {
        if (isLoadingQuestions && practiceActive) { console.log("[fetchMoreQuestions] Fetch already in progress."); return; }
        isLoadingQuestions = true;
        console.log(`[fetchMoreQuestions] Fetching for level ${targetLevel}...`);
        try {
            const response = await fetch(`api/get_questions.php?level=${targetLevel}`);
            if (!response.ok) { throw new Error(`API 請求失敗 (${response.status})`); }
            const contentType = response.headers.get("content-type"); if (!contentType || !contentType.includes("application/json")) { const text = await response.text(); console.error("Non-JSON response:", text); throw new Error(`API回應格式錯誤 (${contentType})`); }
            const newQuestions = await response.json();
            // (新) Log the raw response for inspection
            console.log("[fetchMoreQuestions] Raw API response:", JSON.stringify(newQuestions));

            // Ensure response is an array before processing
            if (!Array.isArray(newQuestions)) {
                console.error("[fetchMoreQuestions] API did not return a valid array:", newQuestions);
                throw new Error("API 回應的題目資料格式不正確");
            }

            if (newQuestions.length > 0) {
                // Process valid array
                newQuestions.forEach(q => {
                    // Check if q and q.content exist and q.content is a string
                    if (q && typeof q.content === 'string') {
                        const lines = q.content.split(/\r?\n/).filter(line => line.trim().length > 0);
                        questionLines = questionLines.concat(lines);
                    } else {
                        console.warn("[fetchMoreQuestions] Invalid question object or content:", q);
                    }
                });
                // Recalculate fullText based on potentially filtered lines
                fullText = questionLines.join('\n'); // Assuming newline separator for fullText state
                console.log("[fetchMoreQuestions] Total lines now:", questionLines.length, "Full text length:", fullText.length);
            } else {
                 console.warn("[fetchMoreQuestions] No new questions fetched from API.");
                 // Throw error only if it was the initial fetch and resulted in no lines
                 if(questionLines.length === 0 && !hasStarted) {
                      throw new Error('題庫中找不到此程度的題目。');
                 }
            }
        } catch (error) {
            console.error("[fetchMoreQuestions] Error fetching or processing questions:", error);
             // Display specific error based on context
             if (!practiceActive && !hasStarted) { showStatusMessage('錯誤：載入題目失敗 - ' + error.message, true); }
             else { console.error("Failed to fetch additional questions during practice."); }
             throw error; // Rethrow so caller knows it failed
        } finally {
             isLoadingQuestions = false; // Ensure flag is reset
        }
    }

    // Finish Practice
    function finishPractice() {
         if (practiceFinished) return; // Prevent multiple finishes
        console.log("Finishing Practice...");
        practiceActive = false; practiceFinished = true; // Mark as finished
        clearInterval(timerInterval); timerInterval = null; // Stop timer
        stopTestBtn.disabled = true; typingInput.disabled = true; // Disable controls
        const finalTime = timeElapsed; const finalChars = totalCharsCompleted; const finalErrors = totalErrors;
        const minutes = finalTime > 0 ? finalTime / 60 : 0; let finalWpm = 0;
        if (minutes > 0) { const correctChars = Math.max(0, finalChars - finalErrors); finalWpm = (correctChars / minutes).toFixed(1); }
        updateTimerDisplay(); // Show final time
        charsTypedDisplay.textContent = finalChars; // Display official count
        errorsDisplay.textContent = finalErrors; speedDisplay.textContent = finalWpm;
        showStatusMessage(`練習完成！ 速度: ${finalWpm} 字/分 (${finalChars}字/${finalErrors}錯)`, false);
        console.log("Practice Finished. Score:", { time: finalTime, chars: finalChars, errors: finalErrors, wpm: finalWpm });
    }

    // --- Initialization ---
    initializePractice(); // Run setup

})();
