// 帳號管理頁面的 JS 邏輯
(function() {
    // --- Global Variables & Constants ---
    const CURRENT_ROLE = window.currentUserRole;
    const CURRENT_USERNAME = window.currentUsername;
    const CURRENT_USER_ID = window.currentUserId;
    let ALL_CLASSES = []; // Cache for class list {id, class_name}

    // --- DOM Elements ---
    const form = document.getElementById("user-form");
    const addEditUserModal = document.getElementById("add-edit-user-modal");
    const showAddUserBtn = document.getElementById("show-add-user-btn");
    const closeUserModalBtn = document.getElementById("close-user-modal");
    const formTitle = document.getElementById("form-title");
    const userIdInput = document.getElementById("user-id");
    const usernameInput = document.getElementById("username");
    const nameInput = document.getElementById("name");
    const classSelect = document.getElementById("class"); // Class multi-select
    const classGroup = document.getElementById("class-group");
    const roleSelect = document.getElementById("role");
    const gradeGroup = document.getElementById("grade-group");
    const gradeSelect = document.getElementById("grade");
    const activeCheckbox = document.getElementById("is-active");
    const submitBtn = document.getElementById("submit-btn"); // Submit button for Add/Edit User
    const cancelBtn = document.getElementById("cancel-btn"); // Form cancel button
    const formMessage = document.getElementById("form-message"); // Message area inside add/edit form
    const showBatchBtn = document.getElementById("show-batch-upload-btn");
    const batchUploadModal = document.getElementById("batch-upload-modal");
    const closeBatchUploadBtn = document.getElementById("close-batch-upload-modal");
    const batchResultModal = document.getElementById("batch-result-modal");
    const closeBatchResultBtn = document.getElementById("close-batch-result-modal");
    const batchResultContent = document.getElementById("batch-result-content");
    const batchForm = document.getElementById("batch-upload-form");
    const batchFileInput = document.getElementById("account-file");
    const batchSubmitBtn = document.getElementById("batch-submit-btn"); // Submit button for Batch Upload
    const batchMessage = document.getElementById("batch-message"); // Message area inside batch form
    const tableBody = document.getElementById("user-table-body");
    const tableMessage = document.getElementById("table-message"); // Message area below user table
    const exportUsersBtn = document.getElementById("export-users-btn");

    // --- Element Existence Check ---
    // Check if all required elements were found
    if (!form || !batchForm || !addEditUserModal || !batchUploadModal || !batchResultModal || !showAddUserBtn || !showBatchBtn || !tableBody || !exportUsersBtn || !submitBtn || !tableMessage) {
         console.error("User management page: Missing one or more required DOM elements. Check HTML IDs.");
         // Optionally display an error to the user in a safe place
         const container = document.getElementById('main-container');
         if(container) container.innerHTML = '<h2 class="error">頁面初始化錯誤，請聯絡管理員。</h2>';
         return; // Stop script execution
    }

    // --- Utility Functions ---
    // 顯示訊息
    function showMessage(text, isError = false, target = formMessage) {
        if (!target) { console.warn("showMessage: Target element not found for message:", text); return; }
        target.textContent = text;
        target.className = isError ? 'form-message-area error' : 'form-message-area success';
        // Ensure result modal content and batch messages wrap correctly
        if (target === batchResultContent || target === batchMessage || target === tableMessage) {
             target.style.whiteSpace = 'pre-wrap';
        }
    }
    // HTML escape utility
    function escapeHTML(str) {
        if (str === null || str === undefined) return '';
        return str.toString().replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/"/g, '&quot;').replace(/'/g, '&#39;');
    }
    // Format Date/Time
    function formatDateTime(isoString) {
        if (!isoString) return 'N/A';
        try { const date = new Date(isoString); return date.toLocaleString('sv-SE', { year: 'numeric', month: '2-digit', day: '2-digit', hour: '2-digit', minute: '2-digit' }); } catch (e) { return isoString; }
    }

    // --- Modal Control Functions ---
    function openModal(modalElement) { if (modalElement) modalElement.style.display = 'flex'; }
    function closeModal(modalElement) {
         if (modalElement) modalElement.style.display = 'none';
         if (modalElement === addEditUserModal) resetForm(); // Reset Add/Edit form on close
         // Batch modals don't need reset on generic close
    }

    // --- Core Data Loading and Rendering ---
    // Load class list for dropdowns
    async function loadAllClasses() {
        try {
            const response = await fetch('api/classes.php?fetch=classlist');
            if (!response.ok) throw new Error('無法載入班級列表');
            ALL_CLASSES = await response.json();
            const selectedValues = Array.from(classSelect.selectedOptions).map(opt => opt.value); // Store selection
            classSelect.innerHTML = ''; // Clear existing options
            classSelect.add(new Option('(尚未分班)', '')); // Add default empty
            ALL_CLASSES.forEach(cls => { classSelect.add(new Option(cls.class_name, cls.id)); }); // Use ID as value
            // Restore selection(s)
            Array.from(classSelect.options).forEach(opt => { if (selectedValues.includes(opt.value)) { opt.selected = true; } });
        } catch (err) {
            console.error("Error loading class list:", err);
            showMessage("錯誤：無法載入班級選單 - " + err.message, true); // Show error in main form message area
        }
    }
    // Load user list
    async function loadUsers() {
        showMessage('', false, tableMessage); // Clear table message area
        if (!tableBody) return;
        tableBody.innerHTML = '<tr><td colspan="7">載入中...</td></tr>';
        try {
            const response = await fetch('api/users.php');
            if (!response.ok) { const errData = await response.json(); throw new Error(errData.message || '無法載入使用者'); }
             const contentType = response.headers.get("content-type"); // Check content type
             if (!contentType || !contentType.includes("application/json")) {
                 const errText = await response.text(); console.error("[loadUsers] Non-JSON response:", errText);
                 throw new Error(`伺服器回應格式錯誤 (${contentType})`);
             }
            const users = await response.json();
             if (!Array.isArray(users)) { throw new Error("API 回應的使用者資料不是有效的陣列"); } // Add check
            renderTable(users);
        } catch (err) {
            console.error("Error loading users:", err);
            tableBody.innerHTML = '<tr><td colspan="7" class="error">載入失敗，請檢查主控台錯誤訊息。</td></tr>';
            showMessage('載入使用者列表失敗: ' + err.message, true, tableMessage); // Show error below table
        }
    }
    // Render the user table
    function renderTable(users) {
        if (!tableBody) return; tableBody.innerHTML = '';
        if (!Array.isArray(users) || users.length === 0) { tableBody.innerHTML = '<tr><td colspan="7">尚無任何帳號</td></tr>'; return; }

        users.forEach((user, index) => {
             try {
                 if (typeof user !== 'object' || user === null) { throw new Error(`項目 ${index} 不是有效的物件`); } // Check object validity
                 const tr = document.createElement('tr'); tr.dataset.id = user.id; tr.dataset.user = JSON.stringify(user);
                 const roleText = { admin: '管理者', teacher: '教師', student: '學生' };
                 let classDisplay = 'N/A';
                 if (user.class && user.role !== 'admin') {
                     try {
                         const classIds = JSON.parse(user.class);
                         if (Array.isArray(classIds) && classIds.length > 0) { classDisplay = classIds.map(id => { const found = ALL_CLASSES.find(c => c.id == id); return found ? found.class_name : `(ID: ${id})`; }).join(', '); }
                         else { classDisplay = '(未分班)'; }
                     } catch(e) { classDisplay = '(格式錯誤)'; }
                 }
                 if (classDisplay === 'null' || !user.class || classDisplay === '[]') classDisplay = '(未分班)';
                 if (user.role === 'admin') classDisplay = 'N/A';

                 tr.innerHTML = `<td>${escapeHTML(user.username)}</td><td>${escapeHTML(user.name)}</td><td>${escapeHTML(classDisplay)}</td><td>${roleText[user.role] || user.role}</td><td>${user.grade || 'N/A'}</td><td><span class="status-badge ${user.is_active ? 'active' : 'inactive'}">${user.is_active ? '啟用' : '停用'}</span></td><td class="actions"><button class="edit-btn">編輯</button><button class="delete-btn">刪除</button></td>`;
                 const deleteBtn = tr.querySelector('.delete-btn');
                 if (user.username === 'typing') { deleteBtn.disabled = true; }
                 if (Number(user.id) === CURRENT_USER_ID) { deleteBtn?.remove(); }
                 tableBody.appendChild(tr);
             } catch (renderError) {
                 console.error(`[renderTable] Error rendering user row at index ${index}:`, user, renderError);
                 const errTr = document.createElement('tr'); errTr.innerHTML = `<td colspan="7" class="error">渲染此列時發生錯誤: ${escapeHTML(renderError.message)}</td>`; tableBody.appendChild(errTr);
             }
        });
    }

    // --- Form Handling ---
    // Initialize the add/edit form
    function initializeForm() {
        roleSelect.innerHTML = '';
        if (CURRENT_ROLE === 'admin') { if (CURRENT_USERNAME === 'typing') { roleSelect.add(new Option('管理者', 'admin')); } roleSelect.add(new Option('教師', 'teacher')); roleSelect.add(new Option('學生', 'student')); }
        else if (CURRENT_ROLE === 'teacher') { roleSelect.add(new Option('學生', 'student')); }
        toggleFieldsVisibility();
    }
    // Show/hide grade and class fields
    function toggleFieldsVisibility() {
        const selectedRole = roleSelect.value;
        gradeGroup.style.display = (selectedRole === 'student') ? 'block' : 'none';
        classGroup.style.display = (selectedRole === 'student' || selectedRole === 'teacher') ? 'block' : 'none';
        classSelect.multiple = (selectedRole === 'teacher');
        const defaultClassOption = classSelect.querySelector('option[value=""]');
        if (selectedRole === 'student' && !defaultClassOption) { classSelect.insertAdjacentHTML('afterbegin', '<option value="">(尚未分班)</option>'); classSelect.value = ''; }
    }
    // Reset the add/edit form
    function resetForm() {
        form.reset(); userIdInput.value = ''; formTitle.textContent = '新增帳號'; submitBtn.textContent = '儲存'; cancelBtn.style.display = 'none'; usernameInput.disabled = false; nameInput.disabled = false; document.getElementById('role-group').style.display = 'block'; showMessage(''); Array.from(classSelect.options).forEach(opt => opt.selected = false); toggleFieldsVisibility(); activeCheckbox.disabled = false; submitBtn.disabled = false;
    }
    // Populate the add/edit form
    function populateForm(user) {
        resetForm(); formTitle.textContent = '編輯帳號'; submitBtn.textContent = '更新'; cancelBtn.style.display = 'inline-block'; userIdInput.value = user.id; usernameInput.value = user.username; nameInput.value = user.name; gradeSelect.value = user.grade || '3'; activeCheckbox.checked = !!user.is_active;
        const classIds = (user.class && user.class !== 'null') ? JSON.parse(user.class) : [];
        Array.from(classSelect.options).forEach(opt => { opt.selected = classIds.includes(Number(opt.value)); });
        if (user.role === 'student' && classIds.length === 0) { classSelect.value = ''; }
        document.getElementById('role-group').style.display = 'none';
        if (user.role === 'student') { gradeGroup.style.display = 'block'; classGroup.style.display = 'block'; classSelect.multiple = false; }
        else if (user.role === 'teacher') { gradeGroup.style.display = 'none'; classGroup.style.display = 'block'; classSelect.multiple = true; }
        else { gradeGroup.style.display = 'none'; classGroup.style.display = 'none'; }
        usernameInput.disabled = true; activeCheckbox.disabled = (Number(user.id) === CURRENT_USER_ID); nameInput.disabled = (user.username === 'typing' && CURRENT_USERNAME !== 'typing'); submitBtn.disabled = false;
    }
    // Handle submission of the add/edit form
    async function handleFormSubmit(e) {
        e.preventDefault(); if (nameInput.value.trim() === '') { showMessage('「姓名」為必要欄位', true); return; }
        const selectedClassIds = Array.from(classSelect.selectedOptions).map(opt => opt.value ? Number(opt.value) : null).filter(id => id !== null);
        let currentEditingRole = roleSelect.value; const id = userIdInput.value; const isUpdate = !!id;
        if (isUpdate) { try { const userJson = document.querySelector(`tr[data-id="${id}"]`)?.dataset.user; if (userJson) currentEditingRole = JSON.parse(userJson).role; } catch(e){} }
        if (currentEditingRole === 'student' && selectedClassIds.length > 1) { showMessage('學生僅能指定一個班級', true); return; }
        showMessage(''); submitBtn.disabled = true;
        const data = { id: id, username: usernameInput.value, name: nameInput.value, class: selectedClassIds, role: roleSelect.value, grade: gradeSelect.value, is_active: activeCheckbox.checked };
        let url = 'api/users.php'; let method = isUpdate ? 'PUT' : 'POST';
        try {
            const response = await fetch(url, { method: method, headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(data) });
            const result = await response.json(); if (!response.ok) throw new Error(result.message || (isUpdate ? '更新失敗' : '新增失敗'));
            showMessage(isUpdate ? '更新成功' : '新增成功', false);
            setTimeout(() => { closeModal(addEditUserModal); loadUsers(); }, 500);
        } catch (err) { console.error("Error submitting user form:", err); showMessage(err.message, true); submitBtn.disabled = false; }
    }
    // Handle user deletion
    async function handleDelete(e) {
        const deleteButton = e.target; const tr = deleteButton.closest('tr'); if (!tr || !tr.dataset.user) return; const id = tr.dataset.id; const user = JSON.parse(tr.dataset.user);
        if (Number(id) === CURRENT_USER_ID) { showMessage('無法刪除您自己的帳號', true, tableMessage); return; }
        if (!confirm(`確定要刪除帳號 "${user.username}" 嗎？`)) return;
        deleteButton.disabled = true;
        try {
            const response = await fetch(`api/users.php?id=${id}`, { method: 'DELETE' });
            const result = await response.json(); if (!response.ok) throw new Error(result.message || '刪除失敗');
            showMessage('刪除成功', false, tableMessage);
            tr.remove();
        } catch (err) { console.error("Error deleting user:", err); showMessage('刪除失敗：' + err.message, true, tableMessage); deleteButton.disabled = false; }
    }
    // Handle clicks within the user table
    function handleTableClick(e) { if (e.target.classList.contains('edit-btn')) { const tr = e.target.closest('tr'); if (tr && tr.dataset.user) { const user = JSON.parse(tr.dataset.user); populateForm(user); openModal(addEditUserModal); } } if (e.target.classList.contains('delete-btn')) { handleDelete(e); } }
    // Handle batch upload form submission
    async function handleBatchUpload(e) {
         e.preventDefault(); const file = batchFileInput.files[0]; if (!file) { showMessage('請選擇要上傳的檔案', true, batchMessage); return; }
         const formData = new FormData(); formData.append('account_file', file);
         showMessage('上傳中，請稍候...', false, batchMessage); batchSubmitBtn.disabled = true; let resultData = {};
         try {
             const response = await fetch('api/batch-upload.php', { method: 'POST', body: formData }); resultData = await response.json();
             if (!response.ok && response.status !== 200) { if (resultData.message) throw new Error(resultData.message); else throw new Error(`上傳失敗 (HTTP ${response.status})`); }
             batchForm.reset(); closeModal(batchUploadModal); loadUsers();
             let msg = `成功新增 ${resultData.createdCount || 0} 筆帳號。`; let isErrorResult = false;
             if (resultData.errors && resultData.errors.length > 0) { msg += `\n發生 ${resultData.errors.length} 個錯誤：\n` + resultData.errors.join('\n'); isErrorResult = true; }
             showMessage(msg, isErrorResult, batchResultContent); openModal(batchResultModal);
         } catch (err) { console.error("Error during batch upload:", err); showMessage('錯誤: ' + err.message, true, batchMessage); }
         finally { batchSubmitBtn.disabled = false; }
    }

    // --- Event Listeners Setup ---
    form.addEventListener('submit', handleFormSubmit);
    tableBody.addEventListener('click', handleTableClick);
    roleSelect.addEventListener('change', toggleFieldsVisibility);
    showAddUserBtn.addEventListener('click', () => { resetForm(); openModal(addEditUserModal); });
    closeUserModalBtn.addEventListener('click', () => closeModal(addEditUserModal));
    cancelBtn.addEventListener('click', () => closeModal(addEditUserModal));
    addEditUserModal.addEventListener('click', (e) => { if (e.target === addEditUserModal) closeModal(addEditUserModal); });
    showBatchBtn.addEventListener('click', () => { showMessage('', false, batchMessage); batchForm.reset(); batchSubmitBtn.disabled = false; openModal(batchUploadModal); });
    closeBatchUploadBtn.addEventListener('click', () => closeModal(batchUploadModal));
    batchUploadModal.addEventListener('click', (e) => { if (e.target === batchUploadModal) closeModal(batchUploadModal); });
    batchForm.addEventListener('submit', handleBatchUpload);
    closeBatchResultBtn.addEventListener('click', () => { closeModal(batchResultModal); closeModal(batchUploadModal); });
    batchResultModal.addEventListener('click', (e) => { if (e.target === batchResultModal) { closeModal(batchResultModal); closeModal(batchUploadModal); } });
    exportUsersBtn.addEventListener('click', () => { window.location.href = 'api/users.php?action=export'; });

    // --- Initial Load ---
    async function init() {
        console.log("Initializing User Management...");
        try {
            initializeForm();
            await loadAllClasses();
            console.log("[init users] Finished loadAllClasses successfully.");
            await loadUsers();
            console.log("User Management Initialization complete.");
        } catch (initError) {
             console.error("Error during User Management Initialization:", initError);
             if (tableMessage) showMessage("初始化帳號管理失敗，請重新整理頁面或檢查主控台。\n錯誤：" + initError.message, true, tableMessage);
        }
    }
    init();

})();
