/* typing_project.sql */

/* 警告：此指令將刪除現有資料庫 */
DROP DATABASE IF EXISTS typing_gemini;

/* 建立資料庫 */
CREATE DATABASE IF NOT EXISTS typing_gemini CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE typing_gemini;

/* --- 1. 帳號與權限 --- */

/* 班級管理表 (新架構) */
CREATE TABLE classes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    class_name VARCHAR(100) NOT NULL UNIQUE COMMENT '班級名稱 (例如: C101)',
    grade_level INT NULL COMMENT '所屬年級'
    /* (新) 移除 teacher_ids 欄位 */
) COMMENT '班級管理表 (管理者維護)';

/* 使用者帳號表 (新架構) */
CREATE TABLE users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(100) NOT NULL UNIQUE COMMENT '帳號',
    name VARCHAR(100) NOT NULL COMMENT '儲存姓名',
    class JSON DEFAULT NULL COMMENT '班級ID陣列 (教師:[1,2], 學生:[1])',
    role ENUM('admin', 'teacher', 'student') NOT NULL COMMENT '角色',
    grade INT DEFAULT NULL COMMENT '學生年級 (3-6)',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    update_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT '最後更新時間',
    is_active BOOLEAN DEFAULT TRUE COMMENT '帳號是否啟用'
) COMMENT '使用者帳號表';

/* ... (login_tokens, operation_logs, announcements, test_bank ... 同前) ... */
CREATE TABLE login_tokens (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    token CHAR(5) NOT NULL,
    expires_at TIMESTAMP NOT NULL,
    is_used BOOLEAN DEFAULT FALSE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);
CREATE TABLE operation_logs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NULL,
    username VARCHAR(100),
    action VARCHAR(255) NOT NULL,
    details JSON NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
CREATE TABLE announcements (
    id INT AUTO_INCREMENT PRIMARY KEY,
    title VARCHAR(255) NOT NULL,
    content TEXT,
    created_by INT NOT NULL,
    start_time TIMESTAMP NOT NULL,
    end_time TIMESTAMP NOT NULL,
    FOREIGN KEY (created_by) REFERENCES users(id)
);

/* 題庫相關資料表 */
CREATE TABLE test_bank (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL COMMENT '題目名稱',
    type_id INT NULL COMMENT '題目類型 ID (FK to question_types)', -- Stores ID now
    content TEXT NOT NULL COMMENT '題目內容',
    length INT NOT NULL DEFAULT 0 COMMENT '題目總字數(含標點)',
    level INT NOT NULL COMMENT '程度/適用年級 (3-6)', -- Renamed from grade
    includes_punctuation BOOLEAN DEFAULT TRUE COMMENT '是否包含標點符號',
    created_by INT NOT NULL COMMENT '建立者 (管理者或教師)',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    update_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT '最後編修日期時間',
    FOREIGN KEY (created_by) REFERENCES users(id),
    INDEX idx_created_by (created_by) -- Added index
) COMMENT '中文輸入題庫';

CREATE TABLE question_types (
    id INT AUTO_INCREMENT PRIMARY KEY,
    type_name VARCHAR(50) NOT NULL UNIQUE COMMENT '類型名稱'
) COMMENT '題庫題目類型表';

INSERT INTO question_types (type_name) VALUES ('文章'),('古詩詞賦'),('現代詩'),('名言');
/****************/

CREATE TABLE tests (
    id INT AUTO_INCREMENT PRIMARY KEY,
    title VARCHAR(255) NOT NULL,
    teacher_id INT NOT NULL,
    start_time TIMESTAMP NOT NULL,
    end_time TIMESTAMP NOT NULL,
    mode ENUM('timed', 'measured') NOT NULL,
    settings JSON NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (teacher_id) REFERENCES users(id)
);
CREATE TABLE test_questions (
    test_id INT NOT NULL,
    question_id INT NOT NULL,
    PRIMARY KEY (test_id, question_id),
    FOREIGN KEY (test_id) REFERENCES tests(id) ON DELETE CASCADE,
    FOREIGN KEY (question_id) REFERENCES test_bank(id) ON DELETE CASCADE
);
CREATE TABLE test_participants (
    test_id INT NOT NULL,
    student_id INT NOT NULL,
    PRIMARY KEY (test_id, student_id),
    FOREIGN KEY (test_id) REFERENCES tests(id) ON DELETE CASCADE,
    FOREIGN KEY (student_id) REFERENCES users(id) ON DELETE CASCADE
);
CREATE TABLE test_submissions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    student_id INT NOT NULL,
    test_id INT NOT NULL,
    time_taken_seconds INT NOT NULL,
    chars_completed INT NOT NULL,
    errors INT NOT NULL,
    wpm DECIMAL(5, 1) NOT NULL,
    submitted_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (student_id) REFERENCES users(id),
    FOREIGN KEY (test_id) REFERENCES tests(id)
);
CREATE TABLE competitions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    title VARCHAR(255) NOT NULL,
    organizer_id INT NOT NULL,
    details TEXT,
    reg_start_time TIMESTAMP NOT NULL,
    reg_end_time TIMESTAMP NOT NULL,
    event_time TIMESTAMP NOT NULL,
    eligibility_rules JSON,
    mode_settings JSON NOT NULL,
    FOREIGN KEY (organizer_id) REFERENCES users(id)
);
CREATE TABLE competition_registrations (
    id INT AUTO_INCREMENT PRIMARY KEY,
    student_id INT NOT NULL,
    competition_id INT NOT NULL,
    registered_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (student_id) REFERENCES users(id),
    FOREIGN KEY (competition_id) REFERENCES competitions(id)
);
CREATE TABLE competition_scores (
    id INT AUTO_INCREMENT PRIMARY KEY,
    student_id INT NOT NULL,
    competition_id INT NOT NULL,
    time_taken_seconds INT NOT NULL,
    chars_completed INT NOT NULL,
    errors INT NOT NULL,
    wpm DECIMAL(5, 1) NOT NULL,
    submitted_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (student_id) REFERENCES users(id),
    FOREIGN KEY (competition_id) REFERENCES competitions(id)
);

/* --- 5. 插入預設資料 (新架構) --- */

/* (新) 建立預設班級 (id: 1, 2) */
INSERT INTO classes (id, class_name, grade_level) VALUES
(1, 'C101', 3),
(2, 'C102', 3);

/* (新) 系統預設帳號 (教師 class: [1,2], 學生 class: [1]) */
INSERT INTO users (username, name, class, role, grade, is_active) VALUES
('typing', '系統管理者', NULL, 'admin', NULL, TRUE),
('teacher', '預設教師', '[1, 2]', 'teacher', NULL, TRUE),
('student', '預設學生', '[1]', 'student', 3, TRUE);
