<?php
session_start();
include 'db_connect.php'; // 引入資料庫連線
include 'log_helper.php'; // 引入 Log

// --- Error Reporting (for debugging) ---
ini_set('display_errors', 0);
error_reporting(E_ALL);
// ini_set('log_errors', 1);
// ini_set('error_log', '/path/to/your/php-error.log');

header('Content-Type: application/json');

// --- 1. Access Control ---
if (!isset($_SESSION['user'])) {
    http_response_code(401);
    echo json_encode(['message' => '未授權，請先登入']);
    exit;
}
$currentUser = $_SESSION['user'];

// --- 2. Get Parameters ---
$level = filter_input(INPUT_GET, 'level', FILTER_VALIDATE_INT, [
    'options' => ['min_range' => 3, 'max_range' => 6]
]);
$targetLength = filter_input(INPUT_GET, 'targetLength', FILTER_VALIDATE_INT, [
    'options' => ['min_range' => 1] // Minimum 1 character
]);

// Default level if not provided
if ($level === false || $level === null) {
    $level = $currentUser['grade'] ?? 3;
    if ($level < 3 || $level > 6) { $level = 3; }
}
// Default length if not provided
if ($targetLength === false || $targetLength === null) {
    $targetLength = 500; // Default fallback length
}

$pdo = null;
$logDetails = [ 'action' => 'FETCH_QUESTIONS_ONCE', 'requested_level' => $level, 'requested_length' => $targetLength ];
$questions_content = [];
$total_length_fetched = 0;

try {
    $pdo = get_db_connection();
    log_operation($pdo, $currentUser['id'], $currentUser['username'], 'GET_QUESTIONS_START', json_encode($logDetails, JSON_UNESCAPED_UNICODE));

    // --- 3. Fetch Questions (Accumulate until targetLength is met) ---
    
    // Helper function to fetch and append questions
    function fetchAndAccumulate($pdo, $sql, $params, $targetLength, &$questions_content, &$total_length_fetched) {
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            if (!empty($row['content'])) {
                $questions_content[] = $row['content']; // Add content block
                
                // (修正) Use stored length ONLY if it's valid (> 0), otherwise recalculate
                $contentLength = (isset($row['length']) && $row['length'] > 0) ? (int)$row['length'] : mb_strlen($row['content'], 'UTF-8');
                
                $total_length_fetched += $contentLength;
                if ($total_length_fetched >= $targetLength) {
                    return true; // Target met
                }
            }
        }
        return false; // Target not met
    }

    // Attempt 1: Specific level
    $sql_level = "SELECT content, length FROM test_bank WHERE level = ? ORDER BY RAND()";
    $targetMet = fetchAndAccumulate($pdo, $sql_level, [$level], $targetLength, $questions_content, $total_length_fetched);

    // Attempt 2: Fallback
    if (!$targetMet) {
        error_log("[get_questions] Target length {$targetLength} not met with level {$level} (found {$total_length_fetched}). Fetching fallback.");
        log_operation($pdo, $currentUser['id'], $currentUser['username'], 'GET_QUESTIONS_FALLBACK_START', json_encode(['target' => $targetLength, 'found' => $total_length_fetched]));
        
        $sql_fallback = "SELECT content, length FROM test_bank WHERE level != ? ORDER BY RAND()";
        $targetMet = fetchAndAccumulate($pdo, $sql_fallback, [$level], $targetLength, $questions_content, $total_length_fetched);
    }
    
    if ($total_length_fetched === 0) {
        // This means database is empty or has no content
        throw new Exception('題庫中找不到任何題目。');
    }

    // --- 4. Process and Return JSON ---
    $full_text = implode("\n", $questions_content);
    $questionLines = [];
    if (!empty($full_text)) {
        // Split by both \r\n and \n
        $lines = preg_split('/\r\n|\r|\n/', $full_text);
        foreach ($lines as $line) {
            $trimmedLine = trim($line);
            if (mb_strlen($trimmedLine, 'UTF-8') > 0) { // Filter out empty lines
                $questionLines[] = $trimmedLine;
            }
        }
    }
    
    if (empty($questionLines)) {
         throw new Exception('題庫題目內容格式錯誤 (無法解析行)。');
    }

    log_operation($pdo, $currentUser['id'], $currentUser['username'], 'GET_QUESTIONS_SUCCESS', json_encode(['lines_fetched' => count($questionLines), 'total_chars' => $total_length_fetched]));
    echo json_encode($questionLines, JSON_UNESCAPED_UNICODE); // Return array of lines

} catch (\PDOException $e) {
    error_log("[get_questions] PDOException: " . $e->getMessage());
    if ($pdo) { log_operation($pdo, $currentUser['id'], $currentUser['username'], 'GET_QUESTIONS_PDO_ERROR', json_encode(['error' => $e->getMessage()])); }
    http_response_code(500);
    echo json_encode(['message' => '無法取得題目資料庫資料 (PDO Error)']);
} catch (\Exception $e) {
    error_log("[get_questions] General Exception: " . $e->getMessage());
    if ($pdo) { log_operation($pdo, $currentUser['id'], $currentUser['username'], 'GET_QUESTIONS_ERROR', json_encode(['error' => $e->getMessage()])); }
    http_response_code(500);
    echo json_encode(['message' => $e->getMessage()]); // Send the specific error (e.g., "題庫中找不到...")
}

