<?php
session_start();
include 'db_connect.php';
include 'log_helper.php';

header('Content-Type: application/json');

if (!isset($_SESSION['user'])) {
    http_response_code(401);
    echo json_encode(['message' => '未授權，請先登入']);
    exit;
}
$currentUser = $_SESSION['user'];
$method = $_SERVER['REQUEST_METHOD'];
$pdo = get_db_connection();
$action = $_REQUEST['action'] ?? null; // Check for action

// --- 輔助函數：取得教師可管理的班級 (從 users.class) ---
$teacherClassIDs = [];
if ($currentUser['role'] == 'teacher') {
    // 從登入者 (教師) 的 session 中讀取班級 ID 陣列 [1, 2]
    $teacherClassIDs = json_decode($currentUser['class'] ?? '[]', true);
}

try {
    // --- (新) 滙出 (GET with action=export) ---
    if ($method == 'GET' && $action == 'export') {
        $filename = "users-export-" . date('ymd') . ".csv"; //

        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');

        $output = fopen('php://output', 'w');
        fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF)); // UTF-8 BOM

        // Header row
        fputcsv($output, ['id', 'username', 'name', 'role', 'class_ids', 'grade', 'is_active']);

        // Data rows (apply visibility rules)
        $sql = "SELECT id, username, name, class, role, grade, is_active FROM users";
        $params = [];

        if ($currentUser['role'] == 'teacher') {
            // 教師僅能匯出 (a) 自己班級的學生 + (b) 尚未分班的學生
            $sql .= " WHERE role = 'student' AND (";
            $sql .= " class IS NULL OR class = '[]' OR class = ''";
            if (!empty($teacherClassIDs)) {
                foreach ($teacherClassIDs as $class_id) {
                    $sql .= " OR JSON_CONTAINS(class, CAST(" . (int)$class_id . " AS CHAR))";
                }
            }
            $sql .= ")";
            $stmt = $pdo->prepare($sql);
            $stmt->execute(); // No params needed here
        } else if ($currentUser['role'] == 'admin') {
            // 管理者匯出所有使用者
             $sql .= " ORDER BY role, username"; // Optional ordering
            $stmt = $pdo->query($sql);
        } else {
             // 學生不能匯出
             fclose($output);
             exit; // Exit silently
        }

        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            // Format class IDs for CSV
            $class_ids_str = '';
            if ($row['class']) {
                 try {
                     $ids = json_decode($row['class'], true);
                     if (is_array($ids)) {
                         $class_ids_str = implode(',', $ids); // e.g., "1,2"
                     }
                 } catch (Exception $e) {}
            }
            $row['class'] = $class_ids_str; // Replace JSON with comma-separated string
            fputcsv($output, [
                $row['id'],
                $row['username'],
                $row['name'],
                $row['role'],
                $row['class'], // Use the formatted string
                $row['grade'],
                $row['is_active'] ? '1' : '0' // Use 1/0 for boolean
            ]);
        }
        fclose($output);
        // Log export action (optional)
        log_operation($pdo, $currentUser['id'], $currentUser['username'], 'EXPORT_USERS');
        exit; // End export action
    }


    // --- Original CRUD Operations ---
    switch ($method) {
        // --- 讀取 (GET) ---
        case 'GET':
            $sql = "SELECT id, username, name, class, role, grade, is_active FROM users";
            $params = [];
            if ($currentUser['role'] == 'admin') {
                $sql .= " ORDER BY role, username";
                $stmt = $pdo->query($sql);
            }
            else if ($currentUser['role'] == 'teacher') {
                $sql .= " WHERE role = 'student' AND (";
                $sql .= " class IS NULL OR class = '[]' OR class = ''";
                if (!empty($teacherClassIDs)) {
                    foreach ($teacherClassIDs as $class_id) {
                        $sql .= " OR JSON_CONTAINS(class, CAST(" . (int)$class_id . " AS CHAR))";
                    }
                }
                $sql .= ") ORDER BY username";
                $stmt = $pdo->query($sql);
            }
            else { http_response_code(403); echo json_encode(['message' => '權限不足']); exit; }
            echo json_encode($stmt->fetchAll());
            break;

        // --- 新增 (POST) ---
        case 'POST':
            $data = json_decode(file_get_contents('php://input'), true);
            $newRole = $data['role'];

            // Permissions...
            if ($newRole == 'admin' && $currentUser['username'] != 'typing') { http_response_code(403); echo json_encode(['message' => '權限不足：僅 "typing" 才能建立管理者']); exit; }
            if ($newRole == 'teacher' && $currentUser['role'] != 'admin') { http_response_code(403); echo json_encode(['message' => '權限不足：僅管理者才能建立教師']); exit; }
            if (empty($data['name'])) { http_response_code(400); echo json_encode(['message' => '「姓名」為必要欄位']); exit; }
            if (empty($data['username'])) { http_response_code(400); echo json_encode(['message' => '「帳號」為必要欄位']); exit; }


            // Class data processing
            $class_json = null;
            $class_data = $data['class'] ?? [];
            $class_array = is_array($class_data) ? $class_data : (empty($class_data) ? [] : [$class_data]);
            if ($newRole == 'student' && count($class_array) > 1) { http_response_code(400); echo json_encode(['message' => '學生僅能指定一個班級']); exit; }
            // Ensure class IDs are integers before encoding
            if (!empty($class_array)) { $class_json = json_encode(array_map('intval', $class_array)); }

            // Teacher permission check for adding students to classes
            if ($currentUser['role'] == 'teacher') {
                if ($newRole != 'student') { http_response_code(403); echo json_encode(['message' => '教師僅能建立學生']); exit; }
                $studentClassId = $class_array[0] ?? null;
                // Allow teacher to create unassigned students
                if ($studentClassId && !in_array($studentClassId, $teacherClassIDs)) { http_response_code(403); echo json_encode(['message' => '教師僅能將學生加入自己所屬的班級']); exit; }
            }

            // Insert...
            $sql = "INSERT INTO users (username, name, class, role, grade, is_active) VALUES (?, ?, ?, ?, ?, ?)";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                $data['username'], $data['name'], $class_json, $data['role'],
                ($data['role'] == 'student' ? ($data['grade'] ?? 3) : null), // Default grade 3 for student
                ($data['is_active'] ?? true) ? 1 : 0
            ]);

            $newId = $pdo->lastInsertId();
            log_operation($pdo, $currentUser['id'], $currentUser['username'], 'USER_CREATE', json_encode(['id' => $newId, 'username' => $data['username']]));
            $stmt = $pdo->prepare("SELECT id, username, name, class, role, grade, is_active FROM users WHERE id = ?");
            $stmt->execute([$newId]);
            $newUser = $stmt->fetch();
            http_response_code(201); echo json_encode($newUser);
            break;

        // --- 更新 (PUT) ---
        case 'PUT':
            $data = json_decode(file_get_contents('php://input'), true);
            $targetId = $data['id'];
            if (empty($data['name'])) { http_response_code(400); echo json_encode(['message' => '「姓名」為必要欄位']); exit; }
            if (empty($data['username'])) { http_response_code(400); echo json_encode(['message' => '「帳號」為必要欄位']); exit; }


            $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?"); $stmt->execute([$targetId]); $targetUser = $stmt->fetch();
            if (!$targetUser) { http_response_code(404); echo json_encode(['message' => '找不到目標使用者']); exit; }

            // Class data processing
            $class_json = $targetUser['class']; // Default to old value
            $class_data = $data['class'] ?? [];
            // Ensure it's an array of integers
            $class_array = array_map('intval', is_array($class_data) ? $class_data : (empty($class_data) ? [] : [$class_data]));

            if ($targetUser['role'] == 'student' && count($class_array) > 1) { http_response_code(400); echo json_encode(['message' => '學生僅能指定一個班級']); exit; }
            $class_json = !empty($class_array) ? json_encode($class_array) : NULL;

            // --- Teacher permission check ---
            if ($currentUser['role'] == 'teacher') {
                if ($targetUser['role'] != 'student') { http_response_code(403); echo json_encode(['message' => '教師僅能修改學生']); exit; }
                $targetUserClassIDs = json_decode($targetUser['class'] ?? '[]', true);
                $isUnassigned = empty($targetUserClassIDs);
                $isInTeacherClass = !$isUnassigned && in_array($targetUserClassIDs[0], $teacherClassIDs);
                $newStudentClass = $class_array[0] ?? null;

                if ($isUnassigned) { // Req 3
                    if ($newStudentClass && !in_array($newStudentClass, $teacherClassIDs)) { http_response_code(403); echo json_encode(['message' => '您僅能將無班級學生指派到您所屬的班級']); exit; }
                } else if ($isInTeacherClass) { // Req 4
                    if ($newStudentClass && !in_array($newStudentClass, $teacherClassIDs)) { http_response_code(403); echo json_encode(['message' => '您僅能將學生修改到您所屬的班級']); exit; }
                } else { http_response_code(403); echo json_encode(['message' => '您無權修改此學生']); exit; }
            }
            // --- Admin permission check ---
            if ($currentUser['role'] == 'admin' && $targetUser['role'] == 'admin' && $currentUser['username'] != 'typing') { http_response_code(403); echo json_encode(['message' => '權限不足：僅 "typing" 才能修改管理者']); exit; }
            if ($currentUser['role'] != 'admin' && $targetUser['role'] == 'teacher') { http_response_code(403); echo json_encode(['message' => '權限不足：僅管理者才能修改教師']); exit; }

            // Update execution...
            if ($targetUser['username'] == 'typing' && $currentUser['username'] == 'typing') {
                 // Typing can only edit own name and active status
                 $sql = "UPDATE users SET name = ?, is_active = ? WHERE id = ?";
                 $stmt = $pdo->prepare($sql);
                 $stmt->execute([ $data['name'], ($data['is_active'] ?? true) ? 1 : 0, $targetId ]);
            } else {
                // Username cannot be changed after creation
                $sql = "UPDATE users SET name = ?, class = ?, grade = ?, is_active = ? WHERE id = ?";
                $stmt = $pdo->prepare($sql);
                $stmt->execute([
                    $data['name'], $class_json,
                    ($targetUser['role'] == 'student' ? ($data['grade'] ?? $targetUser['grade']) : null), // Keep old grade if not provided
                    ($data['is_active'] ?? true) ? 1 : 0, $targetId
                ]);
            }

            log_operation($pdo, $currentUser['id'], $currentUser['username'], 'USER_UPDATE', json_encode(['id' => $targetId, 'username' => $targetUser['username']])); // Log with original username
            $stmt = $pdo->prepare("SELECT id, username, name, class, role, grade, is_active FROM users WHERE id = ?"); $stmt->execute([$targetId]); $updatedUser = $stmt->fetch();
            echo json_encode($updatedUser);
            break;

        // --- 刪除 (DELETE) ---
        case 'DELETE':
            $targetId = $_GET['id'] ?? null;
            if (!$targetId) { http_response_code(400); echo json_encode(['message' => '缺少 ID']); exit; }
            $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?"); $stmt->execute([$targetId]); $targetUser = $stmt->fetch();
            if (!$targetUser) { http_response_code(404); echo json_encode(['message' => '找不到目標使用者']); exit; }
            if ($targetUser['username'] == 'typing') { http_response_code(403); echo json_encode(['message' => '系統帳號 "typing" 禁止刪除']); exit; }
            // Prevent deleting self
            if ($targetUser['id'] == $currentUser['id']) { http_response_code(403); echo json_encode(['message' => '無法刪除您自己的帳號']); exit; }


            // Teacher permission check...
            if ($currentUser['role'] == 'teacher') {
                 if ($targetUser['role'] != 'student') { http_response_code(403); echo json_encode(['message' => '教師僅能刪除學生']); exit; }
                 $targetUserClassIDs = json_decode($targetUser['class'] ?? '[]', true);
                 $isInTeacherClass = !empty($targetUserClassIDs) && in_array($targetUserClassIDs[0], $teacherClassIDs);
                 if (!$isInTeacherClass) { http_response_code(403); echo json_encode(['message' => '您僅能刪除您所屬班級的學生']); exit; }
            }
            // Admin permission check...
            if ($currentUser['role'] == 'admin' && $targetUser['role'] == 'admin' && $currentUser['username'] != 'typing') { http_response_code(403); echo json_encode(['message' => '權限不足：僅 "typing" 才能刪除管理者']); exit; }
            if ($currentUser['role'] != 'admin' && $targetUser['role'] == 'teacher') { http_response_code(403); echo json_encode(['message' => '權限不足：僅管理者才能刪除教師']); exit; }

            // Delete execution...
            $sql = "DELETE FROM users WHERE id = ?"; $stmt = $pdo->prepare($sql); $stmt->execute([$targetId]);
            log_operation($pdo, $currentUser['id'], $currentUser['username'], 'USER_DELETE', json_encode(['id' => $targetId, 'username' => $targetUser['username']]));
            echo json_encode(['success' => true]);
            break;

        default:
             if ($action == null) { http_response_code(405); echo json_encode(['message' => '方法不允許']); }
             else { http_response_code(400); echo json_encode(['message' => '無效的操作']); }
            break;
    }
} catch (\PDOException $e) {
    http_response_code(500);
    if ($e->getCode() == 23000) { echo json_encode(['message' => '錯誤：該帳號 (username) 已經存在']); }
    else { echo json_encode(['message' => '資料庫錯誤: ' . $e->getMessage()]); }
} catch (\Exception $e) {
     http_response_code(400);
     echo json_encode(['message' => '處理錯誤: ' . $e->getMessage()]);
}

