document.addEventListener("DOMContentLoaded", () => {
    // --- DOM Elements ---
    const mainContainer = document.getElementById("main-container");
    const nav = document.getElementById("main-nav");
    const usernameDisplay = document.getElementById("username-display");
    const logoutBtn = document.getElementById("logout-btn");

    // --- Element Existence Check ---
    if (!mainContainer || !nav || !usernameDisplay || !logoutBtn) {
        console.error("Dashboard Error: Could not find essential elements. Check dashboard.html.");
        if (mainContainer) { mainContainer.innerHTML = '<h2 class="error">頁面載入錯誤...</h2>'; }
        else { alert("頁面載入錯誤..."); }
        return;
    }
    console.log("Dashboard JS: Essential elements found.");

    let dashboardHomeContent = ''; // Store initial dashboard content
    try { dashboardHomeContent = mainContainer.innerHTML; }
    catch(e) { console.error("Dashboard Error: Failed to store initial dashboard content.", e); }

    // --- Core Functions ---

    // 1. Check Login Status
    async function checkLoginStatus() {
        console.log("Dashboard JS: Checking login status...");
        try {
            const response = await fetch('api/check-session.php');
            if (!response.ok) { console.log("Dashboard JS: Not logged in. Redirecting..."); window.location.href = 'index.html'; return; }
            const data = await response.json();
            console.log("Dashboard JS: Login status OK. User data:", data);
            window.currentUserId = data.id; window.currentUserRole = data.role; window.currentUsername = data.username; window.currentUserClass = data.class;
            usernameDisplay.textContent = data.username;
            // Show/Hide Nav Links based on Role
            const adminLinks = document.querySelectorAll('.admin-link');
            const adminTeacherLinks = document.querySelectorAll('.admin-teacher-link');
            adminLinks.forEach(link => link.style.display = (data.role === 'admin') ? 'inline-block' : 'none');
            adminTeacherLinks.forEach(link => link.style.display = (data.role === 'admin' || data.role === 'teacher') ? 'inline-block' : 'none');
            console.log("Dashboard JS: Nav links updated based on role.");
        } catch (err) { console.error("Dashboard JS: Session check failed with error, redirecting...", err); window.location.href = 'index.html'; }
    }

    // (修正) 2. Load Dynamic Content with CSS Handling
    async function loadContent(page) {
        console.log(`Dashboard JS: Loading content for page: ${page}`);
        // --- Remove Previous Dynamic Content ---
        document.getElementById("dynamic-script")?.remove(); // Remove old JS
        document.getElementById("dynamic-styles")?.remove(); // <<< Remove old CSS

        // --- Highlight Active Nav Link ---
        nav.querySelectorAll('a').forEach(a => a.classList.remove('active'));
        nav.querySelector(`[data-page="${page}"]`)?.classList.add('active');

        try {
            let contentUrl; // URL of the HTML content file
            let scriptUrl;  // URL of the corresponding JS file
            let cssUrl;     // <<< URL of the corresponding CSS file

            // --- Determine URLs based on page ---
            switch (page) {
                case 'typing':
                    contentUrl = 'dashboard/typing-content.html';
                    scriptUrl = 'js/typing.js';
                    cssUrl = 'css/typing.css'; // <<< Add CSS path
                    break;
                case 'admin-users':
                    contentUrl = 'dashboard/users-content.html';
                    scriptUrl = 'js/users.js';
                    cssUrl = 'css/management.css'; // <<< Add CSS path (uses management styles)
                    break;
                case 'admin-classes':
                    contentUrl = 'dashboard/classes-content.html';
                    scriptUrl = 'js/classes.js';
                    cssUrl = 'css/management.css'; // <<< Add CSS path (uses management styles)
                    break;
                case 'admin-testbank':
                    contentUrl = 'dashboard/testbank-content.html';
                    scriptUrl = 'js/testbank.js';
                    // <<< Load both management and testbank specific CSS
                    cssUrl = ['css/management.css', 'css/testbank.css'];
                    break;
                case 'dashboard':
                    if (dashboardHomeContent) { mainContainer.innerHTML = dashboardHomeContent; const userDisplay = mainContainer.querySelector('#username-display'); if (userDisplay) userDisplay.textContent = window.currentUsername || '...'; console.log("Dashboard JS: Restored dashboard home content."); }
                    else { console.error("Dashboard JS: Cannot restore dashboard home, initial content missing."); mainContainer.innerHTML = '<h2>無法載入佈告欄</h2>'; }
                    return; // Done for dashboard home (no dynamic CSS/JS needed)
                default:
                    mainContainer.innerHTML = `<h2>${escapeHTML(page)} 頁面 (建置中)</h2>`;
                    return; // Done for placeholder pages
            }

            // --- Load Dynamic CSS ---
            if (cssUrl) {
                 // Handle single or multiple CSS files
                const cssFiles = Array.isArray(cssUrl) ? cssUrl : [cssUrl];
                cssFiles.forEach(file => {
                     console.log(`Dashboard JS: Loading styles from ${file}`);
                     const link = document.createElement('link');
                     link.rel = 'stylesheet';
                     link.href = file;
                     link.id = 'dynamic-styles'; // <<< Assign ID for easy removal
                     // Add error handling for CSS loading
                     link.onerror = () => {
                         console.error(`Dashboard JS: Failed to load styles: ${file}`);
                         // Optionally display an error message
                     };
                     link.onload = () => {
                          console.log(`Dashboard JS: Styles ${file} loaded successfully.`);
                     };
                     document.head.appendChild(link); // Append to <head>
                });
            }


            // --- Fetch and Inject HTML Content ---
            console.log(`Dashboard JS: Fetching content from ${contentUrl}`);
            const response = await fetch(contentUrl);
            console.log(`Dashboard JS: Fetch content status: ${response.status}`);
            if (!response.ok) throw new Error(`內容載入失敗: ${contentUrl} (${response.status} ${response.statusText})`);
            mainContainer.innerHTML = await response.text();
            console.log(`Dashboard JS: Content for ${page} injected.`);

            // --- Load Dynamic JS ---
            if (scriptUrl) {
                console.log(`Dashboard JS: Loading script from ${scriptUrl}`);
                const script = document.createElement('script');
                script.src = scriptUrl;
                script.id = 'dynamic-script'; // Mark for removal
                script.onerror = () => { console.error(`Dashboard JS: Failed to load script: ${scriptUrl}`); mainContainer.innerHTML += `<p class="error">頁面腳本載入失敗 (${scriptUrl})</p>`; };
                script.onload = () => { console.log(`Dashboard JS: Script ${scriptUrl} loaded successfully.`); };
                document.body.appendChild(script); // Append to <body> (usually safer for execution timing)
            }

        } catch (err) {
            console.error("Dashboard JS: Error loading content:", err);
            mainContainer.innerHTML = `<h2>內容載入錯誤</h2><p>${escapeHTML(err.message)}</p>`;
        }
    }

    // 3. Handle Nav Clicks (Event Delegation on nav element)
    if (nav) {
        nav.addEventListener('click', (e) => {
            if (e.target.id === 'logout-btn') { e.preventDefault(); logout(); return; }
            const link = e.target.closest('a');
            if (link && link.dataset.page) { e.preventDefault(); const page = link.dataset.page; console.log(`Dashboard JS: Nav link clicked for page: ${page}`); loadContent(page); }
        });
    } else { console.error("Dashboard Error: Could not find #main-nav element to attach listener."); }


    // 4. Logout Function
    async function logout() { console.log("Dashboard JS: Logging out..."); try { await fetch('api/logout.php'); } catch (err) { console.error('Dashboard JS: Logout request failed', err); } finally { window.location.href = 'index.html'; } }

     // 5. HTML Escape Utility
     function escapeHTML(str) { if (str === null || str === undefined) return ''; return str.toString().replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/"/g, '&quot;').replace(/'/g, '&#39;'); }

    // --- Initial Load ---
    checkLoginStatus(); // Check login status and set up nav links
    nav.querySelector(`[data-page="dashboard"]`)?.classList.add('active'); // Activate home link

});
