// 中文練習頁面的 JS 邏輯 (採用按鈕啟用/禁用模式 + 修正 IME)
(function() {
    // --- DOM Elements ---
    const modeMeasuredRadio = document.getElementById('mode-measured');
    const modeTimedRadio = document.getElementById('mode-timed');
    const levelSelect = document.getElementById('level-select');
    const measuredSettings = document.getElementById('measured-settings');
    const timedSettings = document.getElementById('timed-settings');
    const charCountSelect = document.getElementById('char-count-select');
    const timeDurationSelect = document.getElementById('time-duration-select');
    const fontNoZhuyinRadio = document.getElementById('font-no-zhuyin');
    const fontZhuyinRadio = document.getElementById('font-zhuyin');
    const stopTestBtn = document.getElementById('stop-test-btn');
    const timerDisplay = document.getElementById('timer-display');
    const charsTypedDisplay = document.getElementById('chars-typed-display');
    const charsTargetDisplay = document.getElementById('chars-target-display');
    const errorsDisplay = document.getElementById('errors-display');
    const speedDisplay = document.getElementById('speed-display');
    const statusMessage = document.getElementById('status-message');
    const textDisplayWindow = document.getElementById('text-display-window');
    const typingInput = document.getElementById('typing-input');

    // --- State Variables ---
    let practiceMode = null; // Start with no mode selected
    let targetLevel = 3;
    let targetCount = 50;
    let targetTime = 10;
    let useZhuyinFont = false;
    let questionLines = [];
    let currentLineIndex = 0;
    let totalCharsCompleted = 0;
    let displayTypedChars = 0;
    let totalErrors = 0;
    let timerInterval = null;
    let startTime = null;
    let timeElapsed = 0;
    let practiceActive = false; // Is the timer running and input being processed?
    let practicePrepared = false; // Has a mode been selected, ready to start on focus?
    let isLoadingQuestions = false; // Flag to prevent *concurrent* fetches
    let currentLineHeight = 0;
    let practiceFinished = false; // (修正) 確保已定義
    let isComposing = false; // (新) Flag for IME composition

    const IGNORED_KEYS = ["Shift", "Control", "Alt", "Meta", "CapsLock", "Tab", "ArrowUp", "ArrowDown", "ArrowLeft", "ArrowRight", "Home", "End", "PageUp", "PageDown", "Insert", "Escape", "ContextMenu", "ScrollLock", "Pause", "NumLock", "F1", "F2", "F3", "F4", "F5", "F6", "F7", "F8", "F9", "F10", "F11", "F12", "PrintScreen"];

    // --- Element Existence Check ---
    if (!modeMeasuredRadio || !modeTimedRadio || !levelSelect || !measuredSettings || !timedSettings || !charCountSelect || !timeDurationSelect || !fontNoZhuyinRadio || !fontZhuyinRadio || !stopTestBtn || !timerDisplay || !charsTypedDisplay || !charsTargetDisplay || !errorsDisplay || !speedDisplay || !statusMessage || !textDisplayWindow || !typingInput) {
        console.error("Typing Practice: Missing required DOM elements.");
        const container = document.getElementById('main-container');
         if(container) { container.innerHTML = '<h2 class="error">練習頁面載入錯誤，缺少必要元素。</h2>'; }
        else { alert("練習頁面載入錯誤..."); }
        return;
    }

    // --- Utility Functions ---
    function escapeHTML(str) { if (str === null || str === undefined) return ''; return str.toString().replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/"/g, '&quot;').replace(/'/g, '&#39;'); }
    function showStatusMessage(text, isError = false) { if (!statusMessage) return; statusMessage.textContent = text; statusMessage.classList.toggle('error', isError); }

    // --- Initialization ---
    function initializePractice() {
        console.log("Initializing Practice...");
        setupEventListeners();
        if(window.currentUserGrade && window.currentUserGrade >= 3 && window.currentUserGrade <= 6) { levelSelect.value = window.currentUserGrade; } else { levelSelect.value = 3; }
        readSettings(); // Read initial settings but don't reset yet
        updateSettingsUI(); // Show correct UI for default mode (if any checked initially)
        setIdleState(false); // Call the correct function
        handleFontChange(); // Apply initial font
    }

    // --- State Management ---
    // Sets the UI and state for when practice is NOT active (Idle or Finished)
    function setIdleState(finished = false) {
        console.log("Setting Idle State. Finished:", finished);
        practiceActive = false;
        practicePrepared = false; // Mode needs to be re-selected
        isLoadingQuestions = false; // Ensure flag is reset
        isComposing = false; // (新) Reset composing flag
        clearInterval(timerInterval); timerInterval = null;
        startTime = null; // Clear start time

        // Reset scores only if starting fresh (not just finishing)
        if (!finished) {
             timeElapsed = 0; currentLineIndex = 0; totalCharsCompleted = 0; totalErrors = 0; displayTypedChars = 0; questionLines = []; fullText = '';
             timerDisplay.textContent = '00:00'; charsTypedDisplay.textContent = '0'; errorsDisplay.textContent = '0'; speedDisplay.textContent = '0.0';
             textDisplayWindow.innerHTML = '<div class="line upcoming" style="top: 0px;">請選擇模式並開始...</div>';
             textDisplayWindow.style.transform = 'translateY(0px)'; currentLineHeight = 0;
             typingInput.value = '';
             // Set the initial prompt only when resetting, not when finishing
             showStatusMessage("選擇模式、程度後，將游標移至輸入框即可開始...");
        } else {
            // Keep final scores displayed
            // Update status message
            const finalWpm = speedDisplay.textContent; // Get final calculated WPM
             showStatusMessage(`練習完成！ 速度: ${finalWpm} 字/分 (${totalCharsCompleted}字/${totalErrors}錯)。請重新選擇模式開始新練習。`, false);
        }

        // --- Control Button States (Idle State) ---
        stopTestBtn.disabled = true;
        modeMeasuredRadio.disabled = false; modeTimedRadio.disabled = false;
        levelSelect.disabled = false; charCountSelect.disabled = false; timeDurationSelect.disabled = false;
        fontNoZhuyinRadio.disabled = false; fontZhuyinRadio.disabled = false;
        typingInput.disabled = false; // Input enabled to receive focus

        // (新) Uncheck mode radios to force re-selection
        modeMeasuredRadio.checked = false;
        modeTimedRadio.checked = false;
        practiceMode = null; // Clear selected mode
        updateSettingsUI(); // Hide mode-specific settings
    }

    // --- Event Listeners ---
    function setupEventListeners() {
        modeMeasuredRadio.addEventListener('click', preparePractice);
        modeTimedRadio.addEventListener('click', preparePractice);
        levelSelect.addEventListener('change', readSettings);
        charCountSelect.addEventListener('change', readSettings);
        timeDurationSelect.addEventListener('change', readSettings);
        
        // (修正) 綁定 click 事件到 label，以模擬 toggle 行為
        fontNoZhuyinRadio.labels[0].addEventListener('click', handleFontChange); // 假設 label 是 radio 的第一個 label
        fontZhuyinRadio.labels[0].addEventListener('click', handleFontChange); // 假設 label 是 radio 的第一個 label
        
        typingInput.addEventListener('focusin', startPracticeIfPrepared);
        typingInput.addEventListener('keydown', handleKeyDown);
        typingInput.addEventListener('input', handleInput); // Use 'input' for IME handling
        stopTestBtn.addEventListener('click', finishPractice);

        // (新) IME Composition Listeners
        typingInput.addEventListener('compositionstart', () => {
            isComposing = true;
            console.log("IME Composition Start");
        });
        typingInput.addEventListener('compositionend', (event) => {
            isComposing = false;
            console.log("IME Composition End");
            // (新) Manually trigger input handler on composition end
            // to ensure the final character is counted immediately
            handleInput(event);
        });

        // Disable copy/paste etc.
        typingInput.addEventListener('paste', e => e.preventDefault());
        typingInput.addEventListener('copy', e => e.preventDefault());
        typingInput.addEventListener('cut', e => e.preventDefault());
        typingInput.addEventListener('contextmenu', e => e.preventDefault());
    }

    // --- Settings Update ---
    // Reads settings from UI into state variables
    function readSettings() {
        practiceMode = document.querySelector('input[name="practice-mode"]:checked')?.value || null; // Get value if checked, else null
        targetLevel = parseInt(levelSelect.value, 10);
        targetCount = parseInt(charCountSelect.value, 10);
        targetTime = parseInt(timeDurationSelect.value, 10);
        console.log("Settings Read:", { practiceMode, targetLevel, targetCount, targetTime });
        updateSettingsUI(); // Update visibility based on read mode
    }
    // Updates visibility of measured/timed settings blocks
    function updateSettingsUI() {
        measuredSettings.style.display = (practiceMode === 'measured') ? 'flex' : 'none';
        timedSettings.style.display = (practiceMode === 'timed') ? 'flex' : 'none';
        updateTargetDisplay();
    }
    // Updates the " / target" display
     function updateTargetDisplay() {
         charsTargetDisplay.textContent = (practiceMode === 'measured') ? ` / ${targetCount}` : '';
         charsTargetDisplay.style.display = (practiceMode === 'measured') ? 'inline' : 'none';
     }
     
    // (修正) 採用您提供的 handleFontChange 邏輯
    function handleFontChange(event) {
        // 阻止 radio button 的預設行為，改為手動控制
        if (event) {
             event.preventDefault();
        }

        // 檢查當前狀態並切換
        if (useZhuyinFont === false) {
            useZhuyinFont = true;
            fontNoZhuyinRadio.checked = false;
            fontZhuyinRadio.checked = true;
        } else {
            useZhuyinFont = false;
            fontNoZhuyinRadio.checked = true;
            fontZhuyinRadio.checked = false;
        }
        
        console.log("Font change. Use Zhuyin:", useZhuyinFont);

        textDisplayWindow.classList.toggle('zhuyin-font', useZhuyinFont);
        currentLineHeight = 0; // Force recalculation
        
        if(practiceActive || practiceFinished) {
            renderTextDisplay();
        }
    }


    // --- Practice Logic ---
    // Prepare Practice - Called when a mode radio is clicked
    function preparePractice(event) {
        console.log("Prepare Practice triggered by mode selection.");
        if (practiceActive) return;
        setIdleState(false); // (修正) Call correct function name
        readSettings(); // Read settings
        practicePrepared = true;
         // Ensure the clicked radio remains checked
         if (event && event.target === modeMeasuredRadio) modeMeasuredRadio.checked = true;
         if (event && event.target === modeTimedRadio) modeTimedRadio.checked = true;
         readSettings(); // Re-read to set practiceMode
        showStatusMessage("已選擇模式，請將游標移至輸入框開始練習...", false);
    }

    // Start Practice - Triggered by focusin, only if prepared
    async function startPracticeIfPrepared() {
        if (!practicePrepared || practiceActive || isLoadingQuestions) return;
        console.log("Focus detected and practice prepared, starting...");
        // isLoadingQuestions = true; // <<< (修正) REMOVED
        practiceActive = true; practiceFinished = false; // Set states
        practicePrepared = false; // Consume 'prepared' state

        // --- Update UI for Active State ---
        showStatusMessage('載入題目中...', false);
        textDisplayWindow.innerHTML = '<div class="line upcoming" style="top: 0px;">載入題目中...</div>';
        stopTestBtn.disabled = true;
        modeMeasuredRadio.disabled = true; modeTimedRadio.disabled = true;
        levelSelect.disabled = true; charCountSelect.disabled = true; timeDurationSelect.disabled = true;
        fontNoZhuyinRadio.disabled = true; fontZhuyinRadio.disabled = true;
        typingInput.disabled = true; // <<< Disable input during load

        try {
            // Calculate required length
            let requiredLength = (practiceMode === 'measured') ? (targetCount + 50) : (Math.max(500, (targetTime / 60 * 200) + 50));
            console.log(`[startPractice] Mode: ${practiceMode}, Calculated Target Length: ${requiredLength}`);

            // (修正) fetchQuestions will now set isLoadingQuestions = true itself
            await fetchQuestions(requiredLength);
            if (questionLines.length === 0) { throw new Error('無法載入任何題目。'); }

            // --- Questions Loaded Successfully ---
            console.log("Questions loaded, Practice Starting Now...");
            // isLoadingQuestions is now false (set by fetchQuestions)
            stopTestBtn.disabled = false; // Enable stop button
            typingInput.disabled = false; // <<< Enable input
            showStatusMessage('練習中...', false);
            renderTextDisplay();
            startTime = Date.now(); timeElapsed = 0;
            startTimer(); // Start the timer
            typingInput.focus(); // Ensure focus remains

        } catch (error) {
            console.error("Failed to start practice:", error);
            showStatusMessage(`開始練習失敗：${error.message}`, true);
            // --- Reset to Idle State on Failure ---
            setIdleState(true); // Treat as finished on error
             showStatusMessage(`開始練習失敗：${error.message}`, true); // Re-set error message
             // isLoadingQuestions should be false (set by fetchQuestions finally block)
        }
    }

    function startTimer() { console.log("startTimer called."); if (!practiceActive) { console.warn("startTimer called but practice not active."); return; } clearInterval(timerInterval); timeElapsed = 0; updateTimerDisplay(); updateSpeedDisplay(); timerInterval = setInterval(() => { if (!practiceActive) { clearInterval(timerInterval); return; } timeElapsed = Math.floor((Date.now() - startTime) / 1000); updateTimerDisplay(); updateSpeedDisplay(); if (practiceMode === 'timed' && timeElapsed >= targetTime) { finishPractice(); } }, 1000); }
    function updateTimerDisplay() { let displayTime = (practiceMode === 'timed') ? Math.max(0, targetTime - timeElapsed) : timeElapsed; const prefix = (practiceMode === 'timed') ? '剩餘 ' : ''; const minutes = String(Math.floor(displayTime / 60)).padStart(2, '0'); const seconds = String(displayTime % 60).padStart(2, '0'); timerDisplay.textContent = prefix + `${minutes}:${seconds}`; }
    function updateSpeedDisplay() { const minutes = timeElapsed > 0 ? timeElapsed / 60 : 0; let wpm = 0; if (minutes > 0) { const correctChars = Math.max(0, totalCharsCompleted - totalErrors); wpm = (correctChars / minutes).toFixed(1); } speedDisplay.textContent = wpm; }
    
    // (新) Handle Input - Checks for IME composition
    function handleInput(event) {
        if (!practiceActive || practiceFinished || isLoadingQuestions) return;
        
        // event.isComposing is true while IME is active (e.g., typing Zhuyin)
        if (isComposing) {
            console.log("IME Composition in progress, skipping count update.");
            return; // Don't count intermediate characters
        }
        // If not composing (or composition just finished), update the display
        updateDisplayCharCount();
    }

    // Update Character Count Display (Real-time)
     function updateDisplayCharCount() {
         if (practiceActive) {
             const currentInputLength = typingInput.value.length;
             displayTypedChars = totalCharsCompleted + currentInputLength;
             charsTypedDisplay.textContent = displayTypedChars;
             
             // Check for measured mode completion *here*
             if (practiceMode === 'measured' && displayTypedChars >= targetCount) {
                 console.log(`Measured mode target reached via input (${displayTypedChars} >= ${targetCount}). Finishing.`);
                 finishPractice();
             }
         } else if (!practiceFinished) {
             charsTypedDisplay.textContent = '0';
         }
         // If finished, do nothing, keep final score
     }

     // Handle KeyDown - Process Enter, ignore others
     async function handleKeyDown(event) {
          if (!practiceActive || practiceFinished || isLoadingQuestions) return;
          // (新) Check isComposing *before* checking ignored keys
          if (isComposing) {
              return;
          }
          if (IGNORED_KEYS.includes(event.key)) { console.log("Ignored key:", event.key); return; }

          if (event.key === 'Enter') {
              event.preventDefault(); console.log("Enter pressed, processing line:", currentLineIndex);
              const inputLine = typingInput.value; const expectedLine = questionLines[currentLineIndex];
              if (expectedLine === undefined) { console.warn("Enter pressed but no expected line found (out of lines)."); if (practiceMode === 'measured') { finishPractice(); } return; }
              let lineErrors = 0; for (let i = 0; i < expectedLine.length; i++) { if (i >= inputLine.length || inputLine[i] !== expectedLine[i]) { lineErrors++; } } if (inputLine.length > expectedLine.length) { lineErrors += (inputLine.length - expectedLine.length); }
              totalCharsCompleted += expectedLine.length; totalErrors += lineErrors;
              console.log(`Line ${currentLineIndex}: Errors=${lineErrors}, TotalChars=${totalCharsCompleted}, TotalErrors=${totalErrors}`);
              charsTypedDisplay.textContent = totalCharsCompleted; errorsDisplay.textContent = totalErrors; updateSpeedDisplay();
              
              if (practiceMode === 'measured' && totalCharsCompleted >= targetCount) {
                  console.log(`Measured mode target reached via Enter (${totalCharsCompleted} >= ${targetCount}). Finishing.`);
                  typingInput.value = ''; // Clear input *before* finishing
                  finishPractice();
                  return;
              }
              // Move to Next Line
              currentLineIndex++; typingInput.value = ''; displayTypedChars = totalCharsCompleted;
              renderTextDisplay();
              // Check if we just ran out of lines
              if (currentLineIndex >= questionLines.length) {
                   console.warn("Reached end of pre-fetched lines.");
                   if (practiceMode === 'measured') { finishPractice(); }
                   else if (practiceMode === 'timed') { showStatusMessage('已完成所有題目，計時繼續...', false); typingInput.disabled = true; }
              }
          }
     }

    // Render Text Display - Line based with highlighting and scrolling
     function renderTextDisplay() {
         if (!textDisplayWindow) return; textDisplayWindow.innerHTML = '';
         if (!Array.isArray(questionLines) || questionLines.length === 0) { textDisplayWindow.innerHTML = `<div class="line upcoming">${isLoadingQuestions ? '載入題目中...' : '(練習結束或無題目)'}</div>`; return; }
         // (修正) Removed absolute positioning logic
         for (let i = 0; i < questionLines.length; i++) {
             const lineDiv = document.createElement('div');
             lineDiv.classList.add('line');
             lineDiv.textContent = questionLines[i] || '';
             // (移除) lineDiv.style.top = ...
             if (i < currentLineIndex) { lineDiv.classList.add('typed-correct'); }
             else if (i === currentLineIndex) { lineDiv.classList.add('current'); } // Highlight current line
             else { lineDiv.classList.add('upcoming'); }
             textDisplayWindow.appendChild(lineDiv);
         }
         // (修正) Use scrollIntoView
         const currentLineDiv = textDisplayWindow.querySelector('.line.current');
         if (currentLineDiv) {
             currentLineDiv.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
         }
     }

    // Fetch Questions - Called only once by startPractice
    async function fetchQuestions(minLength) {
        // (修正) Check flag *before* setting it
        if (isLoadingQuestions) { console.log("[fetchQuestions] Fetch already in progress."); return; }
        isLoadingQuestions = true; // <<< Set flag HERE
        
        console.log(`[fetchQuestions] Fetching for level ${targetLevel}, minLength ${minLength}...`);
        try {
            const response = await fetch(`api/get_questions.php?level=${targetLevel}&targetLength=${minLength}`);
            if (!response.ok) { throw new Error(`API 請求失敗 (${response.status})`); }
            const contentType = response.headers.get("content-type"); if (!contentType || !contentType.includes("application/json")) { const text = await response.text(); console.error("Non-JSON response:", text); throw new Error(`API回應格式錯誤 (${contentType})`); }
            
            const lines = await response.json(); // API returns an array of lines
            
            console.log("[fetchQuestions] Raw API response (lines):", JSON.stringify(lines)); if (!Array.isArray(lines)) { throw new Error("API 回應的資料不是有效的陣列"); }
            
            if (lines.length > 0) {
                questionLines = lines; // Directly set the lines
                console.log("[fetchQuestions] Total lines set:", questionLines.length);
            } else {
                 console.warn("[fetchQuestions] No new questions fetched from API.");
                 throw new Error('題庫中找不到任何題目。');
            }
        } catch (error) {
            console.error("[fetchQuestions] Error fetching or processing questions:", error);
             if (!practiceActive) { showStatusMessage('錯誤：載入題目失敗 - ' + error.message, true); }
             else { console.error("Failed to fetch additional questions during practice."); }
             throw error; // Rethrow so caller (startPracticeIfPrepared) knows it failed
        } finally {
             isLoadingQuestions = false; // <<< Reset flag HERE
        }
    }

    // (修正) Finish Practice - Handle timed mode final input
    function finishPractice() {
         if (practiceFinished) return;
        console.log("Finishing Practice...");
        const wasActive = practiceActive;
        practiceActive = false; practiceFinished = true;
        clearInterval(timerInterval); timerInterval = null;

        const finalInputLine = typingInput.value;
        stopTestBtn.disabled = true;
        typingInput.disabled = true;

        // (修正) Final error calculation for unfinished line (check isComposing)
        if (wasActive && finalInputLine.length > 0 && !isComposing) { // <<< (新) Added !isComposing check
             console.log("Calculating errors/chars for final unfinished input (composition ended)...");
             const expectedLine = questionLines[currentLineIndex];
             if (expectedLine) {
                 let finalLineErrors = 0;
                 // Compare only up to the length of the input
                 for (let i = 0; i < finalInputLine.length; i++) {
                     if (i >= expectedLine.length || finalInputLine[i] !== expectedLine[i]) {
                         finalLineErrors++;
                     }
                 }
                 
                 // Only add to totals if it was TIMED mode
                 if (practiceMode === 'timed') {
                     // Add the *attempted* chars from the last line
                     totalCharsCompleted += finalInputLine.length; // Add *input* length
                     totalErrors += finalLineErrors;
                     console.log(`Final input line (timed): ${finalInputLine.length} chars, ${finalLineErrors} errors. New TotalChars=${totalCharsCompleted}, New TotalErrors=${totalErrors}`);
                 }
                 // If measured mode, we only stop if Enter was pressed (handled in handleKeyDown)
                 // or if 'input' event hit target (handled in handleInput)
                 // or if Stop button pressed.
                 else if (practiceMode === 'measured' && totalCharsCompleted < targetCount) { // Stopped early
                      // Don't count partial line chars, but count errors? Docx says count on "Enter"
                      // Let's count errors only if stopped manually.
                      totalErrors += finalLineErrors;
                      console.log(`Final input line (measured/stop): ${finalLineErrors} errors added. Total Chars ${totalCharsCompleted} (unchanged).`);
                 }
             }
         } else if (wasActive && finalInputLine.length > 0 && isComposing) {
              console.warn("Finishing practice while IME is still composing. Ignoring final input:", finalInputLine);
              // Do not count the unfinished composition as chars or errors
         }

        // Calculate WPM based on final official counts
        const finalTime = timeElapsed; const finalChars = totalCharsCompleted; const finalErrors = totalErrors;
        const minutes = finalTime > 0 ? finalTime / 60 : 0; let finalWpm = 0;
        if (minutes > 0) { const correctChars = Math.max(0, finalChars - totalErrors); finalWpm = (correctChars / minutes).toFixed(1); }

        // Update display with final official counts
        updateTimerDisplay();
        charsTypedDisplay.textContent = finalChars;
        errorsDisplay.textContent = finalErrors;
        speedDisplay.textContent = finalWpm;
        typingInput.value = '';

        // Set UI back to Idle State
        setIdleState(true); // Pass true to indicate it's finishing

        console.log("Practice Finished. Score:", { time: finalTime, chars: finalChars, errors: finalErrors, wpm: finalWpm });
    }

    // --- Initialization ---
    initializePractice(); // Run setup

})();
