<?php
session_start();
include 'db_connect.php';
include 'log_helper.php';

header('Content-Type: application/json');

// 權限：僅管理者
if (!isset($_SESSION['user']) || $_SESSION['user']['role'] != 'admin') {
    http_response_code(403);
    echo json_encode(['message' => '權限不足：僅管理者可存取']);
    exit;
}
$currentUser = $_SESSION['user'];
$method = $_SERVER['REQUEST_METHOD'];
$pdo = get_db_connection();
$action = $_REQUEST['action'] ?? null; // Check for action parameter

try {
    // --- (新) 批次建立 (POST with action=batch_create) ---
    if ($method == 'POST' && $action == 'batch_create') {
        if (!isset($_FILES['class_file']) || $_FILES['class_file']['error'] != UPLOAD_ERR_OK) {
            http_response_code(400);
            echo json_encode(['message' => '檔案上傳失敗']);
            exit;
        }

        $fileTmpPath = $_FILES['class_file']['tmp_name'];
        $fileName = $_FILES['class_file']['name'];
        $fileExtension = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));

        $createdCount = 0;
        $errors = [];

        if ($fileExtension == 'csv') {
            if (($handle = fopen($fileTmpPath, 'r')) !== FALSE) {
                $pdo->beginTransaction();
                $header = fgetcsv($handle); // Read header: class_name, grade_level
                if (!$header || !in_array('class_name', $header)) {
                     fclose($handle); // Close the file handle before throwing
                     throw new Exception("CSV 檔案缺少必要的 'class_name' 標頭");
                }
                $line = 1;
                while (($data = fgetcsv($handle)) !== FALSE) {
                    $line++;
                    // Skip empty rows
                    if (empty(array_filter($data))) continue;
                     if (count($header) != count($data)) {
                        $errors[] = "第 $line 行: 欄位數量不符 (期望 " . count($header) . " 個, 實際 " . count($data) . " 個)";
                        continue;
                    }
                    $row = array_combine($header, $data);

                    $className = $row['class_name'] ?? null;
                    $gradeLevel = $row['grade_level'] ?? null;

                    if (empty($className)) {
                        $errors[] = "第 $line 行: 班級名稱為必填";
                        continue;
                    }

                    // Validate grade_level if provided
                    if ($gradeLevel !== null && $gradeLevel !== '' && !in_array($gradeLevel, ['3', '4', '5', '6'])) {
                         $errors[] = "第 $line 行: 年級必須是 3 到 6 之間的數字";
                         continue;
                    }

                    $sql = "INSERT INTO classes (class_name, grade_level) VALUES (?, ?)";
                    $stmt = $pdo->prepare($sql);
                    // Use PDO::PARAM_INT for grade_level if it's a number, otherwise null
                    $stmt->execute([
                        $className,
                        ($gradeLevel !== null && $gradeLevel !== '') ? (int)$gradeLevel : null
                    ]);
                    $createdCount++;
                }
                fclose($handle);
                $pdo->commit();
            } else {
                 throw new Exception("無法開啟 CSV 檔案");
            }
        } else {
             $errors[] = "不支援的檔案格式，僅接受 .csv";
        }

        if ($createdCount > 0) {
            log_operation($pdo, $currentUser['id'], $currentUser['username'], 'BATCH_CREATE_CLASSES', json_encode(['created' => $createdCount, 'errors' => count($errors)]));
        }

        echo json_encode(['success' => true, 'createdCount' => $createdCount, 'errors' => $errors]);
        exit; // End batch create action

    }
    // --- (新) 滙出 (GET with action=export) ---
    else if ($method == 'GET' && $action == 'export') {
        $filename = "classes-export-" . date('ymd') . ".csv"; //

        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');

        $output = fopen('php://output', 'w');
        // Add UTF-8 BOM for Excel compatibility
        fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF));

        // Header row
        fputcsv($output, ['id', 'class_name', 'grade_level']);

        // Data rows
        $stmt = $pdo->query("SELECT id, class_name, grade_level FROM classes ORDER BY grade_level, class_name");
        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            fputcsv($output, $row);
        }
        fclose($output);
        // Optional: Log export action
        // log_operation($pdo, $currentUser['id'], $currentUser['username'], 'EXPORT_CLASSES');
        exit; // End export action
    }

    // --- Original CRUD Operations ---
    switch ($method) {
        // --- 讀取 (GET) ---
        case 'GET':
            // 取得所有班級列表 (for 學生/教師帳號下拉選單)
            if (isset($_GET['fetch']) && $_GET['fetch'] == 'classlist') {
                 $stmt = $pdo->query("SELECT id, class_name FROM classes ORDER BY class_name");
                 $classes = $stmt->fetchAll();
                 echo json_encode($classes);
            }
            // 預設：取得班級管理列表
            else {
                $sql = "SELECT id, class_name, grade_level FROM classes ORDER BY grade_level, class_name";
                $stmt = $pdo->query($sql);
                echo json_encode($stmt->fetchAll());
            }
            break;

        // --- 新增 (POST) ---
        case 'POST':
            $data = json_decode(file_get_contents('php://input'), true);
            if (empty($data['class_name'])) { http_response_code(400); echo json_encode(['message' => '班級名稱為必填']); exit; }

            $sql = "INSERT INTO classes (class_name, grade_level) VALUES (?, ?)";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                $data['class_name'],
                ($data['grade_level'] !== null && $data['grade_level'] !== '') ? (int)$data['grade_level'] : null
             ]);
            $newId = $pdo->lastInsertId();

            log_operation($pdo, $currentUser['id'], $currentUser['username'], 'CLASS_CREATE', json_encode(['id' => $newId, 'name' => $data['class_name']]));
            echo json_encode(['success' => true, 'id' => $newId]);
            break;

        // --- 更新 (PUT) ---
        case 'PUT':
            $data = json_decode(file_get_contents('php://input'), true);
            $targetId = $data['id'];
            if (empty($targetId) || empty($data['class_name'])) { http_response_code(400); echo json_encode(['message' => '缺少 ID 或班級名稱']); exit; }

            $sql = "UPDATE classes SET class_name = ?, grade_level = ? WHERE id = ?";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                $data['class_name'],
                 ($data['grade_level'] !== null && $data['grade_level'] !== '') ? (int)$data['grade_level'] : null,
                $targetId
            ]);

            log_operation($pdo, $currentUser['id'], $currentUser['username'], 'CLASS_UPDATE', json_encode(['id' => $targetId, 'name' => $data['class_name']]));
            echo json_encode(['success' => true]);
            break;

        // --- 刪除 (DELETE) ---
        case 'DELETE':
            $targetId = $_GET['id'] ?? null;
            if (empty($targetId)) { http_response_code(400); echo json_encode(['message' => '缺少 ID']); exit; }

            // Check if any user belongs to this class
            $checkSql = "SELECT COUNT(*) FROM users WHERE JSON_CONTAINS(class, CAST(? AS CHAR))"; // Use CHAR for compatibility
            $checkStmt = $pdo->prepare($checkSql);
            $checkStmt->execute([$targetId]);
	    if ($checkStmt->fetchColumn() <= 0) {
		    $sql = "DELETE FROM classes WHERE id = ?";
		    $stmt = $pdo->prepare($sql);
		    $stmt->execute([$targetId]);

		    log_operation($pdo, $currentUser['id'], $currentUser['username'], 'CLASS_DELETE', json_encode(['id' => $targetId]));
		    echo json_encode(['success' => true]);
	    } else {
                 http_response_code(400);
                 echo json_encode(['message' => '無法刪除：尚有使用者屬於此班級。請先將使用者移至其他班級。']);
            }

            break;

        default:
            // If action wasn't handled above, it's an invalid method for standard CRUD
             if ($action == null) {
                 http_response_code(405);
                 echo json_encode(['message' => '方法不允許']);
             } else {
                 http_response_code(400);
                 echo json_encode(['message' => '無效的操作']);
             }
            break;
    }
} catch (\PDOException $e) {
    if ($pdo->inTransaction()) { $pdo->rollBack(); } // Rollback batch on error
    http_response_code(500);
    if ($e->getCode() == 23000) { echo json_encode(['message' => '錯誤：該班級名稱 (class_name) 已經存在']); }
    else { echo json_encode(['message' => '資料庫錯誤: ' . $e->getMessage()]); }
} catch (\Exception $e) { // Catch general exceptions for file handling
    if ($pdo->inTransaction()) { $pdo->rollBack(); }
    http_response_code(400); // Bad request for file errors
    echo json_encode(['message' => '處理檔案時發生錯誤: ' . $e->getMessage(), 'errors' => $errors ?? []]);
}

