<?php
// api/events.php
// 對齊你的 events 欄位（title/owner_id/start_time/...）
// 不使用 exit()/die()；改以 run() return 統一輸出一次，避免重複輸出或 405 尾巴。
// 亦處理 GET/HEAD/OPTIONS/POST/DELETE，且子表 event_items 自動建立（失敗忽略）。

session_start();
header('Content-Type: application/json; charset=utf-8');
@include_once __DIR__ . '/db_connect.php';

function pdo_conn(){
  static $pdo = null;
  if ($pdo instanceof PDO) return $pdo;
  if (function_exists('get_pdo')) { $pdo = get_pdo(); if ($pdo instanceof PDO) return $pdo; }
  if (function_exists('getPDO')) { $pdo = getPDO(); if ($pdo instanceof PDO) return $pdo; }
  if (isset($GLOBALS['pdo']) && $GLOBALS['pdo'] instanceof PDO) { $pdo = $GLOBALS['pdo']; return $pdo; }
  $host = defined('DB_HOST') ? DB_HOST : 'localhost';
  $name = defined('DB_NAME') ? DB_NAME : '';
  $user = defined('DB_USER') ? DB_USER : 'root';
  $pass = defined('DB_PASS') ? DB_PASS : '';
  $charset = defined('DB_CHARSET') ? DB_CHARSET : 'utf8mb4';
  if ($name === '') return [500, ['success'=>false,'message'=>'伺服器錯誤：資料庫名稱未設定（缺少 DB_NAME）']];
  try{
    $dsn = "mysql:host={$host};dbname={$name};charset={$charset}";
    $pdo = new PDO($dsn, $user, $pass, [
      PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
      PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
    ]);
    return $pdo;
  }catch(Throwable $e){
    return [500, ['success'=>false,'message'=>'資料庫連線失敗：'.$e->getMessage()]];
  }
}

function run(){
  // 統一回傳格式：[httpCode(int), body(array|null)]
  $method = strtoupper($_SERVER['REQUEST_METHOD'] ?? 'GET');
  $action = isset($_GET['action']) ? trim((string)$_GET['action']) : '';

  $pdo = pdo_conn();
  if (is_array($pdo)) return $pdo; // 連線錯誤：直接回傳 [code, body]
  try { $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION); } catch(Throwable $__) {}

  // 嘗試建立子表（無權限或已存在 → 忽略）
  try {
    $pdo->exec("
      CREATE TABLE IF NOT EXISTS event_items (
        id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
        event_id INT NOT NULL,
        item_order INT NOT NULL DEFAULT 1,
        eligibility VARCHAR(64) NOT NULL DEFAULT 'all',
        mode ENUM('measured','timed') NOT NULL,
        char_quota INT DEFAULT NULL,
        time_limit_seconds INT DEFAULT NULL,
        question_source ENUM('random','fixed') NOT NULL DEFAULT 'random',
        grade_scope VARCHAR(64) NOT NULL DEFAULT 'all',
        fixed_bank_ids JSON DEFAULT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY idx_event_items_event (event_id),
        CONSTRAINT fk_event_items_event FOREIGN KEY (event_id) REFERENCES events(id) ON DELETE CASCADE
      )
    ");
    try { $pdo->exec("CREATE INDEX idx_event_items_order ON event_items(event_id, item_order)"); } catch(Throwable $__) {}
  } catch (Throwable $__) {}

  // CORS/預檢
  if ($method === 'OPTIONS') {
    return [200, ['success'=>true]];
  }
  if ($method === 'HEAD') {
    return [200, null]; // 不需要 body
  }

  // 讀取列表 / 單筆
  if ($method === 'GET') {
    if ($action === 'get_event') {
      $id = intval($_GET['id'] ?? 0);
      if (!$id) return [400, ['success'=>false,'message'=>'缺少 id']];

      $stmt = $pdo->prepare("
        SELECT
          id,
          title AS name,
          event_type,
          start_time AS start_at,
          end_time AS end_at,
          reg_start_time AS reg_start_at,
          reg_end_time AS reg_end_at,
          details AS notes,
          owner_id
        FROM events
        WHERE id=?
      ");
      $stmt->execute([$id]);
      $event = $stmt->fetch();
      if (!$event) return [404, ['success'=>false,'message'=>'查無此活動']];

      $items = [];
      try {
        $stmt2 = $pdo->prepare("
          SELECT id, item_order, eligibility, mode, char_quota, time_limit_seconds, question_source, grade_scope, fixed_bank_ids
          FROM event_items
          WHERE event_id=?
          ORDER BY item_order
        ");
        $stmt2->execute([$id]);
        $items = $stmt2->fetchAll();
      } catch (Throwable $__) {
        $items = [];
      }
      if (!$items) {
        $items = [[
          'id'=>null,'item_order'=>1,'eligibility'=>'all','mode'=>'measured',
          'char_quota'=>500,'time_limit_seconds'=>null,'question_source'=>'random',
          'grade_scope'=>'all','fixed_bank_ids'=>null
        ]];
      }

      return [200, ['success'=>true,'event'=>$event,'items'=>$items]];
    }

    // 清單（action 空或 list）
    if ($action === '' || $action === 'list') {
      $rows = $pdo->query("
        SELECT
          id,
          title AS name,
          event_type,
          start_time AS start_at,
          end_time AS end_at,
          reg_start_time AS reg_start_at,
          reg_end_time AS reg_end_at,
          owner_id
        FROM events
        ORDER BY id DESC
      ")->fetchAll();
      return [200, $rows];
    }

    return [405, ['success'=>false,'message'=>'不支援的 GET 請求']];
  }

  // 新增/更新
  if ($method === 'POST') {
    if (empty($_SESSION['user'])) return [401, ['success'=>false,'message'=>'未登入']];
    $input = json_decode(file_get_contents('php://input'), true);
    if (!$input) return [400, ['success'=>false,'message'=>'無法解析送出的資料（非 JSON？）']];
    if (($input['action'] ?? '') !== 'save_event_with_items') return [405, ['success'=>false,'message'=>'不支援的動作']];

    $d = $input['data'] ?? null;
    if (!$d) return [400, ['success'=>false,'message'=>'缺少資料']];

    $name        = trim($d['name'] ?? ''); // -> title
    $event_type  = in_array(($d['event_type'] ?? 'test'), ['test','competition']) ? $d['event_type'] : 'test';
    $start_at    = $d['start_at'] ?? null; // -> start_time
    $end_at      = $d['end_at'] ?? null;   // -> end_time
    $reg_start   = $event_type==='competition' ? ($d['reg_start_at'] ?? null) : null; // -> reg_start_time
    $reg_end     = $event_type==='competition' ? ($d['reg_end_at'] ?? null)   : null; // -> reg_end_time
    $notes       = $d['notes'] ?? null;    // -> details
    $items       = $d['items'] ?? [];

    if ($name === '' || !$start_at || !$end_at) return [400, ['success'=>false,'message'=>'欄位不足']];
    if (!is_array($items) || !count($items)) return [400, ['success'=>false,'message'=>'請至少建立 1 個活動項目']];

    $settingsJson = '{}';    // NOT NULL
    $eligibilityJson = null; // 目前不用（子表管理）

    try {
      $pdo->beginTransaction();

      if (!empty($d['id'])) {
        $event_id = intval($d['id']);
        $stmt = $pdo->prepare("
          UPDATE events
          SET event_type=?, title=?, owner_id=owner_id, start_time=?, end_time=?, reg_start_time=?, reg_end_time=?, details=?, settings=?
          WHERE id=?
        ");
        $stmt->execute([$event_type, $name, $start_at, $end_at, $reg_start, $reg_end, $notes, $settingsJson, $event_id]);
        $pdo->prepare("DELETE FROM event_items WHERE event_id=?")->execute([$event_id]);
      } else {
        $stmt = $pdo->prepare("
          INSERT INTO events (event_type, title, owner_id, start_time, end_time, reg_start_time, reg_end_time, details, eligibility, settings)
          VALUES (?,?,?,?,?,?,?,?,?,?)
        ");
        $stmt->execute([
          $event_type, $name, $_SESSION['user']['id'] ?? 0,
          $start_at, $end_at, $reg_start, $reg_end,
          $notes, $eligibilityJson, $settingsJson
        ]);
        $event_id = $pdo->lastInsertId();
      }

      $ins = $pdo->prepare("
        INSERT INTO event_items
        (event_id, item_order, eligibility, mode, char_quota, time_limit_seconds, question_source, grade_scope, fixed_bank_ids)
        VALUES (?,?,?,?,?,?,?,?,?)
      ");
      $order = 1;
      foreach ($items as $it) {
        $mode = ($it['mode'] ?? 'measured') === 'timed' ? 'timed' : 'measured';
        $quota = intval($it['quota'] ?? 0);
        $char_quota = $mode === 'measured' ? $quota : null;
        $time_limit = $mode === 'timed' ? $quota : null;

        $elig = $it['eligibility'] ?? 'all';
        if (is_array($elig)) $elig = implode(',', array_filter($elig));
        $scope = $it['scope'] ?? 'all';
        if (is_array($scope)) $scope = implode(',', array_filter($scope));

        $src = ($it['source'] ?? 'random') === 'fixed' ? 'fixed' : 'random';
        $fixed = null;
        if (!empty($it['fixed_bank_ids']) && is_array($it['fixed_bank_ids'])) {
          $ids = array_values(array_unique(array_map('intval', $it['fixed_bank_ids'])));
          $ids = array_values(array_filter($ids, fn($v)=>$v>0));
          $fixed = json_encode($ids, JSON_UNESCAPED_UNICODE);
        }

        $ins->execute([$event_id, $order++, $elig ?: 'all', $mode, $char_quota, $time_limit, $src, $scope ?: 'all', $fixed]);
      }

      $pdo->commit();
      return [200, ['success'=>true,'event_id'=>$event_id]];
    } catch (Throwable $e) {
      try { $pdo->rollBack(); } catch (Throwable $__) {}
      return [500, ['success'=>false,'message'=>'伺服器錯誤：'.$e->getMessage()]];
    }
  }

  // 刪除
  if ($method === 'DELETE') {
    if (empty($_SESSION['user'])) return [401, ['success'=>false,'message'=>'未登入']];
    $id = intval($_GET['id'] ?? 0);
    if (!$id) return [400, ['success'=>false,'message'=>'缺少 id']];
    try {
      $stmt = $pdo->prepare("DELETE FROM events WHERE id=?");
      $stmt->execute([$id]);
      return [200, ['success'=>true]];
    } catch (Throwable $e) {
      return [500, ['success'=>false,'message'=>'伺服器錯誤：'.$e->getMessage()]];
    }
  }

  return [405, ['success'=>false,'message'=>'不支援的請求方法']];
}

// ---- 主流程（統一輸出一次）----
list($httpCode, $body) = run();
http_response_code($httpCode);
if ($body !== null) {
  echo json_encode($body, JSON_UNESCAPED_UNICODE);
}

