<?php
// api/get_practice_logs.php
// 取得當前登入使用者的中文輸入練習成績記錄（供「成績」浮窗使用）
session_start();

header('Content-Type: application/json; charset=utf-8');

// 與專案一致的 DB / session 引用
require_once __DIR__ . '/db_connect.php';
require_once __DIR__ . '/log_helper.php';

$data = (isset($_SESSION['user'])) ? $_SESSION['user'] : NULL;

file_put_contents("/tmp/typing.log", "DATA: ".json_encode($data)."\n", FILE_APPEND);

// check-session.php 會輸出一個 JSON，例如：
// {
//   "loggedIn": true,
//   "id": 1,
//   "username": "...",
//   "name": "...",
//   "role": "...",
//   "class": "...",
//   "grade": 3   // 你的新版有加 grade
// }

// 若 JSON 解析失敗，或 loggedIn 為 false，就把原本的 JSON 原樣丟回去
if (!$data) {
    // 確保狀態碼是 401
    if (!headers_sent()) {
        http_response_code(401);
    }
    echo $json ?: json_encode([
        'loggedIn' => false,
        'message'  => '未授權'
    ], JSON_UNESCAPED_UNICODE);
    exit;
}

// 已登入，從 check-session 的 JSON 取出帳號資訊
$currentUserId   = $data['id'];
$currentUsername = $data['username'] ?? '';
$currentGrade    = $data['grade'] ?? null;

// 讀取查詢區間參數（配合前端 typing.js）
$range = $_GET['range'] ?? 'today';  // today / week / month / custom
$start = $_GET['start'] ?? '';       // YYYY-MM-DD
$end   = $_GET['end'] ?? '';         // YYYY-MM-DD

try {
    // 使用台北時間做邏輯，再轉成 UTC 去查 DB
    $tzLocal = new DateTimeZone('Asia/Taipei');
    $tzUtc   = new DateTimeZone('UTC');

    switch ($range) {
        case 'week':
            // 本週一 ~ 本週日（依台北時間）
            $startLocal = new DateTime('today', $tzLocal);
            $weekday    = (int)$startLocal->format('N'); // 1=Mon..7=Sun
            $startLocal->modify('-' . ($weekday - 1) . ' day');
            $endLocal = clone $startLocal;
            $endLocal->modify('+6 day');
            break;

        case 'month':
            // 本月 1 號 ~ 本月最後一天（台北時間）
            $startLocal = new DateTime('first day of this month', $tzLocal);
            $endLocal   = new DateTime('last day of this month', $tzLocal);
            break;

        case 'custom':
            if (empty($start) || empty($end)) {
                echo json_encode([
                    'result' => false,
                    'data'   => [],
                    'msg'    => '自定區間需指定起訖日期'
                ], JSON_UNESCAPED_UNICODE);
                exit;
            }
            $startLocal = new DateTime($start, $tzLocal);
            $endLocal   = new DateTime($end,   $tzLocal);
            break;

        case 'today':
        default:
            $startLocal = new DateTime('today', $tzLocal);
            $endLocal   = new DateTime('today', $tzLocal);
            break;
    }

    // 將一天區間定為 00:00:00 ~ 23:59:59（台北時間）
    $startLocal->setTime(0, 0, 0);
    $endLocal->setTime(23, 59, 59);

    // 轉換成 UTC，因為 DB 的 start_time / end_time 是 UTC
    $startUtc = clone $startLocal;
    $startUtc->setTimezone($tzUtc);
    $endUtc = clone $endLocal;
    $endUtc->setTimezone($tzUtc);

    $startStr = $startUtc->format('Y-m-d H:i:s');
    $endStr   = $endUtc->format('Y-m-d H:i:s');

    // 3. 查詢 DB
    $pdo = get_db_connection();

    // ★ 表名與欄位請確認有照你實際建立的來：
    //    這裡假設是 typing_practice_logs，欄位：
    //    id, user_id, start_time, end_time, duration_seconds,
    //    practice_mode, level, target_chars, target_seconds,
    //    total_chars, error_chars, correct_chars, wpm, finish_reason ...
    $sql = "
        SELECT
            start_time,
            end_time,
            duration_seconds,
            practice_mode,
            grade,
            CASE
                WHEN practice_mode = 'measured' THEN target_chars
                WHEN practice_mode = 'timed'    THEN target_seconds
                ELSE NULL
            END AS target_value,
            input_text,
            total_chars,
            error_chars,
            wpm
        FROM typing_practice_logs
        WHERE user_id = :uid
          AND start_time BETWEEN :st AND :ed
        ORDER BY start_time ASC
    ";

    $stmt = $pdo->prepare($sql);
    $stmt->execute([
        ':uid' => $currentUserId,
        ':st'  => $startStr,
        ':ed'  => $endStr,
    ]);

    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

    echo json_encode([
        'result' => true,
        'data'   => $rows,
        'msg'    => ''
    ], JSON_UNESCAPED_UNICODE);

} catch (\Throwable $e) {
    http_response_code(500);
    echo json_encode([
        'result' => false,
        'data'   => [],
        'msg'    => '伺服器錯誤：' . $e->getMessage()
    ], JSON_UNESCAPED_UNICODE);
}

