/* typing_project.sql */

/* 警告：此指令將刪除現有資料庫 */
DROP DATABASE IF EXISTS typing_gemini;

/* 建立資料庫 */
CREATE DATABASE IF NOT EXISTS typing_gemini CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE typing_gemini;

/* --- 1. 帳號與權限 --- */

/* 班級管理表 (新架構) */
CREATE TABLE classes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    class_name VARCHAR(100) NOT NULL UNIQUE COMMENT '班級名稱 (例如: C101)',
    grade_level INT NULL COMMENT '所屬年級'
    /* (新) 移除 teacher_ids 欄位 */
) COMMENT '班級管理表 (管理者維護)';

/* 使用者帳號表 (新架構) */
CREATE TABLE users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(100) NOT NULL UNIQUE COMMENT '帳號',
    name VARCHAR(100) NOT NULL COMMENT '儲存姓名',
    class JSON DEFAULT NULL COMMENT '班級ID陣列 (教師:[1,2], 學生:[1])',
    role ENUM('admin', 'teacher', 'student') NOT NULL COMMENT '角色',
    grade INT DEFAULT NULL COMMENT '學生年級 (3-6)',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    update_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT '最後更新時間',
    is_active BOOLEAN DEFAULT TRUE COMMENT '帳號是否啟用'
) COMMENT '使用者帳號表';

/* ... (login_tokens, operation_logs, announcements, test_bank ... 同前) ... */
CREATE TABLE login_tokens (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    token CHAR(5) NOT NULL,
    expires_at TIMESTAMP NOT NULL,
    is_used BOOLEAN DEFAULT FALSE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);
CREATE TABLE operation_logs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NULL,
    username VARCHAR(100),
    action VARCHAR(255) NOT NULL,
    details JSON NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);
CREATE TABLE announcements (
    id INT AUTO_INCREMENT PRIMARY KEY,
    title VARCHAR(255) NOT NULL,
    content TEXT,
    created_by INT NOT NULL,
    start_time TIMESTAMP NOT NULL,
    end_time TIMESTAMP NOT NULL,
    FOREIGN KEY (created_by) REFERENCES users(id)
);

/* 題庫相關資料表 */
CREATE TABLE test_bank (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL COMMENT '題目名稱',
    type_id INT NULL COMMENT '題目類型 ID (FK to question_types)', -- Stores ID now
    content TEXT NOT NULL COMMENT '題目內容',
    length INT NOT NULL DEFAULT 0 COMMENT '題目總字數(含標點)',
    level INT NOT NULL COMMENT '程度/適用年級 (3-6)', -- Renamed from grade
    includes_punctuation BOOLEAN DEFAULT TRUE COMMENT '是否包含標點符號',
    created_by INT NOT NULL COMMENT '建立者 (管理者或教師)',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    update_at TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP COMMENT '最後編修日期時間',
    FOREIGN KEY (created_by) REFERENCES users(id),
    INDEX idx_created_by (created_by) -- Added index
) COMMENT '中文輸入題庫';

CREATE TABLE question_types (
    id INT AUTO_INCREMENT PRIMARY KEY,
    type_name VARCHAR(50) NOT NULL UNIQUE COMMENT '類型名稱'
) COMMENT '題庫題目類型表';

INSERT INTO question_types (type_name) VALUES ('文章'),('古詩詞賦'),('現代詩'),('名言');
/****************/

CREATE TABLE tests (
    id INT AUTO_INCREMENT PRIMARY KEY,
    title VARCHAR(255) NOT NULL,
    teacher_id INT NOT NULL,
    start_time TIMESTAMP NOT NULL,
    end_time TIMESTAMP NOT NULL,
    mode ENUM('timed', 'measured') NOT NULL,
    settings JSON NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (teacher_id) REFERENCES users(id)
);
CREATE TABLE test_questions (
    test_id INT NOT NULL,
    question_id INT NOT NULL,
    PRIMARY KEY (test_id, question_id),
    FOREIGN KEY (test_id) REFERENCES tests(id) ON DELETE CASCADE,
    FOREIGN KEY (question_id) REFERENCES test_bank(id) ON DELETE CASCADE
);
CREATE TABLE test_participants (
    test_id INT NOT NULL,
    student_id INT NOT NULL,
    PRIMARY KEY (test_id, student_id),
    FOREIGN KEY (test_id) REFERENCES tests(id) ON DELETE CASCADE,
    FOREIGN KEY (student_id) REFERENCES users(id) ON DELETE CASCADE
);
CREATE TABLE test_submissions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    student_id INT NOT NULL,
    test_id INT NOT NULL,
    time_taken_seconds INT NOT NULL,
    chars_completed INT NOT NULL,
    errors INT NOT NULL,
    wpm DECIMAL(5, 1) NOT NULL,
    submitted_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (student_id) REFERENCES users(id),
    FOREIGN KEY (test_id) REFERENCES tests(id)
);
CREATE TABLE competitions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    title VARCHAR(255) NOT NULL,
    organizer_id INT NOT NULL,
    details TEXT,
    reg_start_time TIMESTAMP NOT NULL,
    reg_end_time TIMESTAMP NOT NULL,
    event_time TIMESTAMP NOT NULL,
    eligibility_rules JSON,
    mode_settings JSON NOT NULL,
    FOREIGN KEY (organizer_id) REFERENCES users(id)
);
CREATE TABLE competition_registrations (
    id INT AUTO_INCREMENT PRIMARY KEY,
    student_id INT NOT NULL,
    competition_id INT NOT NULL,
    registered_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (student_id) REFERENCES users(id),
    FOREIGN KEY (competition_id) REFERENCES competitions(id)
);
CREATE TABLE competition_scores (
    id INT AUTO_INCREMENT PRIMARY KEY,
    student_id INT NOT NULL,
    competition_id INT NOT NULL,
    time_taken_seconds INT NOT NULL,
    chars_completed INT NOT NULL,
    errors INT NOT NULL,
    wpm DECIMAL(5, 1) NOT NULL,
    submitted_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (student_id) REFERENCES users(id),
    FOREIGN KEY (competition_id) REFERENCES competitions(id)
);

/* 新架構：合併 `tests` 和 `competitions` 為 `events` */
/* 警告：執行此操作前，請先手動 (DROP) 刪除舊的 `tests`, `competitions`, `test_participants`, `test_submissions`, `competition_registrations`, `competition_scores` 資料表 */

/* 1. (新) 合併的「活動」資料表 */
CREATE TABLE events (
    id INT AUTO_INCREMENT PRIMARY KEY,
    event_type ENUM('test', 'competition') NOT NULL COMMENT '活動類型 (測驗或比賽)',
    title VARCHAR(255) NOT NULL COMMENT '活動名稱',
    owner_id INT NOT NULL COMMENT '建立者 (管理者或教師)',
    start_time TIMESTAMP NOT NULL COMMENT '活動開始時間',
    end_time TIMESTAMP NOT NULL COMMENT '活動結束時間',

    -- 僅「比賽」使用 (測驗時為 NULL) --
    reg_start_time TIMESTAMP NULL COMMENT '報名開始時間',
    reg_end_time TIMESTAMP NULL COMMENT '報名結束時間',
    details TEXT NULL COMMENT '其他注意事項 (對應 "活動辦法")',

    -- 共用欄位 (JSON 格式儲存) --

    /* eligibility: 參與資格
      {
         "type": "all" | "grade" | "class",
         "grades": ["3", "4"], (年級)
         "class_ids": [1, 2] (班級ID)
      }
    */
    eligibility JSON NULL COMMENT '參與資格',

    /* settings: 型式與題目設定
      {
        "mode": "timed" | "measured", (限時/計時)
        "duration": 60, (限時：秒數)
        "char_count": 500, (計時：字數)
        "question_source": "random" | "specific", (隨機/指定)
        "random_grades": ["3", "4"], (隨機：年級)
        "specific_content": "..." (指定：上傳內容)
      }
    */
    settings JSON NOT NULL COMMENT '型式與題目設定',

    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (owner_id) REFERENCES users(id)
) COMMENT '合併的活動(測驗/比賽)設定表';


-- 新增活動子項目（多組設定）
CREATE TABLE IF NOT EXISTS event_items (
  id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  event_id INT NOT NULL,
  item_order INT NOT NULL DEFAULT 1,
  -- 資格：all / 3 / 4 / 5 / 6（後續若需要更細，可改 JSON）
  eligibility VARCHAR(16) NOT NULL DEFAULT 'all',
  -- 模式：計時=measured（打到字數）、限時=timed（秒）
  mode ENUM('measured', 'timed') NOT NULL,
  char_quota INT DEFAULT NULL,       -- 計時：必填字數
  time_limit_seconds INT DEFAULT NULL, -- 限時：必填秒
  -- 題目來源：random（隨機）/ fixed（指定）
  question_source ENUM('random','fixed') NOT NULL DEFAULT 'random',
  -- 題目範圍：all / 3 / 4 / 5 / 6 / 34 / 56（可先這樣；必要時改 JSON）
  grade_scope VARCHAR(16) NOT NULL DEFAULT 'all',
  -- 指定題目（可選）：以 JSON 陣列存 test_bank.id
  fixed_bank_ids JSON DEFAULT NULL,

  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (id),
  KEY idx_event_items_event (event_id),
  CONSTRAINT fk_event_items_event
    FOREIGN KEY (event_id) REFERENCES events(id) ON DELETE CASCADE
);

-- 常用查詢索引
CREATE INDEX idx_event_items_order ON event_items(event_id, item_order);


/* 2. (新) 合併的「活動報名/參與者」資料表 */
CREATE TABLE event_participants (
    id INT AUTO_INCREMENT PRIMARY KEY,
    event_id INT NOT NULL COMMENT '活動 ID',
    student_id INT NOT NULL COMMENT '學生 ID',
    registered_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP COMMENT '報名/加入時間',

    UNIQUE KEY unique_event_student (event_id, student_id), -- 確保同一學生只能參加一次
    FOREIGN KEY (event_id) REFERENCES events(id) ON DELETE CASCADE,
    FOREIGN KEY (student_id) REFERENCES users(id) ON DELETE CASCADE
) COMMENT '活動參與者(報名)表';


/* 3. (新) 合併的「成績」資料表 */
CREATE TABLE event_submissions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    event_id INT NOT NULL COMMENT '活動 ID',
    student_id INT NOT NULL COMMENT '學生 ID',
    time_taken_seconds INT NOT NULL COMMENT '耗時多少秒',
    chars_completed INT NOT NULL COMMENT '完成字數 (依規則計算)',
    errors INT NOT NULL COMMENT '錯幾個字',
    wpm DECIMAL(5, 1) NOT NULL COMMENT '每分鐘正確字數',
    submitted_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    FOREIGN KEY (event_id) REFERENCES events(id) ON DELETE CASCADE,
    FOREIGN KEY (student_id) REFERENCES users(id) ON DELETE CASCADE
) COMMENT '活動(測驗/比賽)成績表';

/* 4. (新) 練習的「成績」資料表 */
CREATE TABLE `typing_practice_logs` (
  `id` BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
  `user_id` BIGINT UNSIGNED NOT NULL,
  `username` VARCHAR(64) NOT NULL,
  `grade` TINYINT UNSIGNED NULL,
  `practice_mode` ENUM('measured','timed') NOT NULL COMMENT 'measured=計字數, timed=限時',
  `target_chars` INT UNSIGNED NULL COMMENT '指定字數（計時模式用）',
  `target_seconds` INT UNSIGNED NULL COMMENT '練習秒數（限時模式用）',
  `start_time` DATETIME NOT NULL,
  `end_time` DATETIME NOT NULL,
  `duration_seconds` INT UNSIGNED NOT NULL COMMENT '實際耗時（秒）',
  `question_text` MEDIUMTEXT NULL COMMENT '題目完整內容（當次載入的題目）',
  `input_text` MEDIUMTEXT NULL COMMENT '實際完成內容（逐行輸入合併）',
  `total_chars` INT UNSIGNED NOT NULL DEFAULT 0 COMMENT '總字數',
  `correct_chars` INT UNSIGNED NOT NULL DEFAULT 0 COMMENT '正確字數',
  `error_chars` INT UNSIGNED NOT NULL DEFAULT 0 COMMENT '錯誤字數',
  `wpm` DECIMAL(5,1) NOT NULL DEFAULT 0.0 COMMENT '每分鐘正確字數',
  `finish_reason` VARCHAR(32) NOT NULL DEFAULT 'unknown' COMMENT 'manual-stop / auto-target / auto-timeout / auto-no-more-questions / other',
  `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
  PRIMARY KEY (`id`),
  KEY `idx_user_time` (`user_id`, `start_time`),
  KEY `idx_mode` (`practice_mode`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COMMENT '練習的「成績」資料表';

/* --- 5. 插入預設資料 (新架構) --- */

/* (新) 建立預設班級 (id: 1, 2) */
INSERT INTO classes (id, class_name, grade_level) VALUES
(1, 'C101', 3),
(2, 'C102', 3);

/* (新) 系統預設帳號 (教師 class: [1,2], 學生 class: [1]) */
INSERT INTO users (username, name, class, role, grade, is_active) VALUES
('typing', '系統管理者', NULL, 'admin', NULL, TRUE),
('teacher', '預設教師', '[1, 2]', 'teacher', NULL, TRUE),
('student', '預設學生', '[1]', 'student', 3, TRUE);
