<?php
session_start();
include 'db_connect.php';
include 'log_helper.php';

header('Content-Type: application/json');

// --- Access Control ---
if (!isset($_SESSION['user']) || !in_array($_SESSION['user']['role'], ['admin', 'teacher'])) { http_response_code(403); echo json_encode(['message' => '權限不足：僅管理者或教師可存取']); exit; }
$currentUser = $_SESSION['user'];
$method = $_SERVER['REQUEST_METHOD'];
$pdo = get_db_connection();
// Initialize variables used across methods/catches
$targetId = null;
$typeName = null;
$logDetails = [];


try {
	switch ($method) {
		// --- GET ---
	case 'GET':
		$stmt = $pdo->query("SELECT id, type_name FROM question_types ORDER BY id");
		$types = $stmt->fetchAll();
		echo json_encode($types, JSON_UNESCAPED_UNICODE);
		break;

		// --- POST ---
	case 'POST':
		$data = json_decode(file_get_contents('php://input'), true);
		$typeName = trim($data['type_name'] ?? '');
		if (empty($typeName)) { /* ... validation ... */ http_response_code(400); echo json_encode(['message' => '類型名稱不得為空']); exit; }
		if (mb_strlen($typeName, 'UTF-8') > 50) { /* ... validation ... */ http_response_code(400); echo json_encode(['message' => '類型名稱不得超過 50 字']); exit; }

		$sql = "INSERT INTO question_types (type_name) VALUES (?)";
		$stmt = $pdo->prepare($sql);
		$stmt->execute([$typeName]);
		$newId = $pdo->lastInsertId();

		log_operation($pdo, $currentUser['id'], $currentUser['username'], 'QTYPE_CREATE_SUCCESS', json_encode(['id' => $newId, 'name' => $typeName]));
		echo json_encode(['id' => $newId, 'type_name' => $typeName], JSON_UNESCAPED_UNICODE);
		break;

		// --- PUT ---
	case 'PUT':
		$data = json_decode(file_get_contents('php://input'), true);
		$targetId = $data['id'] ?? null;
		$typeName = trim($data['type_name'] ?? '');
		$logDetails = ['id' => $targetId, 'name' => $typeName];
		if (empty($targetId)) { /* ... validation ... */ http_response_code(400); echo json_encode(['message' => '缺少類型 ID']); exit; }
		if (empty($typeName)) { /* ... validation ... */ http_response_code(400); echo json_encode(['message' => '類型名稱不得為空']); exit; }
		if (mb_strlen($typeName, 'UTF-8') > 50) { /* ... validation ... */ http_response_code(400); echo json_encode(['message' => '類型名稱不得超過 50 字']); exit; }

		$sql = "UPDATE question_types SET type_name = ? WHERE id = ?";
		$stmt = $pdo->prepare($sql);
		$stmt->execute([$typeName, $targetId]);

		log_operation($pdo, $currentUser['id'], $currentUser['username'], 'QTYPE_UPDATE_SUCCESS', json_encode($logDetails));
		echo json_encode(['id' => $targetId, 'type_name' => $typeName], JSON_UNESCAPED_UNICODE);
		break;

		// --- DELETE ---
	case 'DELETE':
		$targetId = $_GET['id'] ?? null; $logDetails = ['id' => $targetId];
		if (empty($targetId)) {
			/* ... validation ... */ 
			http_response_code(400); 
			echo json_encode(['message' => '缺少類型 ID']);
		} else {
			// Get type name for logging and check
			$stmtName = $pdo->prepare("SELECT type_name FROM question_types WHERE id = ?");
			$stmtName->execute([$targetId]);
			$typeName = $stmtName->fetchColumn();
			if ($typeName) {
				$logDetails['name'] = $typeName;
				// Check if type_id is in use
				$checkSql = "SELECT COUNT(*) FROM test_bank WHERE type_id = ?"; $checkStmt = $pdo->prepare($checkSql); $checkStmt->execute([$targetId]);
				if ($checkStmt->fetchColumn() > 0) {
					log_operation(
						$pdo,
						$currentUser['id'],
						$currentUser['username'],
						'QTYPE_DELETE_FAIL',
						json_encode(array_merge($logDetails, ['error' => 'Type in use']))
					);
					http_response_code(400);
					echo json_encode(['message' => '無法刪除：尚有題目使用此類型。']);
				} else {
					// Delete
					$sql = "DELETE FROM question_types WHERE id = ?"; $stmt = $pdo->prepare($sql); $stmt->execute([$targetId]);

					if ($stmt->rowCount() > 0) {
						log_operation($pdo, $currentUser['id'], $currentUser['username'], 'QTYPE_DELETE_SUCCESS', json_encode($logDetails));
						echo json_encode(['success' => true]);
					} else {
						log_operation($pdo, $currentUser['id'], $currentUser['username'], 'QTYPE_DELETE_FAIL', json_encode(array_merge($logDetails, ['error' => 'Delete failed or not found'])));
						http_response_code(404); echo json_encode(['message' => '刪除失敗：找不到指定的類型 ID 或刪除時發生錯誤']);
					}
				}
			} else {
				/* ... not found error ... */
				http_response_code(404);
				echo json_encode(['message' => '找不到指定的類型 ID']);
			}
		}

		break;

	default:
		http_response_code(405); echo json_encode(['message' => '方法不允許']); break;
	}
} catch (\PDOException $e) {
	// Log DB errors (Complete block)
	$logDetails = ['error' => $e->getMessage(), 'code' => $e->getCode(), 'method' => $method];
	// Add context if available
	if (isset($targetId)) $logDetails['id'] = $targetId;
	if (isset($typeName)) $logDetails['name'] = $typeName; // Use typeName from scope
	if (isset($data) && $method != 'GET' && $method != 'DELETE') $logDetails['data'] = $data; // Log input data for POST/PUT

	log_operation($pdo, $currentUser['id'], $currentUser['username'], 'QTYPE_DB_ERROR', json_encode($logDetails));

	http_response_code(500);
	if ($e->getCode() == 23000) { // Unique constraint violation (duplicate type name)
		echo json_encode(['message' => '錯誤：該類型名稱已經存在']);
	} else {
		error_log("PDOException in question_types.php: " . $e->getMessage()); // Log detailed error
		echo json_encode(['message' => '資料庫錯誤，請稍後再試或聯絡管理員']); // Generic message for user
	}
} catch (\Exception $e) {
	// Log general errors (Complete block)
	$logDetails = ['error' => $e->getMessage(), 'method' => $method];
	if (isset($targetId)) $logDetails['id'] = $targetId;
	if (isset($typeName)) $logDetails['name'] = $typeName;

	log_operation($pdo, $currentUser['id'], $currentUser['username'], 'QTYPE_ERROR', json_encode($logDetails));

	error_log("Exception in question_types.php: " . $e->getMessage()); // Log detailed error
	http_response_code(400); // Usually bad request for general exceptions here
	echo json_encode(['message' => '處理請求時發生錯誤']); // Generic message for user
}

