// 班級管理頁面的 JS 邏輯
(function() {
    // --- DOM 元素 ---
    const form = document.getElementById("class-form");
    const addEditModal = document.getElementById("add-edit-class-modal");
    const showAddBtn = document.getElementById("show-add-class-btn");
    const closeModalBtn = document.getElementById("close-class-modal");
    const formTitle = document.getElementById("form-title");
    const classIdInput = document.getElementById("class-id");
    const classNameInput = document.getElementById("class-name");
    const gradeSelect = document.getElementById("grade-level");
    const submitBtn = document.getElementById("submit-btn"); // Submit button for Add/Edit Class
    const cancelBtn = document.getElementById("cancel-btn");
    const formMessage = document.getElementById("form-message"); // Inside modal

    // Batch Create Modal & Form
    const showBatchClassBtn = document.getElementById("show-batch-class-btn");
    const batchClassModal = document.getElementById("batch-class-modal");
    const closeBatchClassModalBtn = document.getElementById("close-batch-class-modal");
    const batchClassForm = document.getElementById("batch-class-form");
    const batchClassFileInput = document.getElementById("class-file");
    const batchClassSubmitBtn = document.getElementById("batch-class-submit-btn"); // Submit button for Batch Class
    const batchClassMessage = document.getElementById("batch-class-message");
    const batchClassResultModal = document.getElementById("batch-class-result-modal");
    const closeBatchClassResultModalBtn = document.getElementById("close-batch-class-result-modal");
    const batchClassResultContent = document.getElementById("batch-class-result-content");

    // Export Button
    const exportClassesBtn = document.getElementById("export-classes-btn");

    // Table
    const tableBody = document.getElementById("class-table-body");

    // --- Element Existence Check ---
    if (!form || !addEditModal || !showAddBtn || !closeModalBtn || !tableBody ||
        !showBatchClassBtn || !batchClassModal || !closeBatchClassModalBtn || !batchClassForm ||
        !batchClassResultModal || !closeBatchClassResultModalBtn || !exportClassesBtn || !submitBtn || !batchClassSubmitBtn) { // Added batch submit btn check
        console.error("Class management: Missing required DOM elements.");
        return;
    }

    // --- Modal Functions ---
    function openModal(modalElement) { if (modalElement) modalElement.style.display = 'flex'; }
    function closeModal(modalElement) {
        if (modalElement) modalElement.style.display = 'none';
        if (modalElement === addEditModal) resetForm(); // Reset Add/Edit form on close
        // Reset Batch form when its modal is closed? Optional, maybe clear file input
        if (modalElement === batchClassModal) batchClassForm.reset();
    }

    // --- Core Functions ---
    // 顯示訊息
    function showMessage(text, isError = false, target = formMessage) {
        if (!target) return;
        target.textContent = text;
        target.className = isError ? 'form-message-area error' : 'form-message-area success';
         if (target === batchClassResultContent) { target.style.whiteSpace = 'pre-wrap'; }
    }
    // 載入所有班級 (表格)
    async function loadClasses() {
        const mainMsgTarget = document.querySelector('.widget > .form-message-area') || formMessage;
        showMessage('', false, mainMsgTarget);
        if (!tableBody) return;
        tableBody.innerHTML = '<tr><td colspan="4">載入中...</td></tr>';
        try {
            const response = await fetch('api/classes.php');
            if (!response.ok) { const errData = await response.json(); throw new Error(errData.message || '無法載入班級列表'); }
            const classes = await response.json();
            renderTable(classes);
        } catch (err) {
            console.error("Error loading classes:", err);
            tableBody.innerHTML = '<tr><td colspan="4" class="error">載入失敗，請檢查主控台。</td></tr>';
            showMessage('錯誤: ' + err.message, true, mainMsgTarget);
        }
    }
    // 渲染表格
    function renderTable(classes) {
        if (!tableBody) return; tableBody.innerHTML = '';
        if (!Array.isArray(classes) || classes.length === 0) { tableBody.innerHTML = '<tr><td colspan="4">尚無任何班級</td></tr>'; return; }
        classes.forEach(cls => {
            const tr = document.createElement('tr'); tr.dataset.class = JSON.stringify(cls);
            tr.innerHTML = `<td>${cls.id}</td><td>${escapeHTML(cls.class_name)}</td><td>${cls.grade_level || 'N/A'}</td><td class="actions"><button class="edit-btn">編輯</button><button class="delete-btn">刪除</button></td>`;
            tableBody.appendChild(tr);
        });
    }
    // HTML 實用工具
    function escapeHTML(str) { if (str === null || str === undefined) return ''; return str.toString().replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;').replace(/"/g, '&quot;').replace(/'/g, '&#39;'); }
    // 重置表單為 "新增" 狀態
    function resetForm() {
        form.reset(); classIdInput.value = ''; formTitle.textContent = '新增班級'; submitBtn.textContent = '儲存'; cancelBtn.style.display = 'none'; showMessage(''); submitBtn.disabled = false;
    }
    // 填充表單為 "編輯" 狀態
    function populateForm(cls) {
        resetForm(); formTitle.textContent = '編輯班級'; submitBtn.textContent = '更新'; cancelBtn.style.display = 'inline-block'; classIdInput.value = cls.id; classNameInput.value = cls.class_name; gradeSelect.value = cls.grade_level || ''; submitBtn.disabled = false;
    }
    // 處理表單提交 (新增 或 更新)
    async function handleFormSubmit(e) {
        e.preventDefault(); showMessage(''); submitBtn.disabled = true;
        const id = classIdInput.value; const isUpdate = !!id;
        const data = { class_name: classNameInput.value, grade_level: gradeSelect.value || null };
        let url = 'api/classes.php'; let method = 'POST';
        if (isUpdate) { data.id = id; method = 'PUT'; }
        try {
            const response = await fetch(url, { method: method, headers: { 'Content-Type': 'application/json' }, body: JSON.stringify(data) });
            const result = await response.json(); if (!response.ok) throw new Error(result.message || (isUpdate ? '更新失敗' : '新增失敗'));
            showMessage(isUpdate ? '更新成功' : '新增成功', false);
            setTimeout(() => { closeModal(addEditModal); loadClasses(); }, 1000);
        } catch (err) { console.error("Error submitting class form:", err); showMessage(err.message, true); submitBtn.disabled = false; }
    }
    // 處理刪除 (使用 alert)
    async function handleDelete(buttonElement) {
        const tr = buttonElement.closest('tr'); if (!tr || !tr.dataset.class) return;
        const cls = JSON.parse(tr.dataset.class);
        if (!confirm(`確定要刪除班級 "${escapeHTML(cls.class_name)}" 嗎？\n(注意：如果有使用者屬於此班級，將無法刪除)`)) { return; }
        buttonElement.disabled = true; let finalErrorMessage = '刪除失敗 (未知錯誤)';
        try {
            const response = await fetch(`api/classes.php?id=${cls.id}`, { method: 'DELETE' });
            let responseBody = null; let responseOk = response.ok;
            try { responseBody = await response.json(); } catch (e) { try { responseBody = await response.text(); } catch (e2) { responseBody = `伺服器錯誤 ${response.status}`; } }
            if (!responseOk) {
                 let apiMessage = (typeof responseBody === 'string') ? responseBody : (responseBody?.message || finalErrorMessage);
                 if (apiMessage && (apiMessage.includes('尚有使用者屬於此班級') || apiMessage.includes('in use'))) { finalErrorMessage = `無法刪除：班級 ${escapeHTML(cls.class_name)} 使用中。`; }
                 else { finalErrorMessage = apiMessage; }
                 throw new Error(finalErrorMessage);
            }
            alert(`班級 "${escapeHTML(cls.class_name)}" 刪除成功`); tr.remove();
            await loadClasses(); // Reload list on success
        } catch (err) {
            console.error("Error deleting class:", err); alert('刪除失敗：' + err.message);
            buttonElement.disabled = false; // Re-enable button on error
            // Reload list on error as well to ensure UI consistency
            await loadClasses();
        }
    }
    // 處理表格中的點擊 (事件委派)
    function handleTableClick(e) {
        const editButton = e.target.closest('.edit-btn');
        const deleteButton = e.target.closest('.delete-btn');
        if (editButton) { const tr = editButton.closest('tr'); if (tr && tr.dataset.class) { const cls = JSON.parse(tr.dataset.class); populateForm(cls); openModal(addEditModal); } }
        else if (deleteButton) { handleDelete(deleteButton); } // Pass the button
    }

    // (新) Handle batch class upload (Restored full function)
    async function handleBatchClassUpload(e) {
        e.preventDefault();
        const file = batchClassFileInput.files[0];
        if (!file) {
            showMessage('請選擇要上傳的 CSV 檔案', true, batchClassMessage);
            return;
        }

        const formData = new FormData();
        formData.append('class_file', file);

        showMessage('上傳中...', false, batchClassMessage);
        batchClassSubmitBtn.disabled = true;
        let resultData = {};

        try {
            // Call API with action parameter
            const response = await fetch('api/classes.php?action=batch_create', {
                method: 'POST',
                body: formData
            });
            resultData = await response.json();

            // Check response status, allow 200 even if there are errors in data
            if (!response.ok && response.status !== 200) {
                 if (resultData.message) throw new Error(resultData.message);
                 else throw new Error(`上傳失敗 (HTTP ${response.status})`);
            }

            batchClassForm.reset(); // Clear file input
            closeModal(batchClassModal); // Close upload modal
            loadClasses(); // Reload table

            // Prepare result message
            let msg = `成功新增 ${resultData.createdCount || 0} 個班級。`;
            let isErrorResult = false;
            if (resultData.errors && resultData.errors.length > 0) {
                 msg += `\n發生 ${resultData.errors.length} 個錯誤：\n` + resultData.errors.join('\n');
                 isErrorResult = true;
            }

            // Show result modal
            showMessage(msg, isErrorResult, batchClassResultContent);
            openModal(batchClassResultModal);

        } catch (err) {
            console.error("Error batch creating classes:", err);
            showMessage('錯誤: ' + err.message, true, batchClassMessage);
            // Keep upload modal open on error
        } finally {
            batchClassSubmitBtn.disabled = false;
        }
    }


    // --- Event Listeners ---
    form.addEventListener('submit', handleFormSubmit);
    tableBody.addEventListener('click', handleTableClick);
    showAddBtn.addEventListener('click', () => { resetForm(); openModal(addEditModal); });
    closeModalBtn.addEventListener('click', () => closeModal(addEditModal));
    cancelBtn.addEventListener('click', () => closeModal(addEditModal));
    addEditModal.addEventListener('click', (e) => { if (e.target === addEditModal) closeModal(addEditModal); });

    // Batch Create Modals & Form
    showBatchClassBtn.addEventListener('click', () => { showMessage('', false, batchClassMessage); batchClassForm.reset(); batchClassSubmitBtn.disabled = false; openModal(batchClassModal); });
    closeBatchClassModalBtn.addEventListener('click', () => closeModal(batchClassModal));
    batchClassModal.addEventListener('click', (e) => { if (e.target === batchClassModal) closeModal(batchClassModal); });
    batchClassForm.addEventListener('submit', handleBatchClassUpload); // Attach batch submit listener
    closeBatchClassResultModalBtn.addEventListener('click', () => { closeModal(batchClassResultModal); closeModal(batchClassModal); });
    batchClassResultModal.addEventListener('click', (e) => { if (e.target === batchClassResultModal) { closeModal(batchClassResultModal); closeModal(batchClassModal); } });

    // Export Button
    exportClassesBtn.addEventListener('click', () => { window.location.href = 'api/classes.php?action=export'; });

    // --- Initial Load ---
    async function init() { await loadClasses(); }
    init();

})();
