// 中文練習頁面的 JS 邏輯（配合 dashboard/typing-content.html）
// 功能：計時 / 限時練習 + 練習成績記錄到 api/practice_log.php
(function () {
    // --- 1. 抓 DOM 元件 ---
    const modeMeasuredRadio   = document.getElementById('mode-measured');
    const modeTimedRadio      = document.getElementById('mode-timed');
    const levelSelect         = document.getElementById('level-select');
    const measuredSettings    = document.getElementById('measured-settings');
    const timedSettings       = document.getElementById('timed-settings');
    const charCountSelect     = document.getElementById('char-count-select');
    const timeDurationSelect  = document.getElementById('time-duration-select');
    const fontNoZhuyinRadio   = document.getElementById('font-no-zhuyin');
    const fontZhuyinRadio     = document.getElementById('font-zhuyin');
    const stopTestBtn         = document.getElementById('stop-test-btn');

    const timerDisplay        = document.getElementById('timer-display');
    const charsTypedDisplay   = document.getElementById('chars-typed-display');
    const charsTargetDisplay  = document.getElementById('chars-target-display');
    const errorsDisplay       = document.getElementById('errors-display');
    const speedDisplay        = document.getElementById('speed-display');

    const statusMessage       = document.getElementById('status-message');
    const textDisplayWindow   = document.getElementById('text-display-window');
    const typingDisplay       = document.getElementById('typing-display');
    const typingInput         = document.getElementById('typing-input');

    const container           = document.querySelector('.typing-practice-container');

    if (!modeMeasuredRadio || !modeTimedRadio || !levelSelect || !charCountSelect ||
        !timeDurationSelect || !timerDisplay || !charsTypedDisplay || !errorsDisplay ||
        !speedDisplay || !statusMessage || !textDisplayWindow || !typingDisplay ||
        !typingInput || !stopTestBtn || !container) {

        console.error('Typing JS: 必要元素遺失，無法啟動練習頁面。');
        if (container) {
            container.innerHTML = '<h2 class="error">練習頁面載入錯誤，缺少必要元素。</h2>';
        } else {
            alert('練習頁面載入錯誤...');
        }
        return;
    }

    // --- 2. 狀態變數 ---
    let practiceMode    = 'measured'; // 'measured' 計字數 / 'timed' 限時
    let targetLevel     = 3;          // 年級
    let targetCount     = 50;         // 目標字數（計時模式，依「實際輸入字數」）
    let targetTime      = 60;         // 目標秒數（限時模式）

    let useZhuyinFont   = false;

    let questionLines   = []; // 題目逐行文字
    let currentLineIndex = 0; // 題目目前所在行（配合 text-display-window 的 highlight）

    // 使用者輸入：
    // typedLines：按 Enter 完成的一行一行
    // typingInput.value：目前正在打的這一行
    let typedLines      = [];

    // 成績統計：**統一以 typing-display 的文字為準**
    let totalCharsTyped   = 0; // 總輸入字數（全部行＋目前這一行）
    let totalErrors       = 0; // 總錯誤字數
    let totalCorrectChars = 0; // 總正確字數

    let startTime       = null; // ms timestamp
    let timeElapsed     = 0;    // 已經過秒數
    let timerInterval   = null;

    let practiceActive     = false; // 是否正在練習
    let practicePrepared   = false; // 是否已設定好，等待點擊輸入框開始
    let practiceFinished   = false; // 是否已結束本輪練習（只用來防 double-finish）
    let isLoadingQuestions = false;
    let isComposing        = false; // IME 組字中
    let finishReason       = 'unknown';

    const EDIT_KEYS = [
        'Backspace', 'Delete', 'ArrowLeft', 'ArrowRight', 'Home', 'End'
    ];
    const IGNORE_KEYS = [
        'Shift', 'Control', 'Alt', 'Meta',
        'CapsLock', 'NumLock', 'ScrollLock',
        'Tab', 'ContextMenu',
        'ArrowUp', 'ArrowDown',
        'PageUp', 'PageDown', 'Insert',
        'Escape', 'F1', 'F2', 'F3', 'F4', 'F5', 'F6', 'F7', 'F8', 'F9', 'F10', 'F11', 'F12'
    ];

    // --- 3. 工具函式 ---
    function showStatusMessage(text, isError = false) {
        statusMessage.textContent = text;
        statusMessage.classList.toggle('error', !!isError);
    }

    function getUserFullText() {
        // 把「已完成的行」＋「目前輸入中的行」組成 typing-display 的內容
        const currentLine = typingInput.value || '';
        if (!typedLines.length) return currentLine;

        if (currentLine) {
            return typedLines.join('\n') + '\n' + currentLine;
        }
        return typedLines.join('\n');
    }

    function syncTypingDisplay() {
        const fullText = getUserFullText();
        typingDisplay.textContent = fullText;
        typingDisplay.scrollTop   = typingDisplay.scrollHeight;
    }

    function resetScores() {
        timeElapsed        = 0;
        currentLineIndex   = 0;
        questionLines      = [];
        typedLines         = [];

        totalCharsTyped    = 0;
        totalErrors        = 0;
        totalCorrectChars  = 0;

        timerDisplay.textContent      = '00:00';
        charsTypedDisplay.textContent = '0';
        charsTargetDisplay.textContent = '';
        errorsDisplay.textContent     = '0';
        speedDisplay.textContent      = '0.0';

        textDisplayWindow.innerHTML = '';
        typingDisplay.textContent   = '';
        typingInput.value           = '';

        showStatusMessage('請先設定練習模式、年級與字數/時間。', false);
    }

    function updateTimerDisplay() {
        let displayTime = timeElapsed;
        if (practiceMode === 'timed') {
            const remain = Math.max(0, targetTime - timeElapsed);
            displayTime = remain;
        }

        const minutes = Math.floor(displayTime / 60);
        const seconds = displayTime % 60;
        const prefix  = (practiceMode === 'timed') ? '-' : '';

        timerDisplay.textContent =
            prefix +
            String(minutes).padStart(2, '0') + ':' +
            String(seconds).padStart(2, '0');
    }

    function updateSpeedDisplay() {
        const minutes = timeElapsed > 0 ? timeElapsed / 60 : 0;
        let wpm = 0;
        if (minutes > 0) {
            wpm = totalCorrectChars / minutes;
        }
        speedDisplay.textContent = wpm.toFixed(1);
    }

    // **所有成績統計都從 typing-display / getUserFullText() 來**
    function updateStatsFromUserText() {
        const userText = getUserFullText();
        const total    = userText.length;
        const expectedText = questionLines.join('\n');

        let correct = 0;
        let errors  = 0;

        for (let i = 0; i < total; i++) {
            const u = userText[i];
            const e = expectedText[i];

            if (e === undefined || e === null) {
                // 題目沒有這個位置的字，多打算錯
                errors++;
            } else if (u !== e) {
                errors++;
            } else {
                correct++;
            }
        }

        totalCharsTyped   = total;
        totalCorrectChars = correct;
        totalErrors       = errors;

        charsTypedDisplay.textContent = String(totalCharsTyped);
        errorsDisplay.textContent     = String(totalErrors);
        updateSpeedDisplay();

        // 計字數模式自動結束：以「實際輸入字數」判斷
        if (!practiceFinished && practiceActive &&
            practiceMode === 'measured' &&
            totalCharsTyped >= targetCount) {

            finishPractice('auto-target');
        }
    }

    function renderTextDisplay() {
        textDisplayWindow.innerHTML = '';

        if (!questionLines.length) {
            const span = document.createElement('span');
            span.className = 'upcoming-text';
            span.textContent = '尚未載入題目，請選擇模式並開始練習。';
            textDisplayWindow.appendChild(span);
            return;
        }

        const contextLines = 5;
        const start = Math.max(0, currentLineIndex - 2);
        const end   = Math.min(questionLines.length, currentLineIndex + contextLines);

        for (let i = start; i < end; i++) {
            const lineDiv = document.createElement('div');
            lineDiv.classList.add('line');

            const text = questionLines[i] || '';
            lineDiv.textContent = text;

            if (i < currentLineIndex) {
                lineDiv.classList.add('typed-correct');
            } else if (i === currentLineIndex) {
                lineDiv.classList.add('current');
            } else {
                lineDiv.classList.add('upcoming');
            }

            textDisplayWindow.appendChild(lineDiv);
        }

        const currentLineDiv = textDisplayWindow.querySelector('.line.current');
        if (currentLineDiv) {
            currentLineDiv.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
        }
    }

    function updateFont() {
        useZhuyinFont = fontZhuyinRadio.checked;
        if (useZhuyinFont) {
            textDisplayWindow.classList.add('font-zhuyin');
            textDisplayWindow.classList.remove('font-no-zhuyin');
        } else {
            textDisplayWindow.classList.add('font-no-zhuyin');
            textDisplayWindow.classList.remove('font-zhuyin');
        }
    }

    // --- 4-1. 啟用條件檢查 & 控制 typing-input ---
    function isSettingsValid() {
        const selectedMode = document.querySelector('input[name="practice-mode"]:checked');
        if (!selectedMode) return false;

        const mode  = selectedMode.value;
        const level = levelSelect.value;

        if (!level || level === '0') return false;

        if (mode === 'measured') {
            const cnt = charCountSelect.value;
            if (!cnt) return false;
            const n = parseInt(cnt, 10) || 0;
            if (n <= 0) return false;
        } else if (mode === 'timed') {
            const t = timeDurationSelect.value;
            if (!t) return false;
            const n = parseInt(t, 10) || 0;
            if (n <= 0) return false;
        } else {
            return false;
        }

        return true;
    }

    function updateTypingInputAvailability(showMessage) {
        // 如果正在載題目或練習中，不要在這裡碰 typingInput 的 enable 狀態
        if (practiceActive || isLoadingQuestions) {
            return;
        }

        if (isSettingsValid()) {
            typingInput.disabled  = false;
            practicePrepared      = true;
            if (showMessage) {
                showStatusMessage('設定完成，請點擊輸入框開始練習。', false);
            }
        } else {
            typingInput.disabled  = true;
            practicePrepared      = false;
            if (showMessage) {
                showStatusMessage('請先設定練習模式、年級與字數/時間，輸入框才會啟用。', true);
            }
        }
    }

    // --- 4-2. 設定與 UI 狀態 ---
    function readSettings() {
        const selectedMode = document.querySelector('input[name="practice-mode"]:checked');
        practiceMode = selectedMode ? selectedMode.value : 'measured';

        targetLevel = parseInt(levelSelect.value || '0', 10) || 0;
        targetCount = parseInt(charCountSelect.value || '0', 10) || 0;
        targetTime  = parseInt(timeDurationSelect.value || '0', 10) || 0;

        if (practiceMode === 'measured' && targetCount > 0) {
            charsTargetDisplay.style.display = 'inline';
            charsTargetDisplay.textContent = ' / ' + targetCount;
        } else {
            charsTargetDisplay.style.display = 'none';
            charsTargetDisplay.textContent = '';
        }

        updateSettingsUI();
    }

    function updateSettingsUI() {
        if (practiceMode === 'measured') {
            measuredSettings.style.display = '';
            timedSettings.style.display    = 'none';
        } else {
            measuredSettings.style.display = 'none';
            timedSettings.style.display    = '';
        }
    }

    function setIdleState(finished = false) {
        practiceActive     = false;
        isLoadingQuestions = false;

        if (timerInterval) {
            clearInterval(timerInterval);
            timerInterval = null;
        }

        if (!finished) {
            resetScores();
        } else {
            const finalWpm = speedDisplay.textContent;
            showStatusMessage(
                `練習完成！速度: ${finalWpm} 字/分（共 ${totalCharsTyped} 字，錯 ${totalErrors} 字）。`,
                false
            );
        }

        stopTestBtn.disabled        = true;
        modeMeasuredRadio.disabled  = false;
        modeTimedRadio.disabled     = false;
        levelSelect.disabled        = false;
        charCountSelect.disabled    = false;
        timeDurationSelect.disabled = false;
        fontNoZhuyinRadio.disabled  = false;
        fontZhuyinRadio.disabled    = false;

        // 清掉目前輸入內容與畫面
        typingInput.value = '';
        syncTypingDisplay();

        // 結束後主動 blur，一定會在下一次 click 重新觸發 focusin → startPracticeIfPrepared
        typingInput.blur();

        // 根據目前設定狀態，決定輸入框是否可用
        readSettings();
        updateTypingInputAvailability(false);

        practiceFinished = finished;
    }

    function preparePractice() {
        readSettings();
        resetScores();
        practiceFinished = false;
        // 依設定決定是否啟用輸入框
        updateTypingInputAvailability(true);
    }

    // --- 5. 題目載入 & 計時 ---
    async function fetchQuestions(minLength) {
        showStatusMessage('載入題目中...', false);

        const url = `api/get_questions.php?level=${encodeURIComponent(targetLevel)}&targetLength=${encodeURIComponent(minLength)}`;
        const res = await fetch(url);
        if (!res.ok) {
            throw new Error(`API 錯誤 (${res.status})`);
        }

        const lines = await res.json();
        if (!Array.isArray(lines)) {
            throw new Error('API 回應格式錯誤（預期為陣列）');
        }
        if (!lines.length) {
            throw new Error('題庫沒有符合條件的題目。');
        }

        questionLines    = lines;
        currentLineIndex = 0;
        renderTextDisplay();
    }

    function startTimer() {
        if (timerInterval || !practiceActive) return;

        timerInterval = setInterval(() => {
            timeElapsed += 1;
            updateTimerDisplay();
            updateSpeedDisplay();

            if (practiceMode === 'timed' && timeElapsed >= targetTime && !practiceFinished) {
                // 時間結束前，先把目前內容統計一次
                updateStatsFromUserText();
                finishPractice('auto-timeout');
            }
        }, 1000);
    }

    // --- 6. 練習流程 ---
    async function startPracticeIfPrepared() {
        // 僅在「使用者點到輸入框」且已完成設定時才開始
        if (!practicePrepared || practiceActive || isLoadingQuestions) return;

        practiceActive       = false;
        practiceFinished     = false;
        isLoadingQuestions   = true;
        finishReason         = 'unknown';

        showStatusMessage('載入題目中...', false);
        textDisplayWindow.innerHTML = '<div class="line upcoming">載入題目中...</div>';

        stopTestBtn.disabled        = true;
        modeMeasuredRadio.disabled  = true;
        modeTimedRadio.disabled     = true;
        levelSelect.disabled        = true;
        charCountSelect.disabled    = true;
        timeDurationSelect.disabled = true;
        fontNoZhuyinRadio.disabled  = true;
        fontZhuyinRadio.disabled    = true;
        typingInput.disabled        = true;

        try {
            const minLength = (practiceMode === 'measured')
                ? (targetCount + 50)
                : Math.max(500, Math.floor(targetTime / 60 * 200) + 50);

            await fetchQuestions(minLength);

            if (!questionLines.length) {
                throw new Error('無法載入題目。');
            }

            // 題目載入成功，開始計時與輸入
            startTime        = Date.now();
            timeElapsed      = 0;
            practiceActive   = true;
            practicePrepared = false;
            practiceFinished = false;
            typedLines       = [];

            totalCharsTyped   = 0;
            totalErrors       = 0;
            totalCorrectChars = 0;

            charsTypedDisplay.textContent = '0';
            errorsDisplay.textContent     = '0';
            speedDisplay.textContent      = '0.0';
            updateTimerDisplay();
            showStatusMessage('已開始練習，請依照上方題目輸入。', false);

            typingInput.disabled = false;
            typingInput.value    = '';
            syncTypingDisplay();
            typingInput.focus();

            stopTestBtn.disabled = false;
            startTimer();
        } catch (e) {
            console.error('Typing JS: 載入題目失敗', e);
            showStatusMessage('載入題目失敗：' + e.message, true);
            setIdleState(false);
        } finally {
            isLoadingQuestions = false;
        }
    }

    function handleKeyDown(event) {
        // ★ 這裡只看 practiceActive / isLoadingQuestions
        if (!practiceActive || isLoadingQuestions) return;

        if (IGNORE_KEYS.includes(event.key)) {
            return;
        }

        // 方向鍵 / 刪除鍵：讓瀏覽器先修改 input.value，再同步到 display＋統計
        if (EDIT_KEYS.includes(event.key)) {
            setTimeout(() => {
                if (!isComposing && practiceActive) {
                    syncTypingDisplay();
                    updateStatsFromUserText();
                }
            }, 0);
        }

        if (event.key === 'Enter') {
            event.preventDefault();

            if (isComposing) {
                // 理論上 composition 狀態下不應該送 Enter 做換行
                return;
            }

            const currentLine  = typingInput.value || '';
            const expectedLine = questionLines[currentLineIndex] || '';

            // 將這一行視為「已完成的一行」，加入 typedLines
            typedLines.push(currentLine);
            typingInput.value = '';

            syncTypingDisplay();
            updateStatsFromUserText();

            // 題目行向下移一行（用於 highlight）
            if (expectedLine !== undefined) {
                currentLineIndex++;
                renderTextDisplay();
            }

            // 若題目已經沒行了：計字數模式直接結束；限時模式則只禁止繼續輸入
            if (currentLineIndex >= questionLines.length) {
                if (practiceMode === 'measured') {
                    finishPractice('auto-no-more-questions');
                } else {
                    typingInput.disabled = true;
                    showStatusMessage('題目已全部輸入完畢，請等待時間結束。', false);
                }
            }
        }
    }

    function handleInput(event) {
        // ★ 同樣只看 practiceActive / isLoadingQuestions
        if (!practiceActive || isLoadingQuestions) return;
        if (isComposing || event.isComposing) return;

        // 一般輸入（含 IME 已完成的字）→ 同步 display＋統計
        syncTypingDisplay();
        updateStatsFromUserText();
    }

    // --- 7. 練習結束 & 成績寫入 ---
    async function savePracticeResult(reason, stats) {
        try {
            const payload = {
                practice_mode:  practiceMode,
                target_chars:   practiceMode === 'measured' ? targetCount : null,
                target_seconds: practiceMode === 'timed'    ? targetTime  : null,
                start_time:     stats.startEpoch,
                end_time:       stats.endEpoch,
                duration_seconds: stats.durationSeconds,
                question_text:  questionLines.join('\n'),
                input_text:     getUserFullText(),   // ★ 直接用 typing-display 對應的內容
                total_chars:    stats.totalChars,
                error_chars:    stats.errorChars,
                correct_chars:  stats.correctChars,
                wpm:            stats.wpm,
                finish_reason:  reason
            };

            const res = await fetch('api/practice_log.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(payload)
            });

            if (!res.ok) {
                const text = await res.text().catch(() => '');
                throw new Error(`API ${res.status}：${text}`);
            }

            const data = await res.json();
            if (!data.result) {
                console.warn('Typing JS: 成績記錄 API 回應失敗', data);
                showStatusMessage('練習結束，但成績記錄失敗：' + (data.message || ''), true);
            } else {
                console.log('Typing JS: 練習成績已記錄，log_id=', data.log_id);
            }
        } catch (e) {
            console.error('Typing JS: 成績記錄時發生錯誤', e);
            showStatusMessage('練習結束，但成績記錄時發生錯誤：' + e.message, true);
        }
    }

    function finishPractice(reason = 'unknown') {
        if (practiceFinished) return;

        practiceFinished = true;
        finishReason     = reason;

        practiceActive = false;

        if (timerInterval) {
            clearInterval(timerInterval);
            timerInterval = null;
        }

        // 最後一刻再同步一次，確保 typing-display / 成績 是最新的
        syncTypingDisplay();
        updateStatsFromUserText();

        const nowEpoch   = Math.floor(Date.now() / 1000);
        const startEpoch = startTime ? Math.floor(startTime / 1000) : nowEpoch;
        const duration   = Math.max(timeElapsed, nowEpoch - startEpoch);

        const finalChars   = totalCharsTyped;
        const finalErrors  = totalErrors;
        const correctChars = totalCorrectChars;
        const minutes      = duration > 0 ? duration / 60 : 0;
        const finalWpm     = minutes > 0 ? (correctChars / minutes) : 0;

        charsTypedDisplay.textContent = String(finalChars);
        errorsDisplay.textContent     = String(finalErrors);
        speedDisplay.textContent      = finalWpm.toFixed(1);

        const stats = {
            startEpoch:      startEpoch,
            endEpoch:        nowEpoch,
            durationSeconds: duration,
            totalChars:      finalChars,
            errorChars:      finalErrors,
            correctChars:    correctChars,
            wpm:             parseFloat(finalWpm.toFixed(1))
        };

        savePracticeResult(reason, stats);
        setIdleState(true);

        console.log('Typing JS: 練習結束', { reason, stats });
    }

    // --- 8. 初始化 ---
    function init() {
        // 若 dashboard 有傳年級，則帶入；否則看 select 預設值
        if (typeof window.currentUserGrade === 'number' &&
            window.currentUserGrade >= 3 && window.currentUserGrade <= 6 &&
            Array.from(levelSelect.options).some(o => o.value === String(window.currentUserGrade))) {

            levelSelect.value = String(window.currentUserGrade);
        }

        readSettings();
        updateFont();
        resetScores();

        // 初始狀態：依照目前下拉選單 / 單選按鈕是否有值，決定輸入框可不可點
        updateTypingInputAvailability(false);
        practiceFinished = false;

        modeMeasuredRadio.addEventListener('change', preparePractice);
        modeTimedRadio.addEventListener('change', preparePractice);
        levelSelect.addEventListener('change', preparePractice);
        charCountSelect.addEventListener('change', preparePractice);
        timeDurationSelect.addEventListener('change', preparePractice);

        fontNoZhuyinRadio.addEventListener('change', updateFont);
        fontZhuyinRadio.addEventListener('change', updateFont);

        typingInput.addEventListener('focusin', startPracticeIfPrepared);
        typingInput.addEventListener('keydown', handleKeyDown);
        typingInput.addEventListener('input', handleInput);

        stopTestBtn.addEventListener('click', () => {
            if (practiceActive) {
                finishPractice('manual-stop');
            }
        });

        typingInput.addEventListener('compositionstart', () => {
            isComposing = true;
        });

        typingInput.addEventListener('compositionend', (event) => {
            isComposing = false;
            if (!practiceActive || isLoadingQuestions) return;
            // IME 選字完成後，同步 display＋統計
            syncTypingDisplay();
            updateStatsFromUserText();
        });

        // 禁用 copy / paste / context menu
        typingInput.addEventListener('paste', e => e.preventDefault());
        typingInput.addEventListener('copy', e => e.preventDefault());
        typingInput.addEventListener('cut', e => e.preventDefault());
        typingInput.addEventListener('contextmenu', e => e.preventDefault());
    }

    init();
})();

