<?php
// api/auth.php
// 共用：安全標頭、Session 設定、登入/權限檢查、CSRF 產生/驗證。

declare(strict_types=1);

if (session_status() !== PHP_SESSION_ACTIVE) {
  $cookieParams = session_get_cookie_params();
  session_set_cookie_params([
    'lifetime' => 0,
    'path'     => $cookieParams['path'] ?? '/',
    'domain'   => $cookieParams['domain'] ?? '',
    'secure'   => isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on',
    'httponly' => true,
    'samesite' => 'Lax',
  ]);
  session_start();
}

header('X-Frame-Options: SAMEORIGIN');
header('X-Content-Type-Options: nosniff');
header('Referrer-Policy: strict-origin-when-cross-origin');

function auth_current_user(): ?array {
  return isset($_SESSION['user']) && is_array($_SESSION['user']) ? $_SESSION['user'] : null;
}
function auth_is_logged_in(): bool { return auth_current_user() !== null; }

function auth_has_role(string $role): bool {
  $u = auth_current_user();
  if (!$u) return false;

  if (isset($u['role']) && is_string($u['role'])) {
    $r = strtolower($u['role']);
    if ($role === 'admin')   return $r === 'admin';
    if ($role === 'teacher') return $r === 'teacher' || $r === 'admin';
    if ($role === 'student') return $r === 'student';
  }
  if (isset($u['role']) && is_numeric($u['role'])) {
    $val = intval($u['role']);
    if ($role === 'admin')   return $val === 0xff || $val === 255;
    if ($role === 'teacher') return ($val & 0x01) || $val === 255;
    if ($role === 'student') return $val === 0x00 || $val === 0;
  }
  return false;
}

function auth_require_login(): ?array {
  if (!auth_is_logged_in()) return [401, ['success'=>false,'message'=>'未登入']];
  return null;
}
function auth_require_role(array $roles): ?array {
  $login = auth_require_login(); if ($login) return $login;
  foreach ($roles as $r) if (auth_has_role($r)) return null;
  return [403, ['success'=>false,'message'=>'權限不足']];
}

function csrf_get_token(): string {
  if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
  return $_SESSION['csrf'];
}
function csrf_verify_from_request(?array $jsonBody = null): ?array {
  $token = null;
  $headers = function_exists('getallheaders') ? getallheaders() : [];
  if ($headers && isset($headers['X-CSRF-Token'])) $token = $headers['X-CSRF-Token'];
  elseif ($headers && isset($headers['x-csrf-token'])) $token = $headers['x-csrf-token'];
  if (!$token && $jsonBody && isset($jsonBody['csrf'])) $token = $jsonBody['csrf'];
  if (!$token || !hash_equals(csrf_get_token(), (string)$token)) {
    return [419, ['success'=>false,'message'=>'CSRF 驗證失敗']];
  }
  return null;
}

