<?php
session_start();
include 'db_connect.php';
include 'log_helper.php'; // [cite: 5]

header('Content-Type: application/json');

// --- 1. 權限檢查 ---
if (!isset($_SESSION['user'])) {
    http_response_code(401);
    echo json_encode(['message' => '未授權，請先登入']);
    exit;
}
$currentUser = $_SESSION['user'];

// 僅管理者或教師可批次上傳
if (!in_array($currentUser['role'], ['admin', 'teacher'])) {
    http_response_code(403);
    echo json_encode(['message' => '權限不足']);
    exit;
}

// --- 2. 檔案上傳檢查 ---
if (!isset($_FILES['account_file']) || $_FILES['account_file']['error'] != UPLOAD_ERR_OK) {
    http_response_code(400);
    echo json_encode(['message' => '檔案上傳失敗']);
    exit;
}

$fileTmpPath = $_FILES['account_file']['tmp_name'];
$fileName = $_FILES['account_file']['name'];
$fileExtension = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));

$pdo = get_db_connection();
$createdCount = 0;
$errors = [];

// (新) 取得教師可管理的班級
$teacherClasses = [];
if ($currentUser['role'] == 'teacher') {
    $stmt = $pdo->prepare("SELECT class_name FROM classes WHERE JSON_CONTAINS(teacher_ids, CAST(? AS JSON))");
    $stmt->execute([$currentUser['id']]);
    $teacherClasses = $stmt->fetchAll(PDO::FETCH_COLUMN); // [cite: 4]
}

try {
    // --- 3. 根據檔案類型處理 ---
    if ($fileExtension == 'csv') {
        if (($handle = fopen($fileTmpPath, 'r')) !== FALSE) {
            $pdo->beginTransaction();
            
            $header = fgetcsv($handle);
            if (!$header) {
                throw new Exception("CSV 檔案為空或格式錯誤");
            }
            
            $line = 1;
            while (($data = fgetcsv($handle)) !== FALSE) {
                $line++;
                if (count($header) != count($data)) {
                    $errors[] = "第 $line 行: 欄位數量不符";
                    continue;
                }
                $row = array_combine($header, $data); 

                $username = $row['username'] ?? null;
                $name = $row['name'] ?? null;
                $role = $row['role'] ?? null;
                $class_str = $row['class'] ?? null;
                $grade = $row['grade'] ?? null;

                // --- 4. 權限與資料驗證 (同 users.php POST) ---
                if (empty($username) || empty($name) || empty($role)) {
                    $errors[] = "第 $line 行: 帳號、姓名、角色為必填";
                    continue;
                }
                
                // 檢查教師權限
                if ($currentUser['role'] == 'teacher') {
                    if ($role != 'student') {
                        $errors[] = "第 $line 行: 教師僅能建立學生帳號 [cite: 4]";
                        continue;
                    }
                    if ($class_str && !in_array($class_str, $teacherClasses)) {
                         $errors[] = "第 $line 行: 教師僅能將學生加入自己所屬的班級 [cite: 4]";
                         continue;
                    }
                }
                
                // 檢查管理者權限
                if ($currentUser['role'] == 'admin') {
                    if ($role == 'admin' && $currentUser['username'] != 'typing') {
                        $errors[] = "第 $line 行: 僅 'typing' 才能建立管理者";
                        continue;
                    }
                    if ($role == 'teacher') { /* Admin 可建立 */ }
                }

                // 處理班級 JSON
                $class_json = null;
                if ($role == 'student' && !empty($class_str)) {
                    $class_json = json_encode([$class_str]); // 存為 ["C101"]
                }
                // (教師的 class 欄位應為 NULL)

                // --- 5. 寫入資料庫 ---
                $sql = "INSERT INTO users (username, name, role, class, grade, is_active) VALUES (?, ?, ?, ?, ?, ?)";
                $stmt = $pdo->prepare($sql);
                $stmt->execute([
                    $username,
                    $name,
                    $role,
                    $class_json,
                    ($role == 'student' ? $grade : null),
                    1 // 批次建立預設為啟用
                ]);
                $createdCount++;
            }
            fclose($handle);
            $pdo->commit();
        }

    } else if ($fileExtension == 'xlsx') {
        $errors[] = "XLSX 格式目前尚未支援。請安裝 PhpSpreadsheet 函式庫。";
        
    } else {
        $errors[] = "不支援的檔案格式，僅接受 .csv 或 .xlsx";
    }

    if ($createdCount > 0) {
        // 寫入 Log [cite: 5]
        log_operation($pdo, $currentUser['id'], $currentUser['username'], 'BATCH_CREATE_USERS', json_encode([
            'created' => $createdCount,
            'errors' => count($errors)
        ]));
    }

    echo json_encode([
        'success' => true,
        'createdCount' => $createdCount,
        'errors' => $errors
    ]);

} catch (\PDOException $e) {
    $pdo->rollBack();
    if ($e->getCode() == 23000) { 
        $errors[] = '資料庫錯誤：帳號已存在。 (操作已中斷)';
    } else {
        $errors[] = '資料庫錯誤: ' . $e->getMessage();
    }
    echo json_encode(['success' => false, 'createdCount' => $createdCount, 'errors' => $errors]);
} catch (\Exception $e) {
     if ($pdo->inTransaction()) {
        $pdo->rollBack();
     }
     $errors[] = '處理失敗: ' . $e->getMessage();
     echo json_encode(['success' => false, 'createdCount' => $createdCount, 'errors' => $errors]);
}
?>
