<?php
session_start();
include 'db_connect.php';
include 'log_helper.php';

header('Content-Type: application/json; charset=utf-8');

// 必須先登入
if (!isset($_SESSION['user'])) {
    http_response_code(401);
    echo json_encode(['message' => '未授權，請先登入']);
    // 不使用 exit，直接 return
    return;
}

$currentUser = $_SESSION['user'];
$method = $_SERVER['REQUEST_METHOD'];
$action = $_REQUEST['action'] ?? null;
$pdo = get_db_connection();

// 教師可管理的班級
$teacherClassIDs = [];
if ($currentUser['role'] === 'teacher') {
    $cls = $currentUser['class'] ?? '[]';
    if (is_string($cls)) {
        $teacherClassIDs = json_decode($cls, true) ?: [];
    } elseif (is_array($cls)) {
        $teacherClassIDs = $cls;
    }
}

// 取得登入者可見列表
function get_visible_users(PDO $pdo, array $currentUser, array $teacherClassIDs): array {
    $stmt = $pdo->query("SELECT id, username, name, class, role, grade, is_active FROM users");
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

    if ($currentUser['role'] === 'admin') {
        usort($rows, function($a, $b){
            $order = ['admin'=>0,'teacher'=>1,'student'=>2];
            if ($a['role'] === $b['role']) {
                return strcmp($a['username'], $b['username']);
            }
            return ($order[$a['role']] ?? 9) <=> ($order[$b['role']] ?? 9);
        });
        return $rows;
    }

    if ($currentUser['role'] === 'teacher') {
        $res = [];
        foreach ($rows as $u) {
            if ((int)$u['id'] === (int)$currentUser['id']) {
                $res[] = $u;
                continue;
            }
            if ($u['role'] !== 'student') continue;

            $classIds = [];
            if (!empty($u['class'])) {
                $tmp = json_decode($u['class'], true);
                if (is_array($tmp)) $classIds = $tmp;
            }

            $isUnassigned = empty($classIds);
            $inMyClass = false;
            if (!$isUnassigned && !empty($teacherClassIDs)) {
                foreach ($classIds as $cid) {
                    if (in_array((int)$cid, $teacherClassIDs, true)) {
                        $inMyClass = true;
                        break;
                    }
                }
            }

            if ($isUnassigned || $inMyClass) $res[] = $u;
        }
        return $res;
    }

    // 學生只看到自己
    foreach ($rows as $u) {
        if ((int)$u['id'] === (int)$currentUser['id']) {
            return [$u];
        }
    }
    return [];
}
try {

    // =========================================================
    // action = change_password（使用者自行變更密碼）
    // =========================================================
    if ($method === 'POST' && $action === 'change_password') {

        $data = json_decode(file_get_contents('php://input'), true) ?? [];
        $currentPwd = trim($data['current_password'] ?? '');
        $newPwd     = trim($data['new_password'] ?? '');

        if (!in_array($currentUser['role'], ['admin', 'teacher'], true)) {
            http_response_code(400);
            echo json_encode(['message' => '學生帳號無需設定密碼']);
            return;
        }

        if ($newPwd === '') {
            http_response_code(400);
            echo json_encode(['message' => '新密碼不得為空白']);
            return;
        }

        $stmt = $pdo->prepare("SELECT id, username, password FROM users WHERE id = ?");
        $stmt->execute([$currentUser['id']]);
        $userRow = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$userRow) {
            http_response_code(404);
            echo json_encode(['message' => '帳號不存在']);
            return;
        }

        // 若有舊密碼，務必檢查
        if (!empty($userRow['password'])) {
            if ($currentPwd === '' || !password_verify($currentPwd, $userRow['password'])) {
                http_response_code(400);
                echo json_encode(['message' => '目前密碼錯誤']);
                return;
            }
        }

        $hash = password_hash($newPwd, PASSWORD_DEFAULT);
        $upd = $pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
        $upd->execute([$hash, $userRow['id']]);

        log_operation($pdo, $userRow['id'], $userRow['username'], 'USER_CHANGE_PASSWORD');

        echo json_encode(['success' => true, 'message' => '密碼已更新']);
        return;
    }

    // =========================================================
    // action = reset_password（管理者重設他人密碼）
    // =========================================================
    if ($method === 'POST' && $action === 'reset_password') {

        $data = json_decode(file_get_contents('php://input'), true) ?? [];
        $targetId = (int)($data['id'] ?? 0);

        if (!$targetId) {
            http_response_code(400);
            echo json_encode(['message' => '缺少目標帳號 ID']);
            return;
        }

        if ($currentUser['role'] !== 'admin') {
            http_response_code(403);
            echo json_encode(['message' => '僅管理者可以重設密碼']);
            return;
        }

        $stmt = $pdo->prepare("SELECT id, username, role FROM users WHERE id = ?");
        $stmt->execute([$targetId]);
        $targetUser = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$targetUser) {
            http_response_code(404);
            echo json_encode(['message' => '目標帳號不存在']);
            return;
        }

        // 僅 typing 可重設其他 admin 的密碼
        if ($targetUser['role'] === 'admin' && $currentUser['username'] !== 'typing') {
            http_response_code(403);
            echo json_encode(['message' => '僅 typing 可重設其他管理者的密碼']);
            return;
        }

        $hash = password_hash($targetUser['username'], PASSWORD_DEFAULT);
        $upd = $pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
        $upd->execute([$hash, $targetUser['id']]);

        log_operation($pdo, $currentUser['id'], $currentUser['username'], 'USER_RESET_PASSWORD');

        echo json_encode(['success' => true, 'message' => '密碼已重設（預設為帳號）']);
        return;
    }

    // =========================================================
    // action = export（匯出 CSV）
    // =========================================================
    if ($method === 'GET' && $action === 'export') {

        $filename = "users-export-" . date('Ymd') . ".csv";

        header('Content-Type: text/csv; charset=utf-8');
        header("Content-Disposition: attachment; filename=\"{$filename}\"");

        $output = fopen('php://output', 'w');

        // UTF-8 BOM
        fprintf($output, chr(0xEF) . chr(0xBB) . chr(0xBF));

        // 標題
        fputcsv($output, ['id','username','name','role','class_ids','grade','is_active']);

        $rows = get_visible_users($pdo, $currentUser, $teacherClassIDs);
        foreach ($rows as $r) {
            $classIds = '';
            if (!empty($r['class'])) {
                $tmp = json_decode($r['class'], true);
                if (is_array($tmp)) $classIds = implode(',', $tmp);
            }
            fputcsv($output, [
                $r['id'],
                $r['username'],
                $r['name'],
                $r['role'],
                $classIds,
                $r['grade'],
                $r['is_active'] ? '1' : '0'
            ]);
        }

        fclose($output);
        log_operation($pdo, $currentUser['id'], $currentUser['username'], 'EXPORT_USERS');
        return;
    }
    // =========================================================
    // 一般 CRUD 區域
    // =========================================================
    switch ($method) {

        // ===========================
        // GET：取得列表或單筆
        // ===========================
        case 'GET':

            $id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

            if ($id > 0) {
                // 讀取單筆
                $stmt = $pdo->prepare("SELECT id, username, name, class, role, grade, is_active 
                                       FROM users WHERE id = ?");
                $stmt->execute([$id]);
                $user = $stmt->fetch(PDO::FETCH_ASSOC);

                if (!$user) {
                    http_response_code(404);
                    echo json_encode(['message' => '帳號不存在']);
                    break;
                }

                // ===== 權限檢查 =====

                $canView = false;

                if ($currentUser['role'] === 'admin') {
                    $canView = true;

                } elseif ($currentUser['role'] === 'teacher') {

                    if ((int)$user['id'] === (int)$currentUser['id']) {
                        $canView = true;

                    } elseif ($user['role'] === 'student') {

                        $classIds = [];
                        if (!empty($user['class'])) {
                            $tmp = json_decode($user['class'], true);
                            if (is_array($tmp)) $classIds = $tmp;
                        }

                        $isUnassigned = empty($classIds);
                        $inMyClass   = false;

                        if (!$isUnassigned && !empty($teacherClassIDs)) {
                            foreach ($classIds as $cid) {
                                if (in_array((int)$cid, $teacherClassIDs, true)) {
                                    $inMyClass = true;
                                    break;
                                }
                            }
                        }

                        if ($isUnassigned || $inMyClass)
                            $canView = true;
                    }

                } else {
                    // 學生
                    if ((int)$user['id'] === (int)$currentUser['id'])
                        $canView = true;
                }

                if (!$canView) {
                    http_response_code(403);
                    echo json_encode(['message' => '權限不足']);
                    break;
                }

                echo json_encode($user);
                break;
            }

            // 取得列表（可見範圍）
            $rows = get_visible_users($pdo, $currentUser, $teacherClassIDs);
            echo json_encode($rows);
            break;

        // ===========================
        // POST：新增帳號（change_password / reset_password 已在前面）
        // ===========================
        case 'POST':

            if ($action !== null) {
                http_response_code(400);
                echo json_encode(['message' => '無效的操作']);
                break;
            }

            $data = json_decode(file_get_contents('php://input'), true);

            if (!$data) {
                http_response_code(400);
                echo json_encode(['message' => 'Invalid JSON']);
                break;
            }

            $newRole = $data['role'] ?? null;

            if (!in_array($newRole, ['admin','teacher','student'], true)) {
                http_response_code(400);
                echo json_encode(['message' => '角色不正確']);
                break;
            }

            if (empty($data['name'])) {
                http_response_code(400);
                echo json_encode(['message' => '「姓名」為必要欄位']);
                break;
            }

            if (empty($data['username'])) {
                http_response_code(400);
                echo json_encode(['message' => '「帳號」為必要欄位']);
                break;
            }

            // === 權限：建立 admin / teacher / student ===

            if ($newRole === 'admin') {

                if ($currentUser['username'] !== 'typing') {
                    http_response_code(403);
                    echo json_encode(['message' => '僅 typing 可建立管理者帳號']);
                    break;
                }

            } elseif ($newRole === 'teacher') {

                if ($currentUser['role'] !== 'admin') {
                    http_response_code(403);
                    echo json_encode(['message' => '僅管理者可建立教師帳號']);
                    break;
                }

            } else { // student

                if (!in_array($currentUser['role'], ['admin','teacher'], true)) {
                    http_response_code(403);
                    echo json_encode(['message' => '權限不足：僅管理者或教師可建立學生帳號']);
                    break;
                }
            }

            // === 處理班級 ===

            $class_json = null;
            $class_data = $data['class'] ?? [];
            $class_array = is_array($class_data)
                ? $class_data
                : (strlen((string)$class_data) === 0 ? [] : [$class_data]);

            if ($newRole === 'student') {
                if (count($class_array) > 1) {
                    http_response_code(400);
                    echo json_encode(['message' => '學生僅能指定一個班級']);
                    break;
                }
            } else {
                // 管理者、教師不需班級
                $class_array = [];
            }

            if (!empty($class_array)) {
                $class_array = array_map('intval', $class_array);
                $class_json  = json_encode($class_array);
            }

            // 教師只能指定自己班級
            if ($currentUser['role'] === 'teacher' && $newRole === 'student' && !empty($class_array)) {
                $targetClass = (int)$class_array[0];

                if (!in_array($targetClass, $teacherClassIDs, true)) {
                    http_response_code(403);
                    echo json_encode(['message' => '您僅能將學生指派到您所屬班級']);
                    break;
                }
            }

            // grade
            $grade = null;
            if ($newRole === 'student') {
                if (isset($data['grade'])) {
                    $grade = (int)$data['grade'];
                }
            }

            // 啟用狀態
            $is_active = isset($data['is_active']) ? (bool)$data['is_active'] : true;

            // === 管理者/教師密碼（預設帳號） ===
            $passwordHash = null;

            if (in_array($newRole, ['admin','teacher'], true)) {
                $plainPwd = trim($data['password'] ?? '');
                if ($plainPwd === '') $plainPwd = $data['username'];
                $passwordHash = password_hash($plainPwd, PASSWORD_DEFAULT);
            }

            // 寫入
            $stmt = $pdo->prepare(
                "INSERT INTO users 
                 (username, name, class, role, password, grade, is_active)
                 VALUES (?, ?, ?, ?, ?, ?, ?)"
            );

            $stmt->execute([
                $data['username'],
                $data['name'],
                $class_json,
                $newRole,
                $passwordHash,
                $grade,
                $is_active ? 1 : 0
            ]);

            $newId = $pdo->lastInsertId();

            log_operation($pdo, $currentUser['id'], $currentUser['username'], 'USER_CREATE');

            echo json_encode(['id' => $newId, 'message' => 'User created successfully']);
            break;
        // ===========================
        // PUT：更新使用者
        // ===========================
        case 'PUT':

            $targetId = isset($_GET['id']) ? (int)$_GET['id'] : 0;
            $data     = json_decode(file_get_contents('php://input'), true) ?? [];

            if ($targetId <= 0) {
                http_response_code(400);
                echo json_encode(['message' => '缺少 id']);
                break;
            }

            // 讀取目標帳號
            $stmt = $pdo->prepare(
                "SELECT id, username, name, class, role, grade, is_active
                 FROM users WHERE id = ?"
            );
            $stmt->execute([$targetId]);
            $targetUser = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$targetUser) {
                http_response_code(404);
                echo json_encode(['message' => '帳號不存在']);
                break;
            }

            // === 不能修改 username / role ===
            $newName     = $data['name'] ?? $targetUser['name'];
            $newIsActive = isset($data['is_active']) ? (bool)$data['is_active'] : (bool)$targetUser['is_active'];
            $newGrade    = $targetUser['grade'];

            // 處理班級
            $class_json  = $targetUser['class'];
            $class_data  = $data['class'] ?? null;
            $class_array = null;

            if ($class_data !== null) {
                $class_array = is_array($class_data)
                    ? $class_data
                    : (strlen((string)$class_data) === 0 ? [] : [$class_data]);

                $class_array = array_map('intval', $class_array);

                // 學生僅一個班級
                if ($targetUser['role'] === 'student' && count($class_array) > 1) {
                    http_response_code(400);
                    echo json_encode(['message' => '學生僅能指定一個班級']);
                    break;
                }

                $class_json = !empty($class_array) ? json_encode($class_array) : null;
            }

            // 年級（僅學生）
            if ($targetUser['role'] === 'student' && isset($data['grade'])) {
                $newGrade = (int)$data['grade'];
            }

            // === 權限檢查 ===

            if ($currentUser['role'] === 'teacher') {

                if ($targetUser['role'] !== 'student') {
                    http_response_code(403);
                    echo json_encode(['message' => '教師僅能修改學生']);
                    break;
                }

                // 教師只能修改自己班級或無班級學生
                $tClass = [];
                if (!empty($targetUser['class'])) {
                    $tmp = json_decode($targetUser['class'], true);
                    if (is_array($tmp)) $tClass = $tmp;
                }

                $isUnassigned = empty($tClass);
                $inMyClass    = false;

                if (!$isUnassigned && !empty($teacherClassIDs)) {
                    foreach ($tClass as $cid) {
                        if (in_array((int)$cid, $teacherClassIDs, true)) {
                            $inMyClass = true;
                            break;
                        }
                    }
                }

                $newAssigned = null;
                if ($class_array !== null && !empty($class_array)) {
                    $newAssigned = (int)$class_array[0];
                }

                if ($isUnassigned) {
                    // 無班級 → 只能指派至自己的班級
                    if ($newAssigned !== null && !in_array($newAssigned, $teacherClassIDs, true)) {
                        http_response_code(403);
                        echo json_encode(['message' => '您僅能將學生指派到您所屬的班級']);
                        break;
                    }
                } elseif ($inMyClass) {
                    // 已在我的班級 → 不可改到別的班級
                    if ($newAssigned !== null && !in_array($newAssigned, $teacherClassIDs, true)) {
                        http_response_code(403);
                        echo json_encode(['message' => '您僅能將學生留在您所屬的班級']);
                        break;
                    }
                } else {
                    http_response_code(403);
                    echo json_encode(['message' => '您無權修改此學生']);
                    break;
                }

            } elseif ($currentUser['role'] === 'admin') {

                // 管理者帳號只能由 typing 修改
                if ($targetUser['role'] === 'admin' &&
                    $currentUser['username'] !== 'typing') {

                    http_response_code(403);
                    echo json_encode(['message' => '僅 typing 可修改管理者帳號']);
                    break;
                }

            } else {
                // 學生無權限
                http_response_code(403);
                echo json_encode(['message' => '權限不足']);
                break;
            }

            // === 執行更新 ===
            $upd = $pdo->prepare(
                "UPDATE users SET name=?, class=?, grade=?, is_active=? WHERE id=?"
            );
            $upd->execute([
                $newName,
                $class_json,
                ($targetUser['role'] === 'student' ? $newGrade : null),
                $newIsActive ? 1 : 0,
                $targetId
            ]);

            log_operation($pdo, $currentUser['id'], $currentUser['username'], 'USER_UPDATE');

            echo json_encode(['message' => 'User updated successfully']);
            break;
        // ===========================
        // DELETE：刪除帳號
        // ===========================
        case 'DELETE':

            $targetId = isset($_GET['id']) ? (int)$_GET['id'] : 0;

            if ($targetId <= 0) {
                http_response_code(400);
                echo json_encode(['message' => '缺少 id']);
                break;
            }

            // 讀取目標帳號
            $stmt = $pdo->prepare(
                "SELECT id, username, role, class 
                 FROM users WHERE id = ?"
            );
            $stmt->execute([$targetId]);
            $targetUser = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$targetUser) {
                http_response_code(404);
                echo json_encode(['message' => '帳號不存在']);
                break;
            }

            // === 權限檢查 ===

            if ($currentUser['role'] === 'teacher') {

                if ($targetUser['role'] !== 'student') {
                    http_response_code(403);
                    echo json_encode(['message' => '教師僅能刪除學生']);
                    break;
                }

                $tClass = [];
                if (!empty($targetUser['class'])) {
                    $tmp = json_decode($targetUser['class'], true);
                    if (is_array($tmp)) $tClass = $tmp;
                }

                $isUnassigned = empty($tClass);
                $inMyClass    = false;

                if (!$isUnassigned && !empty($teacherClassIDs)) {
                    foreach ($tClass as $cid) {
                        if (in_array((int)$cid, $teacherClassIDs, true)) {
                            $inMyClass = true;
                            break;
                        }
                    }
                }

                if (!($isUnassigned || $inMyClass)) {
                    http_response_code(403);
                    echo json_encode(['message' => '您無權刪除此學生']);
                    break;
                }

            } elseif ($currentUser['role'] === 'admin') {

                // 管理者帳號只能由 typing 刪除
                if ($targetUser['role'] === 'admin' &&
                    $currentUser['username'] !== 'typing') {

                    http_response_code(403);
                    echo json_encode(['message' => '僅 typing 可刪除管理者帳號']);
                    break;
                }

            } else {

                http_response_code(403);
                echo json_encode(['message' => '權限不足']);
                break;
            }

            // 執行刪除
            $del = $pdo->prepare("DELETE FROM users WHERE id = ?");
            $del->execute([$targetId]);

            log_operation($pdo, $currentUser['id'], $currentUser['username'], 'USER_DELETE');

            echo json_encode(['message' => 'User deleted successfully']);
            break;

        // ===========================
        // OTHER：未支援的方法
        // ===========================
        default:
            http_response_code(405);
            echo json_encode(['message' => '方法不允許']);
            break;
    } // end switch

} catch (PDOException $e) {

    http_response_code(500);

    if ($e->getCode() == 23000) {
        echo json_encode(['message' => '錯誤：該帳號 (username) 已存在']);
    } else {
        echo json_encode(['message' => '資料庫錯誤: ' . $e->getMessage()]);
    }

} catch (Exception $e) {

    http_response_code(400);
    echo json_encode(['message' => '處理錯誤: ' . $e->getMessage()]);
}


