// 活動設定頁（新版彈窗、多組項目、24h 時分、複選與指定檔案）
(function () {
  const CURRENT_ROLE = window.currentUserRole;
  const CURRENT_USER_ID = window.currentUserId;

  const modal = document.getElementById('add-edit-event-modal');
  const openBtn = document.getElementById('show-add-event-btn');
  const closeBtn = document.getElementById('close-event-modal');
  const form = document.getElementById('event-form');
  const formTitle = document.getElementById('form-title');
  const submitBtn = document.getElementById('submit-event-btn');
  const cancelBtn = document.getElementById('cancel-event-btn');
  const msg = document.getElementById('event-form-message');

  const idInput = document.getElementById('event-id');
  const nameInput = document.getElementById('event-name');
  const notesInput = document.getElementById('event-notes');

  const regRow = document.getElementById('reg-dates-row');
  const regStartDate = document.getElementById('reg-start-date');
  const regStartHour = document.getElementById('reg-start-hour');
  const regStartMinute = document.getElementById('reg-start-minute');
  const regEndDate = document.getElementById('reg-end-date');
  const regEndHour = document.getElementById('reg-end-hour');
  const regEndMinute = document.getElementById('reg-end-minute');

  const evtStartDate = document.getElementById('evt-start-date');
  const evtStartHour = document.getElementById('evt-start-hour');
  const evtStartMinute = document.getElementById('evt-start-minute');
  const evtEndDate = document.getElementById('evt-end-date');
  const evtEndHour = document.getElementById('evt-end-hour');
  const evtEndMinute = document.getElementById('evt-end-minute');

  const itemsTable = document.getElementById('event-items-table');
  const itemsTbody = itemsTable.querySelector('tbody');
  const rowTpl = document.getElementById('event-item-row-tpl');
  const addItemBtn = document.getElementById('add-item-btn');

  const tableBody = document.getElementById('event-table-body');
  const tableMessage = document.getElementById('event-table-message');

  if (!form || !modal || !itemsTable) { return; }

  // helpers
  const show = (el, v = true) => (el.style.display = v ? 'flex' : 'none');
  const openModal = () => show(modal, true);
  const closeModal = () => { show(modal, false); resetForm(); };
  const setMessage = (text, isError = false, target = msg) => {
    if (!target) return;
    target.textContent = text || '';
    target.className = 'form-message-area ' + (isError ? 'error' : 'success');
  };
  const esc = (s) => (s ?? '').toString().replace(/[&<>"]/g, (c) => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;'}[c]));

  // 24h selects
  function fillTimeSelects(){
    const hours = [...Array(24)].map((_,i)=>String(i).padStart(2,'0'));
    const minutes = ['00','10','20','30','40','50'];
    [regStartHour,regEndHour,evtStartHour,evtEndHour].forEach(el=>{
      el.innerHTML = hours.map(h=>`<option value="${h}">${h}</option>`).join('');
      el.value='08';
    });
    [regStartMinute,regEndMinute,evtStartMinute,evtEndMinute].forEach(el=>{
      el.innerHTML = minutes.map(m=>`<option value="${m}">${m}</option>`).join('');
      el.value='00';
    });
  }
  fillTimeSelects();

  const toDbTs = (d, hh, mm) => d ? `${d} ${(hh||'00')}:${(mm||'00')}:00` : null;
  const getMulti = (sel) => [...sel.options].filter(o=>o.selected).map(o=>o.value);

  // items
  function renumber(){ [...itemsTbody.querySelectorAll('tr')].forEach((tr,i)=> tr.querySelector('.em-col-order').textContent = i+1 ); }
  function injectQuota(tr){
    const sel = tr.querySelector('.sel-quota');
    const mode = tr.querySelector('input[type=radio][name^="mode_"]:checked').value;
    sel.innerHTML='';
    if (mode==='timed'){
      [60,120,180,240,300,600].forEach(s=> sel.insertAdjacentHTML('beforeend', `<option value="${s}">${s===600?'10 分鐘':(s/60)+' 分鐘'}</option>`));
      sel.value='180';
    }else{
      [300,500,800,1000,1200,1500].forEach(c=> sel.insertAdjacentHTML('beforeend', `<option value="${c}">${c}</option>`));
      sel.value='500';
    }
  }
  function toggleScopeOrFile(tr){
    const src = tr.querySelector('input[type=radio][name^="src_"]:checked').value;
    const scopeSel = tr.querySelector('.sel-scope');
    const fileInput = tr.querySelector('.file-fixed');
    if (src==='fixed'){
      scopeSel.style.display='none';
      fileInput.style.display='';
    }else{
      scopeSel.style.display='';
      fileInput.style.display='none';
      fileInput.value='';
      fileInput._parsedIds = undefined;
    }
  }
  function parseFile(file, cb){
    const rd = new FileReader();
    rd.onload=()=> {
      const ids = String(rd.result||'').split(/\r?\n|,|;|\s/).map(s=>s.trim()).filter(Boolean).map(Number).filter(n=>Number.isInteger(n)&&n>0);
      cb(ids);
    };
    rd.readAsText(file, 'utf-8');
  }
  function addRow(prefill){
    const tr = rowTpl.content.firstElementChild.cloneNode(true);
    const uid = Math.random().toString(36).slice(2,8);
    tr.querySelectorAll('input[type=radio]').forEach(r=>{
      r.name = r.name.replace('__MODE__',`mode_${uid}`).replace('__SRC__',`src_${uid}`);
    });
    // bindings
    injectQuota(tr);
    tr.querySelectorAll(`input[name="mode_${uid}"]`).forEach(r=> r.addEventListener('change', ()=>injectQuota(tr)));
    tr.querySelectorAll(`input[name="src_${uid}"]`).forEach(r=> r.addEventListener('change', ()=>toggleScopeOrFile(tr)));
    tr.querySelector('.file-fixed').addEventListener('change', e=>{
      const f = e.target.files?.[0]; if (f) parseFile(f, ids => e.target._parsedIds = ids);
    });
    tr.querySelector('.btn-del').addEventListener('click', ()=>{ tr.remove(); renumber(); });

    if (prefill){
      const setMulti = (sel, val)=> {
        const arr = Array.isArray(val)? val : String(val||'').split(',').filter(Boolean);
        [...sel.options].forEach(o=> o.selected = arr.includes(o.value));
      };
      setMulti(tr.querySelector('.sel-eligibility'), prefill.eligibility ?? 'all');
      tr.querySelector(`input[name="mode_${uid}"][value="${prefill.mode||'measured'}"]`).checked = true;
      tr.querySelector(`input[name="src_${uid}"][value="${prefill.source||'random'}"]`).checked = true;
      injectQuota(tr);
      if (prefill.quota) tr.querySelector('.sel-quota').value = String(prefill.quota);
      setMulti(tr.querySelector('.sel-scope'), prefill.scope ?? 'all');
    }

    toggleScopeOrFile(tr);
    itemsTbody.appendChild(tr);
    renumber();
  }

  function resetForm(){
    form.reset();
    idInput.value=''; formTitle.textContent='新增活動';
    submitBtn.textContent='儲存'; cancelBtn.style.display='none';
    setMessage('');
    document.querySelector('input[name="event-type"][value="test"]').checked=true;
    regRow.style.display='none';
    itemsTbody.innerHTML=''; addRow();
  }

  function collectPayload(){
    const event_type = document.querySelector('input[name="event-type"]:checked').value;
    const payload = {
      id: idInput.value || null,
      name: nameInput.value.trim(),
      event_type,
      reg_start_at: event_type==='competition' ? toDbTs(regStartDate.value, regStartHour.value, regStartMinute.value) : null,
      reg_end_at:   event_type==='competition' ? toDbTs(regEndDate.value, regEndHour.value, regEndMinute.value) : null,
      start_at:     toDbTs(evtStartDate.value, evtStartHour.value, evtStartMinute.value),
      end_at:       toDbTs(evtEndDate.value, evtEndHour.value, evtEndMinute.value),
      notes: notesInput.value.trim() || null,
      items: []
    };
    if (!payload.name) throw new Error('請輸入活動名稱');
    if (!payload.start_at || !payload.end_at) throw new Error('請設定活動起迄日期時間');
    if (payload.end_at <= payload.start_at) throw new Error('活動結束時間必須晚於開始時間');
    if (event_type==='competition'){
      if (!payload.reg_start_at || !payload.reg_end_at) throw new Error('比賽類型需設定報名起迄');
      if (payload.reg_end_at <= payload.reg_start_at) throw new Error('報名結束時間必須晚於開始時間');
    }

    const rows = [...itemsTbody.querySelectorAll('tr')];
    if (!rows.length) throw new Error('請至少新增 1 個活動項目');

    rows.forEach(tr=>{
      const eligibility = getMulti(tr.querySelector('.sel-eligibility'));
      const mode = tr.querySelector('input[type=radio][name^="mode_"]:checked').value;
      const quota = parseInt(tr.querySelector('.sel-quota').value,10);
      const source = tr.querySelector('input[type=radio][name^="src_"]:checked').value;
      const scopeSel = tr.querySelector('.sel-scope');
      const fileFixed = tr.querySelector('.file-fixed');
      let scope = getMulti(scopeSel);
      let fixed_bank_ids;
      if (source==='fixed'){
        fixed_bank_ids = fileFixed?._parsedIds || [];
        scope = [];
        if (!fixed_bank_ids.length) throw new Error('題目來源為「指定」時，請上傳包含題庫 ID 的檔案');
      }
      payload.items.push({ eligibility, mode, quota, source, scope, fixed_bank_ids });
    });

    return payload;
  }

  // safe JSON
  async function safeJson(res){
    const text = await res.text();
    try { return JSON.parse(text); } catch { return { __raw: text }; }
  }

  async function loadEvents(){
    tableMessage.textContent='';
    tableBody.innerHTML = '<tr><td colspan="7">載入中...</td></tr>';
    try{
      const res = await fetch('/api/events.php', {cache:'no-store'});
      const data = await safeJson(res);
      if (!res.ok) throw new Error((data && data.message) || '讀取活動列表失敗');
      renderEventTable(data);
    }catch(e){
      tableBody.innerHTML = '<tr><td colspan="7" class="error">載入失敗</td></tr>';
      tableMessage.textContent = e.message;
      tableMessage.className = 'form-message-area error';
    }
  }

  function renderEventTable(events){
    tableBody.innerHTML='';
    if (!Array.isArray(events) || !events.length){
      tableBody.innerHTML = '<tr><td colspan="7">尚無建立任何活動</td></tr>'; return;
    }
    const now = new Date();
    events.forEach(ev=>{
      const tr = document.createElement('tr');
      tr.dataset.id = ev.id;
      tr.innerHTML = `
        <td>${ev.id}</td>
        <td>${esc(ev.name)}</td>
        <td>${ev.event_type==='competition' ? '<span class="label competition">比賽</span>' : '<span class="label test">測驗</span>'}</td>
        <td>${
          now < new Date(ev.start_at) ? '<span class="status-badge inactive">未開始</span>' :
          now > new Date(ev.end_at)   ? '<span class="status-badge inactive">已結束</span>' :
                                        '<span class="status-badge active">進行中</span>'
        }</td>
        <td>${ev.start_at} 至 ${ev.end_at}</td>
        <td>${ev.reg_start_at ? `${ev.reg_start_at} 至 ${ev.reg_end_at}` : 'N/A'}</td>
        <td class="actions">
          <button class="edit-btn">編輯</button>
          <button class="delete-btn">刪除</button>
        </td>`;
      if (CURRENT_ROLE==='teacher' && ev.owner_id && String(ev.owner_id)!==String(CURRENT_USER_ID)){
        tr.querySelector('.edit-btn')?.remove();
        tr.querySelector('.delete-btn')?.remove();
      }
      tableBody.appendChild(tr);
    });
  }

  async function saveEvent(payload){
    const res = await fetch('/api/events.php', {
      method:'POST',
      headers:{'Content-Type':'application/json'},
      body: JSON.stringify({ action:'save_event_with_items', data: payload })
    });
    const data = await safeJson(res);
    if (!res.ok || !data.success) throw new Error((data && data.message) || '儲存失敗');
    return data;
  }
  async function fetchEventDetail(id){
    const res = await fetch(`/api/events.php?action=get_event&id=${encodeURIComponent(id)}`);
    const data = await safeJson(res);
    if (!res.ok || !data.success) throw new Error((data && data.message) || '讀取活動失敗');
    return data;
  }
  async function deleteEvent(id){
    const res = await fetch(`/api/events.php?id=${encodeURIComponent(id)}`, {method:'DELETE'});
    const data = await safeJson(res);
    if (!res.ok || !data.success) throw new Error((data && data.message) || '刪除失敗');
    return data;
  }

  // handlers
  document.addEventListener('change', (e)=>{
    if (e.target.name === 'event-type'){
      regRow.style.display = e.target.value==='competition' ? 'grid' : 'none';
    }
  });

  addItemBtn.addEventListener('click', ()=> addRow());

  form.addEventListener('submit', async (e)=>{
    e.preventDefault();
    submitBtn.disabled = true;
    try{
      const payload = collectPayload();
      await saveEvent(payload);
      setMessage('儲存成功');
      setTimeout(()=>{ closeModal(); loadEvents(); }, 300);
    }catch(err){
      setMessage(err.message, true);
      submitBtn.disabled = false;
    }
  });

  openBtn.addEventListener('click', ()=>{ resetForm(); openModal(); });
  closeBtn.addEventListener('click', closeModal);
  cancelBtn.addEventListener('click', closeModal);
  modal.addEventListener('click', (e)=>{ if (e.target===modal) closeModal(); });

  tableBody.addEventListener('click', async (e)=>{
    const tr = e.target.closest('tr'); if (!tr) return;
    const id = tr.dataset.id;

    if (e.target.classList.contains('edit-btn')){
      try{
        const { event, items } = await fetchEventDetail(id);
        idInput.value = event.id;
        formTitle.textContent = '編輯活動';
        submitBtn.textContent = '更新';
        cancelBtn.style.display = 'inline-block';

        nameInput.value = event.name || '';
        document.querySelector(`input[name="event-type"][value="${event.event_type}"]`).checked = true;
        regRow.style.display = event.event_type==='competition' ? 'grid' : 'none';

        const setDT = (stamp, elDate, elH, elM) => {
          if (!stamp) return;
          const [d, t] = stamp.split(' ');
          const [hh, mm] = (t || '00:00:00').split(':');
          elDate.value=d; elH.value=hh; elM.value=mm;
        };
        setDT(event.reg_start_at, regStartDate, regStartHour, regStartMinute);
        setDT(event.reg_end_at, regEndDate, regEndHour, regEndMinute);
        setDT(event.start_at, evtStartDate, evtStartHour, evtStartMinute);
        setDT(event.end_at, evtEndDate, evtEndHour, evtEndMinute);
        notesInput.value = event.notes || '';

        itemsTbody.innerHTML='';
        (items || []).forEach(it=> addRow({
          eligibility: it.eligibility,
          mode: it.mode,
          quota: it.mode==='timed' ? it.time_limit_seconds : it.char_quota,
          source: it.question_source==='fixed' ? 'fixed' : 'random',
          scope: it.grade_scope
        }));
        if (!items || !items.length) addRow();

        openModal();
      }catch(err){ alert(err.message); }
    }

    if (e.target.classList.contains('delete-btn')){
      if (!confirm('確定要刪除此活動？此操作無法復原。')) return;
      try { await deleteEvent(id); loadEvents(); } catch(err){ alert(err.message); }
    }
  });

  (async function init(){ resetForm(); await loadEvents(); })();
})();

